/**
 * @typedef MassRequestItem
 * @property {Object} data данные запроса
 * @property {Function} onSuccess колбек при успешном выполнении запроса
 * @property {Function} onError колбек при неудаче
 */

/**
 * @property {String} this.params.cmd имя контроллера
 * @property {String} this.params.bulkKey имя параметра, под именем которого отправится массив данных запросов
 * @property {Number} this.params.debounceTimeout период ожидания перед отправкой запроса после последнего вызова get
 */
BEM.decl('i-mass-request', {
    onSetMod: {
        js: function() {
            var data = Object.assign({}, this.params.extraParams || {}, {
                cmd: this.params.cmd
            });

            this._request = BEM.create('i-request_type_ajax', {
                url: '/registered/main.pl',
                type: 'POST',
                dataType: 'json',
                callbackCtx: this,
                cache: false,
                data: data,
                timeout: this.params.timeout
            });
            this._requestsArray = [];
            this._get = $.debounce(this._get, this.params.debounceTimeout || 0);
            this._requestId = 1;
        }
    },

    /**
     * Совершает запрос
     * @param {Object} data данные запроса
     * @param {Function} onSuccess колбек при успешном ответе
     * @param {Function} onError колбек при ошибке
     */
    get: function(data, onSuccess, onError) {
        this._requestsArray.push({
            data: data,
            onSuccess: onSuccess || function() {},
            onError: onError || function() {}
        });

        this._get();
    },

    /*
     * Ключ, по которому будут сопоставляться запросы с ответами
     */
    _requestKey: 'requestId',

    /**
     * Следующий идентификатор элемента общего запроса
     */
    _requestId: null,

    /**
     * Выполняет общий запрос
     * В экземплярах блока эта функция задебауншена
     */
    _get: function() {
        var data = {},
            requestData = this._requestsArray.map(function(item) {
                var result = item.data;

                result[this._requestKey] = this ._requestId++;

                return result;
            }, this);

        data[this.params.bulkKey] = JSON.stringify(requestData);

        this._request.get(
            data,
            this._getOnSuccess(this._requestsArray),
            this._getOnError(this._requestsArray));

        this._requestsArray = [];
    },

    /**
     * Возвращает функцию, которая для всех элементов из requestsArray
     * вызовет onSuccess с соответствующим ответом
     * Функция будет являться callback-ом общего запроса
     * @param {MassRequestItem[]} requestsArray массив запросов
     * @returns {}
     */
    _getOnSuccess: function(requestsArray) {
        return function(data) {
            var args = Array.prototype.slice.call(arguments);

            if (this._isBadRequestError(data)) {
                requestsArray.forEach(function(request) {
                    request.onError.apply(this, args);
                }, this);
            } else {
                requestsArray.forEach(function(request) {
                    request.onSuccess(this._getResponseByKey(data, request.data[this._requestKey]));
                }, this);
            }
        }.bind(this);
    },

    /**
     * Проверяет, является ли ответ - описанием ошибки
     * сервер может ответить кодом 200, но при этом в ответе будет описание ошибки
     * @param {Object} data ответ
     * @returns {Boolean}
     */
    _isBadRequestError: function(data) {
        return !Array.isArray(data);
    },

    /**
     * Возвращает функцию, которая для всех элементов из requestsArray
     * вызовет onError со своими аргументами
     * Функция будет являться errback-ом общего запроса
     * @param {Array<MassRequestItem>} requestsArray массив запросов
     * @returns {}
     */
    _getOnError: function(requestsArray) {
        return function() {
            var args = [].slice.call(arguments);

            requestsArray.forEach(function(request) {
                request.onError.apply(this, args);
            }, this);
        }.bind(this);
    },

    /**
     * Возвращает элемент ответа по ключу
     * @param {Array} data "балковый" ответ
     * @param {*} keyValue значение ключа искомого элемента ответа
     * @returns {}
     */
    _getResponseByKey: function(data, keyValue) {
        return data
            .filter(function(item) {
                return item[this._requestKey] === keyValue;
            }, this)
            .pop();
    },

    /**
     * Отменяет запрос
     */
    abort: function() {
        this._request.abort();
    }
});
