describe('i-mass-request', function() {
    var clock,
        server,
        getBlock = function(params) {
            return BEM.create('i-mass-request', {
                cmd: params.cmd,
                bulkKey: params.bulkKey,
                debounceTimeout: params.debounceTimeout
            });
        };

    before(function () {
        clock = sinon.useFakeTimers();
    });

    beforeEach(function() {
        server = sinon.fakeServer.create();
    });

    afterEach(function() {
        server.restore();
    });

    after(function () {
        clock.restore();
    });

    describe('get', function() {
        it('должен делать запросы на контроллер из params.cmd', function() {
            var cmd = 'getGeoRestrictions',
                request = getBlock({
                    cmd: cmd,
                    bulkKey: 'json_data'
                });

            request.get({});

            clock.tick(10000);

            expect(server.requests[0].requestBody.indexOf('cmd=' + cmd)).not.to.be.equal(-1);
        });

        it('должен ожидать params.debounceTimeout после последнего вызова get перед выполнением запроса', function() {
            var cmd = 'getGeoRestrictions',
                request = getBlock({
                    cmd: cmd,
                    bulkKey: 'json_data',
                    debounceTimeout: 100
                });

            request.get({});

            clock.tick(99);

            expect(server.requests.length).to.be.equal(0);

            clock.tick(1);

            expect(server.requests.length).to.be.equal(1);
        });

        it('Должен выполнять запрос в следующем тике, если не задан debounceTimeout', function() {
            var cmd = 'getGeoRestrictions',
                request = getBlock({
                    cmd: cmd,
                    bulkKey: 'json_data'
                });

            request.get({});

            expect(server.requests.length).to.be.equal(0);

            clock.tick(1);

            expect(server.requests.length).to.be.equal(1);
        });

        it('должен группировать данные запросов по ключу bulkKey', function() {
            var bulkKey = 'json_data',
                data1 = { bid: 1 },
                data2 = { bid: 2 },
                request = getBlock({
                    cmd: 'getGeoRestrictions',
                    bulkKey: bulkKey
                });

            request.get(data1);
            request.get(data2);

            clock.tick(10000);

            var expectedResult = 'json_data=' + encodeURIComponent(JSON.stringify([data1, data2]));
            expect(server.requests[0].requestBody.indexOf(expectedResult)).not.to.be.equal(-1);
        });

        it('должен добавлять к элементам запроса requestId', function() {
            var request = getBlock({
                    cmd: 'getGeoRestrictions',
                    bulkKey: 'json_data'
                }) ;

            request.get({ key: 'value' });

            clock.tick(10000);

            expect(JSON.parse(u.parseRequestBody(server.requests[0].requestBody).json_data)[0]).to.have.property('requestId');
        });

        it('должен вызывать onSuccess всех собранных запросов при успехе, распределяя ответы по ключу requestKey', function() {
            var onSuccess1 = sinon.spy(),
                onSuccess2 = sinon.spy(),
                request = getBlock({
                    cmd: 'getGeoRestrictions',
                    bulkKey: 'json_data'
                }),
                response1 = { requestId: 1, bid: 1, data: 'abc' },
                response2 = { requestId: 2, bid: 2, data: 'cde' };

            request.get({ bid: 1 }, onSuccess1);
            request.get({ bid: 2 }, onSuccess2);

            clock.tick(10000);

            server.requests[0].respond(200, { "Content-Type": "application/json" }, JSON.stringify([response1, response2]));

            expect(onSuccess1.calledWith(response1)).to.be.equal(true);
            expect(onSuccess2.calledWith(response2)).to.be.equal(true);
        });

        it('должен вызывать onError всех собранных запросов при ошибке', function() {
            var onError1 = sinon.spy(),
                onError2 = sinon.spy(),
                request = getBlock({
                    cmd: 'getGeoRestrictions',
                    bulkKey: 'json_data'
                });

            request.get({ bid: 1 }, function() {}, onError1);
            request.get({ bid: 2 }, function() {}, onError2);

            clock.tick(10000);

            server.requests[0].respond(500, { "Content-Type": "application/json" }, JSON.stringify({ error_code: 6 }));

            expect(onError1.called).to.be.equal(true);
            expect(onError2.called).to.be.equal(true);
        });

        it('должен вызывать onError все собранных запросов если успешный ответ - описание ошибки', function() {
            var onError1 = sinon.spy(),
                onError2 = sinon.spy(),
                request = getBlock({
                    cmd: 'getGeoRestrictions',
                    bulkKey: 'json_data'
                }),
                errorData = { error: 'ошибка!', error_no: 6 };

            request.get({ bid: 1 }, function() {}, onError1);
            request.get({ bid: 2 }, function() {}, onError2);

            clock.tick(10000);

            server.requests[0].respond(200, { "Content-Type": "application/json" }, JSON.stringify(errorData));

            expect(onError1.called).to.be.equal(true);
            expect(onError2.called).to.be.equal(true);
        });

        it('корректно выполняет 2 поочередных (через большой промежуток времени) запроса на один cmd', function() {
            var onSuccess = sinon.spy(),
                request = getBlock({
                    cmd: 'getGeoRestrictions',
                    bulkKey: 'json_data'
                }),
                response1 = { requestId: 1, data: 'response1' },
                response2 = { requestId: 2, data: 'response2' };

            request.get({ bid: 1 }, onSuccess);
            clock.tick(10000);
            server.requests[0].respond(200, { "Content-Type": "application/json" }, JSON.stringify([response1]));
            expect(onSuccess.calledWith(response1)).to.be.equal(true);

            request.get({ bid: 2 }, onSuccess);
            clock.tick(10000);
            server.requests[1].respond(200, { "Content-Type": "application/json" }, JSON.stringify([response2]));
            expect(onSuccess.calledWith(response2)).to.be.equal(true);
        });

        it('корректно выполняет запрос, если параллельно выполняется запрос на другой cmd', function() {
            var onSuccess11 = sinon.spy(),
                onSuccess12 = sinon.spy(),
                onSuccess21 = sinon.spy(),
                onSuccess22 = sinon.spy(),
                request1 = getBlock({
                    cmd: 'getGeoRestrictions',
                    bulkKey: 'json_data'
                }),
                request2 = getBlock({
                    cmd: 'ajaxCheckStreet',
                    bulkKey: 'json_data'
                }),
                response11 = { requestId: 1, data: 'response11' },
                response12 = { requestId: 2, data: 'response12' },
                response21 = { requestId: 1, data: 'response21' },
                response22 = { requestId: 2, data: 'response22' };

            //2 запроса с разными ключами
            request1.get({ bid: 1 }, onSuccess11);
            request2.get({ bid: 3 }, onSuccess21);
            //2 запроса с одинаковыми ключами
            request1.get({ bid: 2 }, onSuccess12);
            request2.get({ bid: 2 }, onSuccess22);

            clock.tick(10000);

            server.requests[0].respond(200, { "Content-Type": "application/json" }, JSON.stringify([response11, response12]));
            server.requests[1].respond(200, { "Content-Type": "application/json" }, JSON.stringify([response21, response22]));

            expect(onSuccess11.calledWith(response11)).to.be.equal(true);
            expect(onSuccess12.calledWith(response12)).to.be.equal(true);
            expect(onSuccess21.calledWith(response21)).to.be.equal(true);
            expect(onSuccess22.calledWith(response22)).to.be.equal(true);
        });

    });

});
