/**
 * Провайдер данных для контрола множественного выбора b-badges-multiselect
 * Работает на основе статического массива
 *
 * @param {Array} [params.items] – источник данных провайдера
 * @param {Array} [params.disabledItems] – исключенные из выборки элементы
 * @param {Number|Function} [params.limit] – ограничение кол-ва, возвращаемых элементов
 */
BEM.decl({ name: 'i-multiselect-static-provider', implements: 'i-multiselect-provider' }, {

    onSetMod: {

        js: function() {
            var params = this.params;

            this._items = params.items || [];
            this._disabled = {};
            this._limit = params.limit || Number.POSITIVE_INFINITY;

            params.disabledItems && this.disableItems(params.disabledItems);
            this._itemsHash = this._items.reduce(function(accum, item) {
                accum[this._getIdFromItem(item)] = item;

                return accum;
            }.bind(this), {});

        }

    },

    /**
     * Возвращает результат наложения поискового фильтра
     * @param {String} part – поисковая фраза
     * @returns {Array}
     * @private
     */
    _applyFilter: function(part) {
        var reg = part && new RegExp(u.escape.regExp(part), 'i');

        return this._items
            .filter(function(item) {
                var text = '';

                if (typeof item === 'string') {
                    text = item;
                } else if (item.title) {
                    text = item.title;
                }

                return !this._disabled[this._getIdFromItem(item)] && (!reg || text.match(reg));
            }, this);
    },

    /**
     * Формирует найденный набор элементов
     * @param {Array} items – элементы
     * @param {String} part – поисковая фраза
     * @returns {Array}
     * @private
     */
    _createResponseItems: function(items, part) {
        return items.map(function(item) {
            return ['highlight', item, part];
        });
    },

    /**
     * Прокидывает в callback список элементов, удовлетворяющих критерию criteria
     * @param {String} part – поисковая фраза
     * @param {Number} pos – позиция
     * @param {Function} callback
     */
    get: function(part, pos, callback) {
        var total = this._applyFilter(part),
            favorites = total.filter(function(item) {
                return item.favorite;
            });

        callback({
            total: total.length,
            favorites: favorites.length,
            items: this._createResponseItems(
                total.splice(0, $.isFunction(this._limit) ? this._limit(total.length) : this._limit), part)
        });
    },

    /**
     * Прокидывает в callback список элементов, удовлетворяющих критерию criteria, игнорируя ограничения
     * @param {String} part – поисковая фраза
     * @param {Number} pos – позиция
     * @param {Function} callback
     */
    getAllMatched: function(part, pos, callback) {
        var total = this._applyFilter(part);

        callback({
            total: total.length,
            items: this._createResponseItems(total, part)
        });
    },

    /**
     * Прокидывает в callback список favorites элементов, удовлетворяющих критерию criteria, игнорируя ограничения
     * @param {String} part – поисковая фраза
     * @param {Number} pos – позиция
     * @param {Function} callback
     */
    getFavoritesMatched: function(part, pos, callback) {
        var total = this._applyFilter(part),
            favorites = total.filter(function(item) {
                return item.favorite;
            });

        callback({
            favorites: favorites.length,
            items: this._createResponseItems(favorites, part)
        });
    },

    /**
     * Возвращает найденные по id элементы в наборе данных
     * @param {Array} ids
     * @returns {Array}
     */
    getByIds: function(ids) {

        return ids.reduce(function(res, id) {
            var item = this._itemsHash[id];

            item && !this._disabled[id] && res.indexOf(item) === -1 && res.push(item);

            return res;
        }.bind(this), []);
    },

    /**
     * Исключает элементы items из выборки
     * @param {Array} items
     */
    disableItems: function(items) {
        items.forEach(function(item) {
            this._disabled[this._getIdFromItem(item)] = true;
        }, this);
    },

    /**
     * Делает элементы items доступными для выборки
     * @param {Array} items
     */
    enableItems: function(items) {
        items.forEach(function(item) {
            var id = item.id || item;

            this._disabled[id] && (this._disabled[id] = false);
        }, this);
    },

    /**
     * Возвращает id элемента
     * @param {Object|Number} item – элемент
     * @private
     */
    _getIdFromItem: function(item) {
        return item.id || item;
    }

});
