/**
 * @event b-retargeting-conditions-list#editConditionCompleted
 * @type {Object}
 * @property {Object} condition данные условия для редактирования
 *
 * @event i-retargeting-controllert#deleteConditionCompleted
 * @type {Object}
 * @property {String} conditionId идентификатор удаленного условия
 *
 * @fires i-retargeting-controller#editConditionCompleted событие успешного изменения данных/добавления условия
 *     ретаргетинга
 * @fires i-retargeting-controller#deleteConditionCompleted событие успешного удаления условия ретаргетинга
 */
BEM.decl('i-retargeting-controller', {
    onSetMod: {
        js: {
            inited: function() {
                this._subscribe = BEM.create('i-subscription-manager');
                this._conditionsListBlock = this.params.conditionsListBlock;
                this._initDeleteConditionAction();
                this._subscribe
                    .on(this._conditionsListBlock, 'edit-condition add-condition', this._openEditPopup, this);
            }
        }
    },

    /**
     * Удаляет блок и подписки на события
     */
    destruct: function() {
        this._subscribe.dispose();
        return this.__base.apply(this, arguments);
    },

    /**
     * Инициализирует функциональность удаления условия ретаргетинга
     */
    _initDeleteConditionAction: function() {
        var _this = this;

        this._subscribe.wrap(this._conditionsListBlock)
            .on('deleteCondition', function(event, data) {
                var conditionIdToDelete = data.conditionId, conditionsListBlock = _this._conditionsListBlock;

                BEM.blocks['b-confirm'].open({
                    message: iget2('i-retargeting-controller', 'vy-uvereny-chto-hotite-103', 'Вы уверены, что хотите удалить выбранное условие?'),
                    onYes: function() {
                        conditionsListBlock.setDeleteConditionLoading(true, conditionIdToDelete);

                        u['retargeting-dataprovider'].deleteCondition([conditionIdToDelete])
                            .then(function(response) {
                                if (response.success) {
                                    conditionsListBlock.deleteCondition(conditionIdToDelete);

                                    this.trigger('deleteConditionCompleted', { conditionId: conditionIdToDelete });
                                } else {
                                    this._onDeleteConditionFailed(
                                        conditionIdToDelete,
                                        iget2(
                                            'i-retargeting-controller',
                                            'pri-udalenii-usloviya-proizoshla',
                                            'При удалении условия произошла ошибка, повторите попытку позже.'
                                        )
                                    );
                                }
                            }.bind(this))
                            .catch(function(response) {
                                var errors = u._.get(response, 'validation_result.errors', []);

                                this._onDeleteConditionFailed(
                                    conditionIdToDelete,
                                    errors
                                        .map(function(error) {
                                            return error.description || error.text
                                        })
                                        .join('\n')
                                );

                            }.bind(this));
                    }
                }, this);
            }, this);
    },

    /**
     * Обработчик события неуспешного удаления условия ретаргетинга сервером
     * @param {String} conditionId идентификатор удаляемого условия ретаргетинга
     * @param {String} errorText текст ошибки
     */
    _onDeleteConditionFailed: function(conditionId, errorText) {
        this._conditionsListBlock.setDeleteConditionLoading(false, conditionId);

        this.trigger('deleteConditionCompleted', { error: errorText });
    },

    /**
     * Открывает попап редактирования/создания условия ретаргетинга
     * @param {Event} e
     * @param {Object} [data] данные условия ретаргетинга (указываются для открытия попапа в режиме редактирования)
     */
    _openEditPopup: function(e, data) {
        var editConditionBlock = BEM.DOM.blocks['b-retargeting-condition-edit-popup'].createInstance({
            parentPopup: this.params.parentPopup,
            conditionId: data ? data.conditionId : 0
        });

        editConditionBlock
            .on('created', function(e, condition) {
                var data = u['b-retargetings-data'].mapResponseConditionToModel(condition),
                    condModel = BEM.MODEL.getOne({
                        name: 'm-retargeting-condition',
                        id: data.ret_cond_id
                    });
                if (condModel) {
                    condModel.update(data);
                } else {
                    BEM.MODEL.create({
                        name: 'm-retargeting-condition',
                        id: data.ret_cond_id
                    }, data);
                }
                this._conditionsListBlock.addCondition(
                    u['b-retargetings-data'].mapResponseConditionToModel(condition)
                );
            }, this)
            .on('changed', function(e, condition) {
                this._conditionsListBlock.updateCondition(
                    u['b-retargetings-data'].mapResponseConditionToModel(condition)
                );
            }, this);
    }
});
