/**
 * @typedef {Object} selectedChangeData
 * @property {String} key ключ элемента
 * @property {Boolean} isSelected состояние выбран/снят элемента
 */

/**
 * @typedef {Object} selectedChangeEventData
 * @property {BEM} source источник события
 * @property {selectedChangeData} data данные об изменении
 */

/**
 * @event i-selectable#selectedChanged
 * @type {selectedChangeEventData}
 */

/**
 * @fires i-selectable#selectedChanged
 * alkaline@todo обрабатывать вложенность
 */
BEM.DOM.decl('i-selectable', {
    /**
     * Объект, содержащий состояние выбран/снят элементов. Ключи - идентификаторы элементов,
     * содержимое - объекты вида { isSelected: {Boolean}, data: {Object} }
     * @type {Object}
     */
    _items: null,

    /**
     * Объект, содержащий соответствующие элементам дочерние блоки b-selectable-control по ключам элементов
     * @type {Object}
     */
    _itemNodes: null,

    onSetMod: {
        js: function() {
            this.considerItemNodes();

            BEM.blocks['b-selectable-control'].on(this.domElem, 'selectedChanged', function(e, data) {
                this._onControlSelectedChanged(data);
            }, this);
        }
    },

    /**
     * Возвращает массив строковых ключей существующих элементов
     * @returns {String[]}
     */
    getKeysList: function() {
        return Object.keys(this._items);
    },

    /**
     * Возвращает массив строковых ключей существующих элементов, которые не задисаблены
     * @returns {String[]}
     */
    getEnabledKeysList: function() {
        return Object.keys(this._items).filter(function(key) {
            return !this._items[key].data.isDisabled;
        }, this);
    },

    /**
     * Возвращает массив выбранных элементов
     * @returns {Object[]}
     */
    getSelected: function() {
        return this.getKeysList()
            .filter(function(key) {
                return this._items[key].isSelected;
            }, this)
            .map(function(key) {
                return this._items[key].data;
            }, this);
    },

    /**
     * Выбирает или снимает выбор с указанных элементов
     * @param {Object} selectedMap - объект, ключи которого - ключи элементов, а значения - желаемые состояния выбран/снят элементов
     * @example selectable.setSelected({ 2: true, 42: false })
     * @fires i-selectable#selectedChanged
     */
    setSelected: function(selectedMap) {
        Object.keys(selectedMap).map(function(key) {
            var value = selectedMap[key],
                item = this._items[key],
                itemNode;

            if (item.isSelected != value) {
                item.isSelected = value;

                itemNode = this._itemNodes[key];

                if (itemNode) {
                    itemNode._setSelected(key, value);
                }

                this.trigger('selectedChanged', {
                    source: this,
                    data: {
                        key: key,
                        isSelected: value
                    }
                });
            }
        }, this);
    },

    /**
     * Ищет свои выбираемые блоки и запоминает их.
     * Если не существует элемента с ключом, как у найденного блока, то добавится новый элемент.
     * Если элемент с ключом, как у найденного блока, уже существует, блоку выставится состояние элемента
     *
     * Если блоки уже искались, ничего не делает
     * @param {Object} [options] параметры
     * @param {Boolean} options.forceSearch Все равно обновить блоки, даже если они уже искались.
     *                  Необходимо, если динамически были добавлены или удалены блоки, и нужно уведомить i-selectable об их существовании
     * @param {Object} options.items Если у b-selectable-control внутри изменились элементы, нужно сообщить блоку об этом
     * @returns {this}
     * @todo Возможно, стоит вместо параметра force сделать методы
     * _registerItem(b-selectable-control)/_unregisterItem(b-selectable-control) - так должно быть меньше вычислительных затрат
     */
    considerItemNodes: function(options) {
        options = options || {};
        if (!this._items) {
            this._items = this.params.items || {};
        }

        if (!this._itemNodes || options.forceSearch) {
            this._itemNodes = this._getItemNodes().reduce(function(res, node) {
                node._getHandledKeys(options.items).forEach(function(key) {
                    res[key] = node;

                    if (!this._items[key]) {
                        this._items[key] = { isSelected: node._isSelected(key), data: node._getData(key) };
                    } else {
                        node._setSelected(key, this._items[key].isSelected);
                    }
                }, this);

                return res;
            }.bind(this), {});
        }

        return this;
    },

    /**
     * @override {BEM}
     */
    destruct: function() {
        this._items = this._itemNodes = null;
        return this.__base.apply(this, arguments);
    },

    /**
     * Возвращает дочерние блоки b-selectable-control
     * @returns {BEM.DOM}
     */
    _getItemNodes: function() {
        return this.findBlocksInside('b-selectable-control');
    },

    /**
     * Колбек изменения выбрано/снято в дочернем контроле
     * @param {selectedChangeEventData} params
     */
    _onControlSelectedChanged: function(params) {
        this._items[params.data.key].isSelected = params.data.isSelected;

        this.trigger('selectedChanged', params);
    }
});
