BEM.decl('i-smart-filters-manager', {
    onSetMod: {
        js: function() {
            this._statesMap = this.params.statesMap;
            this._filterName = this.params.filterName;
            this._subFilterName = this.params.subFilterName;

            this._initDefaultState();
        }
    },

    /**
     * Доступен ли данный сабфильтр<subFilter> для фильтра<filter>
     * @param {String} filter
     * @param {String} subFilter
     * @returns {Boolean}
     */
    isSubFilterEnabled: function(filter, subFilter) {
        if (!this._state[filter] || !this._state[filter].subFilters[subFilter]) {
            throw new Error('Combination of ' + filter + ', ' + subFilter + ', is illegal');
        }

        return this._state[filter].subFilters[subFilter].isEnabled;
    },

    /**
     * Доступен ли фильтр <filter>
     * @param {String} filter
     * @returns {Boolean}
     */
    isFilterEnabled: function(filter) {
        return this._state[filter].isEnabled;
    },

    /**
     *
     * @param {'add'|'remove'|'change-filter'|'change-sub-filter'} action
     * @param {Object} values
     * @param {String} values.filter
     * @param {String} values.subFilter
     * @param {String} [values.prevFilter]
     * @param {String} [values.prevSubFilter]
     */
    updateAvailableHash: function(action, values) {
        switch (action) {
            case 'add':
                //добавлено новое условие фильтра
                //постепенно удаляем из этого списка выбранные
                this._removeSubFilter(values.filter, values.subFilter);
                break;
            case 'remove':
                //условие фильтра {field: field, relation: relation } удалено
                //добавляем relation как снова доступный для данного фильтра
                this._addSubFilter(values.filter, values.subFilter);
                break;
            case 'change-filter':
                //для values.prevFilter values.subFilter снова доступно
                this._addSubFilter(values.prevFilter, values.prevSubFilter);
                //для выбранного this._state[values.filter] выбранное условие снова недоступно
                this._removeSubFilter(values.filter, values.subFilter);

                break;
            case 'change-sub-filter':
                //prevRelation стало снова доступно
                this._addSubFilter(values.filter, values.prevSubFilter);
                //а выбранное теперь values.subFilter стало недоступным для выбора
                this._removeSubFilter(values.filter, values.subFilter);
                break;
        }
    },

    /**
     * Возвращает первый из возможных фильтров, которые еще не были использованы
     * @returns {Object}
     */
    getNewFilterData: function() {
        var filterName = this._getFirstEnabledFilter(),
            data = {};

        data[this._filterName] = filterName;
        data[this._subFilterName] = this._getFirstEnabledSubFilter(filterName);

        return data;
    },

    /**
     * Если изменилось значение filter, то значение subFilter не обязательно останется тем же. Если оно недоступно - вместо
     * него вернется первое доступное
     * @param {Object} values
     * @param {String} values.filter
     * @param {String} values.subFilter
     * @returns {Object}
     */
    getAdjustedNewFilterData: function(values) {
        var newFilterData = {},
            orderedSubFiltersList = this.getOrderedSubFiltersList(values.filter);

        if (!this.isFilterEnabled(values.filter)) {
            throw new Error('There is no available subfilters for filter ' + values.filter);
        }

        //Если сабфильтр не содержится в списке сабфильтров данного фильтра
        if (orderedSubFiltersList.indexOf(values.subFilter) == -1 ||
            //или содержится но недоступен
            !this.isSubFilterEnabled(values.filter, values.subFilter)) {
            values.subFilter = this._getFirstEnabledSubFilter(values.filter);
        }

        newFilterData[this._filterName] = values.filter;
        newFilterData[this._subFilterName] = values.subFilter;

        return newFilterData;
    },

    /**
     * Возвращает упорядоченный список доступных фильтров
     * @returns {Array}
     */
    getOrderedFiltersList: function() {
        return [].concat(this._statesMap.orderedFiltersNames);
    },

    /**
     * Возвращает упорядоченный список доступных сабфильтров для фильтра <filter>
     * @param {String} filter
     * @returns {Array}
     */
    getOrderedSubFiltersList: function(filter) {
        return [].concat(this._statesMap[filter].orderedSubFiltersNames);
    },

    /**
     * Возвращаем имя первого доступного фильтра
     * @returns {String}
     * @private
     */
    _getFirstEnabledFilter: function() {
        var orderedFiltersNames = this.getOrderedFiltersList(),
            enabledFilterIndex = u._.findIndex(orderedFiltersNames, function(filterName) {
                return this.isFilterEnabled(filterName);
            }, this);

        return orderedFiltersNames[enabledFilterIndex];
    },

    /**
     * Возвращает название первого из доступных сабфильтров для данного фильтра
     * @param {String} filter
     * @returns {String}
     * @private
     */
    _getFirstEnabledSubFilter: function(filter) {
        var orderedSubFiltersNames = this.getOrderedSubFiltersList(filter),
            enabledSubFilterIndex = u._.findIndex(orderedSubFiltersNames, function(subFilterName) {
                return this.isSubFilterEnabled(filter, subFilterName);
            }, this);

        return orderedSubFiltersNames[enabledSubFilterIndex];
    },

    /**
     * Инициализируем начальное состояние
     * @private
     */
    _initDefaultState: function() {
        this._state = {};

        if (!u._.isArray(this._statesMap.orderedFiltersNames)) {
            throw new Error('i-smart-filters-manager: statesMap must have field orderedFiltersNames of type array');
        }

        this._statesMap.orderedFiltersNames.forEach(function(filterName) {
            if (!u._.isArray(this._statesMap[filterName].orderedSubFiltersNames)) {
                throw new Error('i-smart-filters-manager: statesMap[' + filterName + '] ' +
                    'must have field orderedFiltersNames of type array');
            }

            this._state[filterName] = {
                isEnabled: true,
                subFilters: {}
            };

            this._statesMap[filterName].orderedSubFiltersNames.forEach(function(subFilterName) {
                this._state[filterName].subFilters[subFilterName] = {
                    isEnabled: true
                };
            }, this);
        }, this);
    },

    /**
     * Добавляем сабфильтр в список доступных
     * @param {String} filter
     * @param {String} subFilter
     * @private
     */
    _addSubFilter: function(filter, subFilter) {
        this._state[filter].subFilters[subFilter].isEnabled = true;

        this._adjustFilterAvailability(filter);
    },

    /**
     * Удаляем сабфильтр из списка доступных
     * @param {String} filter
     * @param {String} subFilter
     * @private
     */
    _removeSubFilter: function(filter, subFilter) {
        this._state[filter].subFilters[subFilter].isEnabled = false;

        this._adjustFilterAvailability(filter);
    },

    /**
     * Корректируем доступность фильтра по наличию/отсутствию у него доступных сабфильтров
     * @param {String} filter
     * @returns {BEM}
     * @private
     */
    _adjustFilterAvailability: function(filter) {
        this._state[filter].isEnabled = u._.some(this._state[filter].subFilters, function(subFilter) {
            return subFilter.isEnabled;
        });

        return this;
    }
});
