/**
 * Блок для вывода сообщения при уходе пользователя со страницы.
 *
 * В браузерах, которые поддерживают событие окна beforeunload вешается на это событие
 * и при попытке пользователя уйти со страницы (или обновить, или закрыть) вызывает функцию чекер.
 * Если чекер вернул правдивое значение, то пользователь будет предупрежден об уходе.
 *
 * Для браузеров, не поддерживающих такое поведение - блок обработчик клика на все ссылки.
 * При клике на ссылку, уводящую со страницы будет выведено предупреждение.
 * Такое поведение только для старых версий оперы (до Blink).
 * После того как опера версии 12.16 (4,8% на текущий момент).
 * и ниже перестанет поддерживаться, это поведение можно убрать.
 */
BEM.DOM.decl('i-unload-warning', {}, {

    /**
     * Флаг, показывающий вешался ли блок на событие страницы
     */
    _binded: false,

    /**
     * Список чекеров для проверки
     */
    _checkers: [],

    /**
     * Стандартное сообщение
     */
    _defaultMessage: iget2(
        'i-unload-warning',
        'eta-stranica-prosit-vas',
        'Эта страница просит вас подтвердить, что вы хотите уйти — при этом введённые вами данные могут не сохраниться.'
    ),

    /**
     * Обработчик события ухода со страницы.
     * @param {Event} event Событие.
     * @returns {String} Текст сообщения.
     * @private
     */
    _onUnload: function(event) {
        var results = [],
            messages,
            text;

        this._checkers.forEach(function(checker) {
            var result = checker.fn.call(checker.ctx);

            // чекер сработал, если вернул правдивый результат
            if (result) results.push(result);
        });

        // Если какой то чекер сработал
        if (results.length) {
            // Оставляем только текстовые сообщения
            messages = results.filter(function(m) {
                return Object.prototype.toString.call(m) == '[object String]';
            });

            event || (event = window.event);

            // Если есть сообщения - выводим их, если нет, то стандартное
            text = messages.length ? messages.join('\n\n') : this._defaultMessage;

            event && (event.returnValue = text);

            if (BEM.blocks['i-ua'].opera && !confirm(text)) {
                event.preventDefault();
            } else {
                return text;
            }
        }
    },

    /**
     * Добавляет функцию, разрешающую уход со страницы.
     * @param {Function} checker Функция, которая вызывается при уходе со страницы. Если она вернет текст,
     *   то пользователю будет показано предупреждение об уходе с этим текстом. Если вернет правдивое значение
     *   (но не строку) - будет показано стандартное сообщение.
     * @param {Object} [ctx] Контекст выполнения функции checker.
     * @param {Object} [win] Глобальный объект window. Параметр может использоваться для тестов.
     */
    addChecker: function(checker, ctx, win) {
        var handler;

        win = win || window;

        // Так как нет нужны вешаться всегда на страницу, делаем это только при первом добавлении чекера
        if (!this._binded) {
            this._binded = true;

            handler = this._onUnload.bind(this);

            if (BEM.blocks['i-ua'].opera) {
                $('a').click(handler);
            } else {
                // Добавляем обработчик на window
                if (win.addEventListener) {
                    win.addEventListener('beforeunload', handler);
                } else {
                    window.onbeforeunload = handler;
                }
            }
        }

        // добавляем новый чекер в список
        this._checkers.push({ fn: checker, ctx: ctx });

        return this;
    },

    /**
     * Удаляет чекер. Если не указаны параметры, удаляет все чекеры.
     * @param {Function} [fn] Функция.
     * @param {Object} [ctx] Контекст.
     */
    removeChecker: function(fn, ctx) {
        if (arguments.length === 0) {
            this._checkers.length = 0;
            return;
        }

        this._checkers = this._checkers.filter(function(ch) {
            return !(ch.fn === fn && ch.ctx === ctx);
        });

        return this;
    }
});
