// метод должен соответствовать protected/AutoBroker.pm:calcPrice
// правки надо вносить синхронно
// названия переменных взяты из бэкэнд-кода
// документация на метод https://wiki.yandex-team.ru/users/anyakey/calcPrice/
(function() {
    /**
     * Расчет покрытия
     * @param {Number} effPrice - цена рекламодателя с учетом ограничения на ДБ/остаток средств
     * @param {Array} prices - цены в ротации
     * @param {Array} probs - вероятности выпадения в ротации
     * @returns {*}
     */
    function calcCoverage(effPrice, prices, probs) {
        if (prices.length < 2 || prices.length != probs.length || effPrice < prices[0]) return 0;

        // находим, между какими элементами попадает цена
        var lowIndex = binarySearch(effPrice, prices),
            lowPrice,
            highPrice,
            k;

        // не учитываем последний элемент
        lowIndex = Math.min(prices.length - 1, lowIndex);
        // индекс предыдущего элемента (если указатель не на первом)
        lowIndex = Math.max(0, lowIndex - 1);

        lowPrice = prices[lowIndex];
        highPrice = prices[lowIndex + 1] >= effPrice ? prices[lowIndex + 1] : effPrice;

        k = (effPrice - lowPrice) / ((highPrice - lowPrice) || 1);

        return probs[lowIndex] + k * (probs[lowIndex + 1] - probs[lowIndex]);
    }

    /**
     * Бинарный поиск по массиву
     * @param {Number} value - значение
     * @param {Array} array - массов
     * @returns {Number} - индекс ближайшего снизу элемента
     */
    function binarySearch(value, array) {
        var low = 0,
            high = array.length,
            mid;

        while (low < high) {
            mid = Math.floor((low + high)) / 2;
            array[mid] < value ? (low = mid + 1) : (high = mid);
        }

        return low;
    }

    /**
     * Получение списка допустимых позиций
     * Для автотарегетинга и мало показов нет торгов
     * Для ГО на поиске - 1 ставка
     * Для стратегии "показывать под результатами поиска" нет спецразмещения
     * @param {Object} options - опции
     * @param {Object} options.guarantee - гарантия
     * @param {Object} options.premium - спецразмещение
     * @param {Boolean} options.onlyFirstGuarantee - есть только 1 гарантия
     * @returns {*}
     */
    function getAvaliablePrices(options) {
        var PRICE_PLACES = u.consts('PRICE_PLACES'),
            ENTRY_PLACES = u.consts('ENTRY_PLACES'),
            placesData = {
                guarantee: options.guarantee,
                premium: options.premium
            };

        if (options.onlyFirstGuarantee)
            return [{ price: u.auction.getDataByPlace(placesData, 'max'), place: PRICE_PLACES.GUARANTEE1 }];
        return [
            // 1-ое место в спецразмещение
            { price: u.auction.getDataByPlace(placesData, 'pmax'), place: PRICE_PLACES.PREMIUM1 },
            // 2-ое место в спецразмещение
            { price: u.auction.getDataByPlace(placesData, 'pmax2'), place: PRICE_PLACES.PREMIUM2 },
            // вход в спецразмещение
            { price: u.auction.getDataByPlace(placesData, 'pmin'), place: ENTRY_PLACES.PREMIUM },
            // 1-ое место в гарантированных показах
            { price: u.auction.getDataByPlace(placesData, 'max'), place: PRICE_PLACES.GUARANTEE1 },
            // вход в гарантированные показы
            { price: u.auction.getDataByPlace(placesData, 'min'), place: ENTRY_PLACES.GUARANTEE }
        ]
    }

    /**
     * Расчет остатка средств на кампании с учетом дневного бюджета
     * @param {Number} dayBudget - дневной бюджет
     * @param {Number} spentToday - потрачено сегодня
     * @param {Number} campRest - остаток средств на кампании
     * @returns {Number}
     */
    function calcCampRest(dayBudget, spentToday, campRest) {
        var dayBudgetRest;

        if (dayBudget > 0) {
            dayBudgetRest = dayBudget - spentToday;

            if (dayBudgetRest > 0) {
                // для кампаний совсем без денег с включённым дневным бюджетом
                // рассчитываем ставки исходя из того, что денег будет больше, чем сумма дневного бюджета
                // с появлением денег начнём рассчитывать из min(деньги, дневной бюджет)
                return campRest > 0 ? Math.min(campRest, dayBudgetRest) : dayBudgetRest;
            } else {
                // уже перетратили дневной бюджет
                return 0;
            }
        }

        return campRest;
    }

    /**
     * Метод вычисления ставки в ротации
     * @param {Object} options
     * @param {Number} options.effPrice - ставка рекламодателя с ограничением ДБ/остатка средств на кампании
     * @param {Number} options.price - ставка рекламодателя
     * @param {Object} options.rotation - данные ставок и вероятностей в ротации
     * @param {Array} options.rotationPrices - ставки торгов в ротации
     * @param {Object} options.brokerPriceZero - нулевая ставка
     * @returns {Object}
     */
    function calcCoveragePrice(options) {
        var effPrice = options.effPrice,
            rotation = options.rotation,
            rotationPrices = (options.rotationPrices || '').split(',').map(Number),
            coverage = calcCoverage(effPrice, rotation.prices, rotation.probs) / 1e6,
            brokerPrice = options.brokerPriceZero, // обнулили brockerPrice
            result = {};

        if (coverage) {
            // ограничение остатком средств или ДБ
            result.truncated = +(options.price > effPrice);

            // вычисляем цену автоброкера
            brokerPrice.bidPrice = brokerPrice.amnestyPrice = rotationPrices
                // ищем ставки ротации, которые меньше цены/остатка средств на кампании
                .filter(function(val) { return val <= effPrice; })
                // берем самый маленькую ставку или ставку, ограниченную остатком средств
                .reduce(function(result, value) { return Math.min(result, value) }, Number.MAX_VALUE) || effPrice;
        }

        result.brokerPrice = brokerPrice;

        return result;
    }

    /**
     * @param {object} data
     * @param {number} data.price ставка пользователя (для ручных стратегий) или ставка от торгов (для автоматики)
     * @param {Array} data.guarantee - ставки гарантии (от торгов)
     * @param {Array} data.premium - ставки спецразмещения (от торгов)
     * @param {String} data.larr '1000:1000,2000:2000|1000,2000' - данные для ротиации
     * @param {String} data.minPrice - минимальная ставка с ТЗ торгов
     * @param {number} data.campRest - остаток средств на кампании
     * @param {number} data.dayBudget - сумма дневного ограничения бюджета (0, если ограничение не установлено)
     * @param {number} data.spentToday - истраченная кампанией за сегодня сумма
     * @param {Object} data.strategy - стратегия кампании
     * @param {'Yes'|'No'} data.autobudget - флаг, что стратегия автобюджетная
     * @param {Number} data.autobudgetBid - максимальная цена клика в автобюджетной стратегии
     * @param {Number} data.timetargetingCoef - текущий коэффициент таймтаргетинга
     * @param {String} data.currency -  валюта аккаунта
     * @param {Boolean} data.onlyFirstGuarantee - флаг, что допустима только 1 гарантия
     *
     * @return
     *      {
     *          price: Number|0 - цена на поиске (0, если не показываем)
     *          truncated: 1|0 - флаг, что цена ограничена сверху остатком средств или дневным бюджетом
     *          placeName: Number - номер места, куда попали по массиву констант PRICE_PLACES
     *          placeNameWithoutCoef: Number - номер места, куда попали по массиву констант PRICE_PLACES без учета
     *     таймтаргетинга
     *      }
     *
     */
    function calcPrice(data) {
        var currency = data.currency,
            MIN_PRICE = u.currencies.getConst(currency, 'MIN_PRICE'),
            PRICE_PLACES = u.consts('PRICE_PLACES'),
            // данные от бэкэнда про
            // а) ставки и вероятности попадания в ротацию
            // б) автоброкерные ставки на ротацию
            larrParts = ('' + data.larr).split('|'),
            price = data.price,
            // значения bidPrice amnestyPrice в массивах premium и guarantee приходят в виде val * 1e6
            // для точности вычислений
            premium = data.premium.map(function(item) {
                return {
                    bidPrice: +item.bid_price,
                    amnestyPrice: +item.amnesty_price
                };
            }),
            guarantee = data.guarantee.map(function(item) {
                return {
                    bidPrice: +item.bid_price,
                    amnestyPrice: +item.amnesty_price
                };
            }),
            // считаем ставки и вероятности попадания в ротацию
            // вероятности дают частоту выпада в ротации (= охват аудитории на поиске)
            bottom = larrParts[0].split(',').reduce(function(obj, larr) {
                var parts = larr.split(':'),
                    index = obj.prices.length;

                parts[1] && (obj.probs[index] = +parts[1]);
                obj.prices[index] = +parts[0];

                return obj;
            }, { prices: [], probs: [] }),
            // ставки на ротацию
            bottomAutobroker = larrParts[1] && larrParts[1].split(',') || [],
            // домножаем на 1e6 для точности вычислений
            minPrice = Math.floor(1e6 * (data.minPrice || 0) + 0.5),
            campRest = Math.floor(1e6 * (data.campRest || 1000) + 0.5),
            dayBudget = Math.floor(1e6 * data.dayBudget),
            spentToday = Math.floor(1e6 * (data.spentToday || 0)),
            autobudgetBid = Math.floor(1e6 * (data.autobudgetBid || 0) + 0.5),
            timeTargetCoef = data.timetargetingCoef,
            // стратегия, если нет - считаем, что нераздельное размещение
            // поиск и сеть включены
            // на поиске - наивысшая позиция
            // на поиске -  максимальный охват
            strategy = data.strategy || { name: '', search: { name: 'default' }, net: { name: 'default' } },
            // поисковая стратегия
            searchStrategy = strategy.search,
            // нулевая ставка ставка
            brokerPriceZero = { bidPrice: 0, amnestyPrice: 0 },
            // минимальная ставка для попадания хоть куда-нибудь
            effMinPrice = Math.min(premium[premium.length - 1].bidPrice,
                Math.max(1e6 * MIN_PRICE, bottomAutobroker[0] || 0)),
            autobudget = data.autobudget,
            // флаг о том, что ставка чем-то ограничена (остатком средств или дневным бюджетом)
            truncated = 0,
            // цена рекламодателя, ограниченная с учетом остатка средств и дневного бюджета
            effPrice,
            // цена клика по позиции торгов
            brokerPrice,
            // место, куда попали в торгах
            place,
            // доступные позиции вообще
            availablePrices,
            // данные без учета таймтаргетинга
            priceDataWithoutCoef,
            // вспомогательный объект для хранения промежуточных данных
            auxObject;

        // рассчитываем остаток средств на кампании с учетом дневного бюджеиа
        campRest = calcCampRest(dayBudget, spentToday, campRest);

        // расчет цены с поправкой на таймтаргетинг
        if (autobudget != 'Yes' && timeTargetCoef > 0 && timeTargetCoef < 100) {
            // расчёт без учёта параметра timeTargetCoef
            priceDataWithoutCoef = this.calcPrice({
                price: data.price,
                guarantee: data.guarantee,
                premium: data.premium,
                larr: data.larr,
                minPrice: data.minPrice,
                campRest: data.campRest,
                dayBudget: data.dayBudget,
                spentToday: data.spentToday,
                strategy: data.strategy,
                autobudget: data.autobudget,
                autobudgetBid: data.autobudgetBid,
                currency: data.currency,
                onlyFirstGuarantee: data.onlyFirstGuarantee
            });

            price *= timeTargetCoef / 100;
        }

        // цена не может быть меньше константы MIN_PRICE
        price = Math.floor(1e6 * Math.max(price, MIN_PRICE) + 0.5);

        // при автобюджете с указанной максимальной ставкой цена не может быть выше этой ставки
        autobudget == 'Yes' && autobudgetBid && (price = Math.min(price, autobudgetBid));

        // если есть остаток средств, то цена не может быть больше, чем остаток средств
        effPrice = campRest > 0 ? Math.min(campRest, price) : price;

        // рассчитываем допустимые позиции
        availablePrices = getAvaliablePrices({
            onlyFirstGuarantee: data.onlyFirstGuarantee,
            premium: premium,
            guarantee: guarantee
        });

        // вычисляем цену
        availablePrices.some(function(item) {
            var bidPrice = item.price.bidPrice;

            // случай, когда мы попали в позицию, но ограничены остатком на кампанию
            // или дневным бюджетом
            if (bidPrice > effPrice && bidPrice <= price) {
                truncated = 1;
            } else if (bidPrice <= effPrice) { // попали в позицию и средств достаточно
                brokerPrice = item.price;

                return true;
            }

            return false;
        });

        // если не нашлось место в гарантии - считаем покрытие
        if (!(brokerPrice && brokerPrice.bidPrice)) {
            // если есть только 1 гарантия - то не можем попасть в ротацию,
            // кидаем сразу brokerPriceZero
            if (data.onlyFirstGuarantee) {
                brokerPrice = brokerPriceZero;
            } else {
                // считаем ставки в ротации
                auxObject = calcCoveragePrice({
                    effPrice: effPrice,
                    price: price,
                    rotation: bottom,
                    rotationPrices: larrParts[1],
                    brokerPriceZero: brokerPriceZero
                });

                truncated = auxObject.truncated;
                brokerPrice = auxObject.brokerPrice;
            }
        }

        // обнуляем ставку на поиске, если она меньше минимальной ставки торгов
        // "Нижний" минимум - до 8-го места
        if (brokerPrice.bidPrice < Math.max(minPrice, effMinPrice)) {
            brokerPrice = brokerPriceZero;
        }

        // учитываем автобюджет: если есть ограничение цены клика и оно меньше минимальной ставки торгов
        // или цена рекламодателя ниже минимальной ставки торгов - обнуляем ставку
        if (autobudget == 'Yes' && minPrice && (autobudgetBid && minPrice < autobudgetBid || price < minPrice)) {
            brokerPrice = brokerPriceZero;
        }

        // вычисляем место - фильтруем все подходящие позиции
        // дописываем в конец ротацию (на случай пустого массива, если никуда не попали)
        // и берем самую верхнюю из них
        place = availablePrices.filter(function(position) {
            return position.price && effPrice >= position.price.bidPrice;
        }).concat(PRICE_PLACES.ROTATION)[0].place || 0;

        return {
            price: u.currencies.roundPriceToStep(brokerPrice.amnestyPrice / 1e6, currency, 'down'),
            truncated: +!!truncated,
            placeName: place,
            placeNameWithoutCoef: (priceDataWithoutCoef || {}).placeName || place
        };
    }

    u.register({
        autobroker: {
            calcPrice: calcPrice
        }
    });
}());
