(function() {
    u.register({
        'content-categories-targeting': {
            /**
             * Преобразовывает серверные данные списка категорий и жанров
             * @param {Array} cryptaSegments серверные данные крипты
             * @returns { contentCategoriesTree, contentCategoriesIds }
             */
            getSegmentsIdsAndTree: function(cryptaSegments) {
                if (!cryptaSegments) {

                    return {
                        contentCategoriesTree: [],
                        contentCategoriesIds: []
                    }
                }

                var categories = u['crypta'].splitSegmentsByType(cryptaSegments).content_category || [],
                    genres = u['crypta'].splitSegmentsByType(cryptaSegments).content_genre || [],
                    contentCategories = categories.concat(genres);

                return {
                    contentCategoriesTree: u['content-categories-targeting'].getSegmentsTree(contentCategories),
                    contentCategoriesIds: u['content-categories-targeting'].getSegmentsIds(contentCategories)
                }
            },
            /**
             * Возвращает хеш всех категорий и жанров
             * @param {Array<CryptaSegmentServer>} cryptaSegments серверные данные крипты c типом content_category и content_genre
             * @return {{
             *    [id]: {Object}
             * }}
             */
            getSegmentsIds: function(cryptaSegments) {
                var ids = {};

                cryptaSegments.forEach(function(segment) {
                    ids[segment.id] = {
                        name: segment.name,
                        parentId: segment.parent_id,
                        isRestChildForbidden: segment.isRestChildForbidden // сегмент "Остальное"
                    };
                });

                return ids;
            },
            /**
             * Возвращает дерево всех категорий и жанров
             * @param {Array<CryptaSegmentServer>} cryptaSegments серверные данные крипты c типом content_category и content_genre
             * @return {Object}
             */
            getSegmentsTree: function(cryptaSegments) {
                if (cryptaSegments.length === 0) {
                    return {};
                }

                var groups = {};

                cryptaSegments.forEach(function(segment) {
                    var parentId = segment.parent_id;

                    if (!parentId && parentId !== 0) {
                        throw new Error('interest segments without parent');
                    }

                    if (!groups[parentId]) {
                        groups[parentId] = [];
                    }

                    groups[parentId].push(segment);
                });

                // сортировка по установленному порядку или по алфавиту
                u._.forOwn(groups, function(group) {
                    group.sort(function(a, b) {
                        if (a.order && b.order) {
                            return a.order - b.order;
                        }

                        return a.name.localeCompare(b.name);
                    });
                });

                return groups;
            },
            /**
             * Сортировка выбранных категорий и жанров в порядке расположения их в дереве
             * @param {Array} checkedInterestsIds - список выбранных категорий и жанров
             * @param {Object} interestsTree - дерево категорий и жанров
             * @param {string} parentId - id родительского сегмента
             * @param {Array} sortedIds - результирующий список категорий и жанров
             * @returns {Array}
             */
            getSortedCheckedInterests: function(checkedInterestsIds, interestsTree, parentId, sortedIds) {
                var interests = interestsTree[parentId];

                if (interests && interests.length) {
                    interests.forEach(function(interest) {
                        var interestId = interest.id;

                        if (checkedInterestsIds.includes(interestId)) {
                            sortedIds.push(interestId);
                        }

                        u['content-categories-targeting'].getSortedCheckedInterests(checkedInterestsIds, interestsTree, interestId, sortedIds);
                    });
                }

                return sortedIds;
            },
            /**
             * Преобразование отсортированного списка категорий и жанров для отображения их в интерфейсе
             * @param {Array} sortedCheckedList - отсортированный список выбранных категорий и жанров
             * @param {Object} interestsIds - хеш всех категорий и жанров
             * @param {Object} interestsTree - дерево всехм категорий и жанров
             * @returns {Array}
             */
            getNestedCheckedList: function(sortedCheckedList, interestsIds, interestsTree ) {
                var currentParentId = 0; // DEFAULT_PARENT_ID

                return sortedCheckedList.reduce(function(sortedNestedList, id) {
                    var parentId = interestsIds[id].parentId,
                        sortedNestedListLength = sortedNestedList.length;

                    if (!parentId || parentId === 0) {
                        sortedNestedList.push({
                            title: '',
                            segments: [id]
                        });

                        return sortedNestedList;
                    }

                    if (parentId !== currentParentId) {
                        currentParentId = parentId;

                        sortedNestedList.push({
                            title: interestsIds[currentParentId].name,
                            segments: [id],
                            showShortSegmentsText: !(interestsIds[currentParentId].isRestChildForbidden),
                            count: interestsTree[currentParentId].length
                        });
                    } else {
                        sortedNestedListLength ?
                            sortedNestedList[sortedNestedListLength - 1].segments.push(id) :
                            sortedNestedList.push({
                                title: interestsIds[currentParentId].name,
                                segments: [id],
                                showShortSegmentsText: !(interestsIds[currentParentId].isRestChildForbidden),
                                count: interestsTree[currentParentId].length
                            });
                    }

                    return sortedNestedList;
                }, []);
            },
            /**
             * Преобразование выбранных категорий и жанров в строку для отображения их в интерфейсе
             * @param {Array} checkedInterestsId - список выбранных категорий и жанров
             * @param {Object} interestsTree - дерево всехм категорий и жанров
             * @param {Object} interestsIds - хеш всех категорий и жанров
             * @returns {String}
             */
            getContentCategoriesText: function(checkedInterestsId, interestsTree, interestsIds) {
                var sortedCheckedInterests = u['content-categories-targeting'].getSortedCheckedInterests(checkedInterestsId, interestsTree, 0, []),
                    nestedCheckedList = u['content-categories-targeting'].getNestedCheckedList(sortedCheckedInterests, interestsIds, interestsTree);

                return nestedCheckedList.map(function(item) {
                    var areAllChildrenChecked = item.showShortSegmentsText && item.count === item.segments.length,
                        childrenList = item.segments.map(function(id) { return interestsIds[id].name}).join(', ');

                    return item.title ? item.title + (areAllChildrenChecked ? '' : ': ' + childrenList) : childrenList;
                }).join(' · ');
            }
        }
    });
})();
