describe('i-utils__dna', function() {
    describe('getUserInfo()', function () {
        function getData() {
            return {
                UID: 'uid-1',
                uname: 'login-1',
                display_name: {
                    avatar: {
                        default: {
                            content: '1/2-3'
                        }
                    },
                    name: 'display-name-1'
                },
                address_list: {
                    address: [
                        {
                            default: '0',
                            content: 'login-1.not-default@yandex.ru'
                        },
                        {
                            default: '1',
                            content: 'login-1.default@yandex.ru'
                        },
                        {
                            default: '0',
                            content: 'login-1.not-default@yandex.ru'
                        }
                    ]
                },
                blackbox_users: {
                    'uid-2': {
                        uid: {
                            content: 'uid-2'
                        },
                        login: {
                            content: 'login-2'
                        },
                        display_name: {
                            avatar: {
                                default: {
                                    content: '4/5-6'
                                }
                            },
                            name: 'display-name-2'
                        },
                        'address-list': {
                            address: [
                                {
                                    default: '1',
                                    content: 'login-2.default@yandex.ru'
                                },
                                {
                                    default: '0',
                                    content: 'login-2.not-default@yandex.ru'
                                }
                            ]
                        }
                    },
                    'uid-3': {
                        uid: {
                            content: 'uid-3'
                        },
                        login: {
                            content: 'login-3'
                        },
                        display_name: {
                            avatar: {
                                default: {
                                    content: '7/8-9'
                                }
                            },
                            name: 'display-name-3'
                        },
                        'address-list': {
                            address: [
                                {
                                    default: '0',
                                    content: 'login-3.not-default@yandex.ru'
                                },
                                {
                                    default: '1',
                                    content: 'login-3.default@yandex.ru'
                                }
                            ]
                        }
                    }
                },
                allow_more_users: '1'
            };
        }
    
        function eachAccount(data, fn) {
            var uids = Object.keys(data.blackbox_users);
            uids.forEach(function(uid) {
                fn(data.blackbox_users[uid]);
            });
        }

        it('should return user accounts with multisession data', function() {
            var data = getData();
            var userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-2',
                canAddMore: true,
                accounts: [
                    {
                        uid: 'uid-2',
                        login: 'login-2',
                        defaultEmail: 'login-2.default@yandex.ru',
                        displayName: {
                            name: 'display-name-2',
                            avatarId: '4/5-6'
                        }
                    },
                    {
                        uid: 'uid-3',
                        login: 'login-3',
                        defaultEmail: 'login-3.default@yandex.ru',
                        displayName: {
                            name: 'display-name-3',
                            avatarId: '7/8-9'
                        }
                    }
                ]
            });
        });

        it('should return user accounts without multisession data', function() {
            var data = getData();
            delete data.blackbox_users;
            delete data.allow_more_users;

            var userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-1',
                canAddMore: false,
                accounts: [
                    {
                        uid: 'uid-1',
                        login: 'login-1',
                        defaultEmail: 'login-1.default@yandex.ru',
                        displayName: {
                            name: 'display-name-1',
                            avatarId: '1/2-3'
                        }
                    }
                ]
            });
        });

        it('should return undefined without data.UID', function() {
            var data = getData();
            delete data.UID;

            var userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.undefined;
        });

        it('should return undefined without data.uname', function() {
            var data = getData();
            delete data.uname;

            var userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.undefined;
        });

        it('should works without address-list', function() {
            var data = getData();
            delete data.address_list;
            eachAccount(data, function(account) {
                delete account['address-list'];
            });

            var userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-2',
                canAddMore: true,
                accounts: [
                    {
                        uid: 'uid-2',
                        login: 'login-2',
                        defaultEmail: undefined,
                        displayName: {
                            name: 'display-name-2',
                            avatarId: '4/5-6'
                        }
                    },
                    {
                        uid: 'uid-3',
                        login: 'login-3',
                        defaultEmail: undefined,
                        displayName: {
                            name: 'display-name-3',
                            avatarId: '7/8-9'
                        }
                    }
                ]
            });

            delete data.blackbox_users;
            userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-1',
                canAddMore: true,
                accounts: [
                    {
                        uid: 'uid-1',
                        login: 'login-1',
                        defaultEmail: undefined,
                        displayName: {
                            name: 'display-name-1',
                            avatarId: '1/2-3'
                        }
                    }
                ]
            });
        });

        it('should works without default email', function() {
            var data = getData();
            data.address_list.address.forEach(function(email) {
                email.default = '0';
            });
            eachAccount(data, function(account) {
                account['address-list'].address.forEach(function(email) {
                    email.default = '0';
                });
            });

            var userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-2',
                canAddMore: true,
                accounts: [
                    {
                        uid: 'uid-2',
                        login: 'login-2',
                        defaultEmail: undefined,
                        displayName: {
                            name: 'display-name-2',
                            avatarId: '4/5-6'
                        }
                    },
                    {
                        uid: 'uid-3',
                        login: 'login-3',
                        defaultEmail: undefined,
                        displayName: {
                            name: 'display-name-3',
                            avatarId: '7/8-9'
                        }
                    }
                ]
            });

            delete data.blackbox_users;
            userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-1',
                canAddMore: true,
                accounts: [
                    {
                        uid: 'uid-1',
                        login: 'login-1',
                        defaultEmail: undefined,
                        displayName: {
                            name: 'display-name-1',
                            avatarId: '1/2-3'
                        }
                    }
                ]
            });
        });

        it('should works without display name', function() {
            var data = getData();
            delete data.display_name;
            eachAccount(data, function(account) {
                delete account.display_name;
            });

            var userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-2',
                canAddMore: true,
                accounts: [
                    {
                        uid: 'uid-2',
                        login: 'login-2',
                        defaultEmail: 'login-2.default@yandex.ru',
                        displayName: {
                            name: 'login-2',
                            avatarId: '0/0-0'
                        }
                    },
                    {
                        uid: 'uid-3',
                        login: 'login-3',
                        defaultEmail: 'login-3.default@yandex.ru',
                        displayName: {
                            name: 'login-3',
                            avatarId: '0/0-0'
                        }
                    }
                ]
            });

            delete data.blackbox_users;
            userInfo = u.getUserInfo(data);
            
            expect(userInfo).to.be.eql({
                defaultUid: 'uid-1',
                canAddMore: true,
                accounts: [
                    {
                        uid: 'uid-1',
                        login: 'login-1',
                        defaultEmail: 'login-1.default@yandex.ru',
                        displayName: {
                            name: 'login-1',
                            avatarId: '0/0-0'
                        }
                    }
                ]
            });
        });

        it('should works without avatar', function() {
            var data = getData();
            delete data.display_name.avatar;
            eachAccount(data, function(account) {
                delete account.display_name.avatar;
            });

            var userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-2',
                canAddMore: true,
                accounts: [
                    {
                        uid: 'uid-2',
                        login: 'login-2',
                        defaultEmail: 'login-2.default@yandex.ru',
                        displayName: {
                            name: 'display-name-2',
                            avatarId: '0/0-0'
                        }
                    },
                    {
                        uid: 'uid-3',
                        login: 'login-3',
                        defaultEmail: 'login-3.default@yandex.ru',
                        displayName: {
                            name: 'display-name-3',
                            avatarId: '0/0-0'
                        }
                    }
                ]
            });

            delete data.blackbox_users;
            userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-1',
                canAddMore: true,
                accounts: [
                    {
                        uid: 'uid-1',
                        login: 'login-1',
                        defaultEmail: 'login-1.default@yandex.ru',
                        displayName: {
                            name: 'display-name-1',
                            avatarId: '0/0-0'
                        }
                    }
                ]
            });
        });

        it('should return false if adding new users disallowed', function() {
            var data = getData();
            data.allow_more_users = '0';
            
            var userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-2',
                canAddMore: false,
                accounts: [
                    {
                        uid: 'uid-2',
                        login: 'login-2',
                        defaultEmail: 'login-2.default@yandex.ru',
                        displayName: {
                            name: 'display-name-2',
                            avatarId: '4/5-6'
                        }
                    },
                    {
                        uid: 'uid-3',
                        login: 'login-3',
                        defaultEmail: 'login-3.default@yandex.ru',
                        displayName: {
                            name: 'display-name-3',
                            avatarId: '7/8-9'
                        }
                    }
                ]
            });
        });

        it('should filter accounts without uid', function() {
            var data = getData();
            delete data.blackbox_users['uid-2'].uid;

            var userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-3',
                canAddMore: true,
                accounts: [
                    {
                        uid: 'uid-3',
                        login: 'login-3',
                        defaultEmail: 'login-3.default@yandex.ru',
                        displayName: {
                            name: 'display-name-3',
                            avatarId: '7/8-9'
                        }
                    }
                ]
            });
        });

        it('should filter accounts without login', function() {
            var data = getData();
            delete data.blackbox_users['uid-2'].login;

            var userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-3',
                canAddMore: true,
                accounts: [
                    {
                        uid: 'uid-3',
                        login: 'login-3',
                        defaultEmail: 'login-3.default@yandex.ru',
                        displayName: {
                            name: 'display-name-3',
                            avatarId: '7/8-9'
                        }
                    }
                ]
            });
        });

        it('should return UID user if all accounts were filtered', function() {
            var data = getData();
            delete data.blackbox_users['uid-2'].login;
            delete data.blackbox_users['uid-3'].login;

            var userInfo = u.getUserInfo(data);

            expect(userInfo).to.be.eql({
                defaultUid: 'uid-1',
                canAddMore: true,
                accounts: [
                    {
                        uid: 'uid-1',
                        login: 'login-1',
                        defaultEmail: 'login-1.default@yandex.ru',
                        displayName: {
                            name: 'display-name-1',
                            avatarId: '1/2-3'
                        }
                    }
                ]
            });
        });
    });
});
