/**
 * @typedef {Object} BackendValidationError объект ошибки про конкретную сущность
 * @property {String} code код ошибки из API
 * @property {String} name идентификатор типа ошибки
 * @property {String} text краткое описание ошибки
 * @property {String} description развернутое описание ошибки
 */

(function() {

    var
        /**
         * Строит путь по имеющемуся и добавочному ключу.
         * При числовом ключе добавляется строка-индекс вида '[N]'.
         * При строковом ключе добавляется строка-аксессор вида '.keyValue', если существующий путь пуст то просто
         * 'keyValue'. Служебные ключи array_errors, object_errors, generic_errors игнорируются.
         * @param {String} path существующий путь
         * @param {String|Number} key ключ для добавления к пути
         * @returns {String}
         */
        buildPath = function(path, key) {
            // числовые ключи могут быть как у массива так и у объекта (вида { 2: ... } )
            if (key !== '' && !isNaN(key)) return path + '[' + key + ']';
            // игнорируем служебные ключи, они нужны в ошибках только при их вложенном состоянии
            if (u._.includes(['array_errors', 'object_errors', 'generic_errors'], key)) return path;

            return path ? path + '.' + key : key;
        },
        /**
         * Проверяет аргумент на соответствие условию объекта ошибки
         * @param {Object|Object[]|*} obj
         * @returns {Boolean}
         */
        isMatch = function(obj) {
            // считаем что упёрлись в ошибку, если все элементы массива соответствуют условию объекта ошибки
            if (u._.isArray(obj)) return !u._.isEmpty(obj) && u._.every(obj, isMatch);
            // объектом ошибки считаем тот объект, который не содержит в себе ничего кроме примитивов
            if (u._.isObject(obj)) {
                return !u._.some(u._.values(obj), function(val) {
                    return val === null || u._.isObject(val);
                });
            }

            return false;
        },
        /**
         * @param {Object|Object[]} errors
         * @param {String} path итеративно наращиваемый путь, используемый для формирования ключей хэша
         * @param {Object} result результирующий плоский список
         * @returns {Object}
         */
        flatten = function(errors, path, result) {
            if (isMatch(errors)) {
                result[path] = u._.isArray(errors) ? u._.flatten(errors, true) : [errors];
                addPath(result[path], path);
            } else if (u._.isArray(errors)) {
                errors.forEach(function(val, i) { flatten(val, buildPath(path, i), result); });
            } else if (u._.isObject(errors)) {
                u._.forOwn(errors, function(val, key) { flatten(val, buildPath(path, key), result); });
            }

            return result;
        },

        /**
         * Добавляет к объекту ошибки path
         * @param {Array|Object} errors
         */
        addPath = function(errors, path) {

            var pathToObj = function(err) {
                err._path = path;
                return err;
            }

            if (u._.isArray(errors)) {
                return errors.map(pathToObj)
            } else if (u._.isObject(errors)) {
                return pathToObj(errors);
            }

            return errors;
        };

    u.register({

        error: {

            /**
             * Создаёт сборщик пути
             * @param {String} [path] префикс пути
             * @returns {Function}
             */
            createPathBuilder: function(path) {
                path || (path = '');

                /**
                 * Строит по полученному списку аргументов путь с учётом префикса пути
                 * @param {String} [key...]
                 * @returns {String}
                 */
                return function(key) {
                    return arguments.length ? u._.reduce(arguments, buildPath, path) : path;
                };
            },

            /**
             * Превращает вложенный объект ошибок серверной валидации в плоский хэш с ключами-путями и
             * значениями-массивами ошибок
             * @param {Object|Object[]} errors
             * @deprecated
             * @returns {Object} объект вида { String path: Object[] errors}
             */
            flattenServer: function(errors) {
                return flatten(errors, '', {});
            },

            /**
             * Превращает вложенный объект ошибок клиентской валидации в плоский хэш с ключами-путями и
             * значениями-массивами ошибок
             * @param {Object|Object[]} errors
             * @param {String} [basePath=''] префикс базового пути
             * @deprecated
             * @returns {Object} объект вида { String path: Object [] errors}
             */
            flattenClient: function(errors, basePath) {
                return flatten(errors, basePath || '', {});
            },

            /**
             * Превращает объект ошибок валидации в "серверном" формате в плоский хеш с ключами-путями и значениями -
             * массивами ошибок
             * @param {Object|Object[]} errors
             * @param {String} [basePath=''] префикс базового пути
             * @returns {Object} объект вида { String path: Object [] errors}
             */
            flatten: function(errors, basePath) {
                return flatten(errors, basePath || '', {});
            },

            /**
             * Преобразует результат валидации модели в универсальный серверный формат
             * @param {Object} validationResult результат выладации модели(model.validate())
             * @returns {Object}
             */
            clientToUniversal: function clientToUniversal(validationResult) {
                if (Array.isArray(validationResult)) {
                    return { array_errors: validationResult.map(clientToUniversal) };
                }

                if (validationResult.valid) {
                    return { object_errors: {} };
                }

                var errorsData = validationResult.errorsData,
                    errors = Object.keys(errorsData).reduce(function(acc, key) {
                        acc[key] = errorsData[key].map(function(error) {
                            return { text: error.text };
                        });

                        return acc;
                    }, {});

                return {
                    object_errors: errors
                };
            }

        }
    });
})();
