/**
 * @typedef {Object} Goal
 * @property {String} id
 * @property {String} name
 * @property {'Active' | 'Deleted'} status
 * @property {String} [domain]
 * @property {Boolean} isDefault
 */

(function() {
    u.register({
        goals: {
            GOAL_STATUS: {
                deleted: 'Deleted',
                active: 'Active'
            },

            /**
             * Возвращает название цели по данным о ней
             * @param {Goal} goal
             * @returns {String}
             */
            getGoalName: function(goal) {
                // у удаленных целей имя может не приходить, так как у пользователя нет доступа к цели,
                // для таких целей отображаем прочерк
                return goal.name ? u.goals.getGoalNameWithDomain(goal) : ('&mdash; (id: ' + goal.id + ')');
            },

            /**
             * Возвращает название цели с информацией о домене
             * @param {Goal} goal
             * @returns {String}
             */
            getGoalNameWithDomain: function(goal) {
                var name = [goal.domain, goal.name]
                    .filter(function(val) { return val; })
                    .join(' :: ');

                return name.replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;');
            },

            /**
             * Возвращает информацию о выбранных целях в серверном формате
             * @param {Array<{goalId: String, price: Number}>} value
             * @returns {Array<{goal_id: String, value: Number}>}
             */
            transformValueToServerFormat: function(value) {
                return value.map(function(valueItem) {
                    return {
                        goal_id: valueItem.goalId,
                        value: valueItem.price
                    }
                });
            },

            /**
             * Возвращает информацию о выбранных целях в клиентском формате на основе значения с сервера
             * @param {Array<{goal_id: String, value: Number}>} value
             * @returns {Array<{goalId: String, price: Number}>}
             */
            transformValueFromServerFormat: function(value) {
                return value.map(function(valueItem) {
                    return {
                        goalId: valueItem.goal_id,
                        price: valueItem.value
                    }
                });
            },

            /**
             * Возвращает информацию о о доступных целях в клиентском формате на основе значения с сервера
             * @param {Object.<{goal_id: String, name: String, status: 'Active' | 'Deleted', domain: [String], is_default_goal: [number]}>} goals
             * данные о целях, приходящие от сервера
             * @returns {Object.<Goal>}
             */
            transformGoalsFromServerFormat: function(goals) {
                return u._.reduce(goals, function(res, goal) {
                    res[goal.goal_id] = {
                        id: String(goal.goal_id),
                        name: goal.name,
                        status: goal.status,
                        domain: goal.goal_domain,
                        isDefault: !!goal.is_default_goal
                    };

                    return res;
                }, {});
            }
        }
    });
})();
