describe('i-utils__regions', function() {

    describe('.getGeoNames', function() {
        [
            {
                funcArgs: [],
                description: 'Если вызвали без параметров, должен возвращать названия ' +
                    'из первого уровня дерева регионов через запятую',
                expectedResult: 'Россия, СНГ (исключая Россию), Европа, Азия, Африка, ' +
                    'Северная Америка, Южная Америка, Австралия и Океания'
            },
            {
                funcArgs: [''],
                description: 'Если единственным аргументом передали пустую строку, должен возвращать названия ' +
                    'из первого уровня дерева регионов через запятую',
                expectedResult: 'Россия, СНГ (исключая Россию), Европа, Азия, Африка, ' +
                    'Северная Америка, Южная Америка, Австралия и Океания'
            },
            {
                funcArgs: [0],
                description: 'Если единственным аргументом передали 0, должен возвращать названия ' +
                    'из первого уровня дерева регионов через запятую',
                expectedResult: 'Россия, СНГ (исключая Россию), Европа, Азия, Африка, ' +
                    'Северная Америка, Южная Америка, Австралия и Океания'
            },
            {
                funcArgs: [999999999999999999],
                description: 'Если единственным аргументом передали не существующий id региона, должен возвращать названия ' +
                    'из первого уровня дерева регионов через запятую',
                expectedResult: 'Россия, СНГ (исключая Россию), Европа, Азия, Африка, ' +
                    'Северная Америка, Южная Америка, Австралия и Океания'
            },
            {
                funcArgs: [1],
                description: 'должен возвращать название региона если передали его id',
                expectedResult: 'Москва и область'
            },
            {
                funcArgs: ['-1, -10723'],
                description: 'Если передали id регионов со знаком "-" должен возвращать ' +
                    '"Все (кроме: {названия регионов id которых передали со знаком "-", перечисленные через запятую})"',
                expectedResult: 'Все (кроме: Москва и область, Дмитров)'
            },
            {
                funcArgs: ['1, -10723, -214, 10995, -1107, -970'],
                description: 'Если передали id региона за которым следуют id со знаком "-" должен возвращать ' +
                    '"{Название региона} (кроме: {названия регионов id которых передали со знаком "-",' +
                    ' перечисленные через запятую})"',
                expectedResult: 'Москва и область (кроме: Дмитров, Долгопрудный), ' +
                    'Краснодарский край (кроме: Анапа, Новороссийск)'
            },
            {
                funcArgs: ['1, 10995', ' & '],
                description: 'Если вторым аргументом передали разделитель, должен возвращать ' +
                    'названия регионов id которых передали разделенные переданным разделителем',
                expectedResult: 'Москва и область & Краснодарский край'
            },
            {
                funcArgs: ['-969, -10867', ' & '],
                description: 'Если в первом аргументе присутствуют несколько id со знаком "-" и ' +
                    'вторым аргументом передали разделитель, должен возвращать ' +
                    'названия регионов id которых передали разделенные переданным разделителем в секции "кроме:"',
                expectedResult: 'Все (кроме: Выборг & Гатчина)'
            }
        ].forEach(function(testData) {
            it(testData.description, function() {
                expect(u.getGeoNames.apply(u, testData.funcArgs)).to.be.equal(testData.expectedResult);
            });
        });
    });

    describe('.getRegionsTreeInfo', function() {
        var getTree = function() {

                return [
                    {
                        id: '225',
                        name: 'Россия',
                        inner: [
                            {
                                id: '3',
                                name: 'Центр',
                                inner: [
                                    {
                                        id: '1',
                                        name: 'Москва и область',
                                        inner: [
                                            {
                                                id: '10716',
                                                name: 'Балашиха',
                                                inner: null
                                            },
                                            {
                                                id: '10717',
                                                name: 'Бронницы',
                                                inner: null
                                            },
                                            {
                                                id: '21735',
                                                name: 'Балашиха2',
                                                inner: [
                                                    {
                                                        id: '217351',
                                                        name: 'Трямдия2',
                                                        inner: null
                                                    }
                                                ]
                                            },
                                            {
                                                id: '98582',
                                                name: 'Балашиха3',
                                                inner: [
                                                    {
                                                        id: '985821',
                                                        name: 'Трямдия',
                                                        inner: null
                                                    }
                                                ]
                                            },
                                            {
                                                id: '214',
                                                name: 'Балашиха4',
                                                inner: [
                                                    {
                                                        id: '2141',
                                                        name: 'Трямдия3',
                                                        inner: null
                                                    }
                                                ]
                                            }
                                        ]
                                    },
                                    {
                                        id: '10650',
                                        name: 'Брянская область',
                                        inner: null
                                    }
                                ]
                            },
                            {
                                id: '52',
                                name: 'Урал',
                                inner: null
                            }
                        ]
                    },
                    {
                        id: '241',
                        name: 'Африка'
                    },
                    {
                        id: '10',
                        name: 'А',
                        inner: [
                            {
                                id: '20',
                                name: 'У',
                                inner: null
                            }
                        ]
                    },
                    {
                        id: '11',
                        name: 'Страна 1',
                        inner: [
                            {
                                id: '111',
                                name: 'Край страны 1',
                                inner: [
                                    {
                                        id: '1111',
                                        name: 'Край края страны 1',
                                        inner: null
                                    }
                                ]
                            }
                        ]
                    }
                ];
            },
            tree,
            campaignData = {
                cid: '1',
                mediaType: 'text',
                groupNames: { '1': 'Группа 1', '2': 'Группа 2'}
            },
            constStub,
            getUrlStab;

        beforeEach(function() {
            tree = getTree();
            constStub = sinon.stub(u, 'consts'),
            getUrlStab = sinon.stub(u, 'getUrl').callsFake(function() { return 'yandex.ru'; });

            constStub.withArgs('bidOffset').returns(100);
        });

        afterEach(function() {
            tree = null;
            constStub.restore();
            getUrlStab.restore();
        });

        it('Должен возвращать объект с полями regions и treeStateValue', function() {
            expect(u.getRegionsTreeInfo(tree, {}, campaignData).regions).to.exist;
            expect(u.getRegionsTreeInfo(tree, {}, campaignData).treeStateValue).to.exist;
        });

        describe('Поле regions', function() {
            it('Эквивалентно исходному дереву, если дополнительная информация не была указана', function() {
                expect(u.getRegionsTreeInfo(tree, undefined, campaignData).regions).to.deep.equal(tree);
            });

            it('Добавляет информацию о ставках, если она присутсвует для соответсвующих регионов в дополнительной информации', function() {
                expect(u.getRegionsTreeInfo(tree, { 3: { multiplier_pct: '100' } }, campaignData).regions[0].inner[0].bid).to.exist;
            });

            it('При указании значения ставки в дополнительной информации - в дереве оно преобразуется на константную величину смещения', function() {
                expect(u.getRegionsTreeInfo(tree, { 3: { multiplier_pct: '100' } }, campaignData).regions[0].inner[0].bid).to.equal(0);
            });

            it('Cвойство defaultBid региона равно значению ставки его ближайшего родителя или 0, если родителей нет', function() {
                var extendedGeo = {
                        1: { multiplier_pct: '200' },
                        225: { multiplier_pct: '300' }
                    },
                    resultTree = u.getRegionsTreeInfo(tree, extendedGeo , campaignData),
                    expects = {
                        225: 0,
                        3: 200,
                        1: 200,
                        10716: 100,
                        10717: 100,
                        21735: 100,
                        98582: 100,
                        2141: 100,
                        985821: 100,
                        217351: 100,
                        214: 100,
                        10650: 200,
                        52: 200,
                        241: 0,
                        10: 0,
                        20: 0,
                        11: 0,
                        111: 0,
                        1111: 0
                    },
                    check = function(region) {
                        expect(region.defaultBid).to.equal(expects[region.id]);

                        region.inner && region.inner.forEach(check);
                    };

                resultTree.regions.forEach(check);
            });

            describe('Информация о различиях в группах', function() {
                var resultTree,
                    getResultTree = function() {
                        var extendedGeo = {
                            225: {
                                partly: { adgroup_ids: ['1'] }
                            },
                            241: {
                                negative: {
                                    partly: { adgroup_ids: ['1'] }
                                }
                            }
                        };

                        return u.getRegionsTreeInfo(tree, extendedGeo, campaignData).regions;
                    },
                    getContrastValueGroupsIds = function(contrastValueGroups) {
                        return contrastValueGroups && contrastValueGroups.map(function(item) {

                            return item.id;
                        });
                    };

                it('Корректно добавяет свойствo contrastValueGroups к региону, если в дополнительной информации для региона указана информация о различиях в группах', function() {
                    resultTree = getResultTree();

                    expect(getContrastValueGroupsIds(resultTree[0].contrastValueGroups)).to.deep.equal(['2']);
                    expect(getContrastValueGroupsIds(resultTree[1].contrastValueGroups)).to.deep.equal(['1']);
                });

                it('Различия в группах наследуются дочерними регионами', function() {
                    var checkContrasts = function(region) {
                            expect(getContrastValueGroupsIds(region.contrastValueGroups)).to.deep.equal(['2']);

                            region.inner && region.inner.forEach(checkContrasts);
                        };

                    resultTree = getResultTree();

                    resultTree[0].inner.forEach(checkContrasts);
                });

                it('Если для дочернего региона указана своя информация о различиях, то она корректно применяется', function() {
                    var extendedGeo = {
                            225: { partly: { adgroup_ids: ['1'] } },
                            52: { all: 1 },
                            10650: { negative: { all: 1 } },
                            10717: { partly: { adgroup_ids: ['1'] } },
                            10716: { partly: { adgroup_ids: ['2'] } },
                            21735: { negative: { partly: { adgroup_ids: ['1'] } } },
                            98582: { negative: { partly: { adgroup_ids: ['2'] } } },
                            214: { partly: { adgroup_ids: ['2'] }, negative: { partly: { adgroup_ids: ['1'] } } }
                        },
                        expectedContrastValueGroupsValuesIds = {
                            225: ['2'],
                            3: ['2'],
                            1: ['2'],
                            52: undefined,
                            10650: undefined,
                            10716: undefined,
                            10717: ['2'],
                            21735: undefined,
                            98582: ['2'],
                            214: ['1'],
                            2141: ['1'],
                            985821: ['2'],
                            217351: undefined
                        },
                        checkContrasts = function(region) {
                            expect(getContrastValueGroupsIds(region.contrastValueGroups))
                                .to.deep.equal(expectedContrastValueGroupsValuesIds[region.id]);

                            region.inner && region.inner.forEach(checkContrasts);
                        };

                    resultTree = u.getRegionsTreeInfo(tree, extendedGeo, campaignData).regions;

                    resultTree[0].inner.forEach(checkContrasts);
                });
            });
        });

        it('Поле treeStateValue корректно вычисляется', function() {
            var extendedGeo = {
                    10: { all: 1 },
                    20: { partly: ['2'] },
                    225: { partly: { adgroup_ids: ['1'] } },
                    52: { all: 1 },
                    10650: { negative: { all: 1 } },
                    10717: { partly: { adgroup_ids: ['1'] } },
                    10716: { partly: { adgroup_ids: ['2'] } },
                    21735: { negative: { partly: { adgroup_ids: ['1'] } } },
                    98582: { negative: { partly: { adgroup_ids: ['2'] } } },
                    214: { partly: { adgroup_ids: ['2'] }, negative: { partly: { adgroup_ids: ['1'] } } },
                    11: { all: 1 },
                    1111: { partly: ['2'] }
                },
                treeStateValue;

            treeStateValue = u.getRegionsTreeInfo(tree, extendedGeo, campaignData).treeStateValue;

            expect(treeStateValue).to.equal('225,-21735,-10650,10,11');
        });

        it([
            'Поле treeStateValue корректно вычисляет последовательность,',
            'исключенные регионы должны идти сразу после родительского региона'
        ].join(', '), function() {
            var extendedGeo = {
                    1: {
                        negative: {
                            all: 1
                        }
                    },
                    52: {
                        negative: {
                            all: 1
                        }
                    },
                    214: {
                        all: 1
                    },
                    225: {
                        all: 1
                    },
                    2141: {
                        negative: {
                            all: 1
                        }
                    },
                    21735: {
                        all: 1
                    },
                    985821: {
                        all: 1
                    }
                },
                treeStateValue;

            treeStateValue = u.getRegionsTreeInfo(tree, extendedGeo, campaignData).treeStateValue;

            expect(treeStateValue).to.equal('225,-1,-52,21735,985821,214,-2141');
        });

        // @belyanskii: Тест создан по мотивам DIRECT-66790
        it('Проверка на корректный negative, когда выбран регион и дочерний регион его дочернего региона', function() {
            var extendedGeo = {
                    3: { all: 1 }, // Центр
                    1: { negative: { all: 1 } }, // Москва и область
                    10716: { all: 1 } // Балашиха
                },
                treeStateValue;

            treeStateValue = u.getRegionsTreeInfo(tree, extendedGeo, campaignData).treeStateValue;

            expect(treeStateValue).to.equal('3,-1,10716');
        });
    });

    describe('.getBidsFromExtendedData', function() {
        it('Корректно возвращает данные о ставках', function() {
            var result = u.getBidsFromExtendedData({
                255: { multiplier_pct: '0' },
                3: { multiplier_pct: '100' }
            });

            expect(result).to.deep.equal({
                255: '0',
                3: '100'
            });
        });
    });
});
