/**
 * @typedef {Object} TreeInfoResult
 * @property {String}
 * @property {Array} regions дерево регионов с дополнительной информацией о ставках и различиях в группах
 * @property {String} treeStateValue строка, на основе которой проставляются галочки выбора регионов
 */

(function() {
    var regionsById;

    /**
     * Разворачивает дерево регионов в плоский хеш где ключами являются id регионов
     *
     * @typedef {Object} RegionsHash
     * @property {Object.<Number>} RegionsHash[key] регион
     * @property {String} RegionsHash[key].name имя региона
     * @property {Array<Number>} RegionsHash[key].inner id вложенных регионов
     *
     * @typedef {Object} Region
     * @property {Number} Region.id идентификатор региона
     * @property {String} Region.name имя региона
     * @property {Array<Region>} Region.inner массив объектов вложенных регионов
     *
     * @param {Object|RegionsHash} result
     * @param {Object|Region[]} region
     * @returns {RegionsHash}
     */
    function flattenRegions(result, region) {
        if (u._.isArray(region)) {
            if (!result[0]) {
                result[0] = {
                    name: iget2('i-utils', 'vse', 'Все'),
                    inner: region.map(function(item) {
                        return item.id;
                    })
                }
            }

            return region.reduce(flattenRegions, result);
        }

        result[region.id] = {
            name: region.name,
            inner: region.inner && region.inner.map(function(innerRegion) {
                return innerRegion.id;
            })
        };

        region.inner && region.inner.reduce(flattenRegions, result);

        return result;
    }

    /**
     * Формирует строку из имен регионов с учетом исключений
     *
     * @typedef {Object} GeoName
     * @property {String} GeoName.name имя региона
     * @property {Array<String>} GeoName.excepts массив имен регионов которые отобразятся в секции "(кроме: )"
     *
     * @param {Array<GeoName>} geoNames имена регионов
     * @param {String} separator разделитель по умолчанию ";"
     * @returns {String}
     */
    function formatGeoNames(geoNames, separator) {
        separator = separator || ', ';

        return geoNames.map(function(geoName) {
            if (geoName.excepts.length) {
                return iget2('i-utils', 's-krome-s', '{foo} (кроме: {bar})', {
                    foo: geoName.name,
                    bar: geoName.excepts.join(separator)
                });
            }

            return geoName.name;
        })
            .join(separator);
    }

    u.register({
        REGIONS: {
            MOSCOW_AND_AREA: '1',
            ST_PETERBURG_AND_AREA: '10174',
            UKRAINE: '187',
            RUSSIA: '225',
            CIS: '166',
            GEORGIA: '169',
            CRIMEA: '977',
            MCB_DISABLED_REGIONS: ['1', '10174']
        },

        /**
         * Формирует строку из имен регионов, перечисленных через переданный сепаратор, по переданным id
         *
         * @param {String} geoIds id регионов через запятую, если id отрицательный регион попадет в секцию "(кроме: )"
         * @param {String} separator
         * @returns {String}
         */
        getGeoNames: function(geoIds, separator) {
            regionsById = regionsById ||
                flattenRegions({}, u._REGIONS_TREE[this.consts('lang') || 'ru']);

            var geoNames = [],
                regs = (geoIds + '' || '0').split(',').reduce(function(result, item) {
                    item = +item.trim();

                    !!regionsById[Math.abs(item)] && result.push(item);

                    return result;
                }, []);

            // если регионы не заданы или первым задан 0, формируем названия из псевдо региона "Все" см. flattenRegions
            if (!regs[0]) {
                regs = regionsById[0].inner.slice(0);
            } else if (regs[0] < 0) { //если первый елемент отрицательный формируем строку "Все (кроме: ... )"
                regs.unshift(0);
            }

            geoNames.push({
                name: regionsById[regs.shift()].name,
                excepts: []
            });

            regs.reduce(function(result, region) {
                if (region >= 0) {
                    result.push({
                        name: regionsById[region].name,
                        excepts: []
                    });
                } else {
                    result[geoNames.length - 1].excepts.push(regionsById[Math.abs(region)].name);
                }

                return result;
            }, geoNames);

            return formatGeoNames(geoNames, separator);
        },

        /**
         * Возвращает дерево регионов по ключу языка
         * @param {String} key язык или "язык для домена", например ru_for_ru
         * @returns {Object} дерево регионов
         */
        getRegionsTree: function(key) {
            return u._REGIONS_TREE[key];
        },

        /**
         * Добовляет дополнительную информацию о ставках и различиях выбора регионов в группах кампании в дерево регионов
         * @param {Object} originalRegionsTree исходное дерево регионов
         * @param {Object} extendedGeoData дополнительные данные
         * @param {Object} campaignData данные кампании
         * @param {Object} campaignData.groupNames хэш с названиями групп кампании
         * @param {String} campaignData.mediaType тип кампании
         * @param {String} campaignData.cid идентификатор кампании
         * @returns {TreeInfoResult}
         */
        getRegionsTreeInfo: function(originalRegionsTree, extendedGeoData, campaignData) {
            var regionsTree = u._.cloneDeep(originalRegionsTree),
                allGroupsIds = Object.keys(campaignData.groupNames),
                updateRegionData = function(region, parentRegion, parentContrastValueGroups) {
                    var allContrastValueGroups,
                        extendedInfo = extendedGeoData[region.id] || {},
                        contrastValueGroups = [],
                        hasRealConflicts,
                        regionHasOwnConflicts = extendedInfo.partly ||
                            extendedInfo.negative && extendedInfo.negative.partly;

                    parentRegion = parentRegion || {};

                    if (!extendedInfo.all && !(extendedInfo.negative || {}).all) {
                        contrastValueGroups = (parentContrastValueGroups || []).map(function(group) {
                            return group.id;
                        });
                    }

                    if (parentRegion.bid != undefined) {
                        region.defaultBid = parentRegion.bid;
                    } else if (parentRegion.defaultBid != undefined) {
                        region.defaultBid = parentRegion.defaultBid;
                    } else {
                        region.defaultBid = 0;
                    }

                    // значение ставки на сервере хранится со смещением
                    extendedInfo.multiplier_pct && (region.bid = extendedInfo.multiplier_pct - u.consts('bidOffset'));

                    // наследуем от родителя различия в группах
                    if (regionHasOwnConflicts) {
                        // если пришла информация о различиях в группе у минус региона,
                        // то прибавляем ее к списку выключенных групп родителя
                        if (extendedInfo.negative && extendedInfo.negative.partly) {
                            contrastValueGroups = u._.union(
                                contrastValueGroups,
                                extendedInfo.negative.partly.adgroup_ids
                            );
                        }

                        // если пришла информация о различиях в группе у плюс региона,
                        // то нужно исключить соответсвующие группы из списка выключенных
                        if (extendedInfo.partly) {
                            if (!contrastValueGroups.length) {
                                contrastValueGroups = allGroupsIds;
                            }

                            contrastValueGroups = u._.difference(
                                contrastValueGroups,
                                extendedInfo.partly.adgroup_ids
                            );
                        }
                    }

                    hasRealConflicts = contrastValueGroups.length !== 0 &&
                        contrastValueGroups.length !== allGroupsIds.length;

                    if (hasRealConflicts) {
                        // TODO DIRECT-67277: b-regions-tree: оптимизировать ссылки на группы
                        region.extraParams = {
                            contrastGroupsCount: contrastValueGroups.length,
                            campaignData: {
                                cid: campaignData.cid,
                                mediaType: campaignData.mediaType
                            }
                        };

                        allContrastValueGroups = contrastValueGroups.map(function(groupId) {
                            return {
                                id: groupId,
                                name: campaignData.groupNames[groupId]
                            }
                        });

                        region.contrastValueGroups = allContrastValueGroups
                            .slice(0, u.consts('maxContrastGroupsInRegion'))
                    }

                    // по умолчанию значение чекнутости наследуется от родителя
                    region.isChecked = parentRegion.isChecked;

                    // если есть различия в группах, либо явно указано, что регион выбран у всех групп,
                    // то галку надо ставить
                    if (
                        // группа явно отмечена у всех
                        extendedInfo.all ||
                        // в группе вычислены конфликты
                        hasRealConflicts ||
                        // конфликтов по результатам вычисления нет, но пришла информация, что это плюс регион
                        !hasRealConflicts && extendedInfo.partly
                    ) {
                        region.isChecked = true;
                    } else if (extendedInfo.negative && extendedInfo.negative.all ||
                        !hasRealConflicts && extendedInfo.negative && extendedInfo.negative.partly) {
                        // если это "минус" регион, он не может быть "чекнут"
                        region.isChecked = false;
                    }

                    region.inner && region.inner.forEach(function(innerRegion) {
                        updateRegionData(innerRegion, region, allContrastValueGroups);
                    });
                };

            extendedGeoData && regionsTree.forEach(function(region) {
                updateRegionData(region);
            });

            return {
                regions: regionsTree,
                treeStateValue: extendedGeoData ? this.getGeoValues(regionsTree).join(',') : ''
            };
        },

        /**
         * Строит строку с id-шниками регионов
         * @param {Object} regionsTree - дерево регионов, в котором уже есть информация о выбранных регионах
         * @returns {String}
         */
        getGeoValues: function(regionsTree) {
            // DIRECT-66790: Не сохранеются вложенные регионы Москвы - Зеленоград, Троицк, Щербинка, но не Москва.
            // правильный обход дерева регионов, который формирует строку с верной последовательность для b-regions-tree
            // TODO DIRECT-67228: b-regions-tree:
            // TODO оптимизировать обход дерева и научится строить дерево по id-шникам регионов в любом порядке
            var state = [], // массив из веток где обрывается наследие
                geoValues,
                /**
                 * DFS по дереву регионов
                 * ищет регионы которые не наследуют поведение родителя:
                 *  текущий регион чекнут а родитель нет - создаст новый элемент в state
                 *  текущий регион не чекнут а родитель чекнут - попадет excluded региона от которого идет наследование
                 * @param {Region} region - информация по текущему региону
                 * @param {Region} importantParent - родитель от которого идет наследование
                 */
                trip = function(region, importantParent) {
                    if (region.isChecked && (!importantParent || !importantParent.isChecked)) {
                        // отводим отдельную ветку для текущего региона
                        state.push({
                            regionId: region.id,
                            excluded: [] // исключенные регионы
                        });

                        // все последующие регионы наследую поведение текущего(если их не трогали)
                        importantParent = region;
                    } else if (!region.isChecked && (importantParent && importantParent.isChecked)) {
                        // находим ветку к которой относится текущий регион
                        var stateItem = u._.find(state, { regionId: importantParent.id });

                        stateItem.excluded.push(-region.id);
                        // разрываем связь с родителем
                        importantParent = null;
                    }

                    region.inner && region.inner.forEach(function(innerRegion) {
                        trip(innerRegion, importantParent);
                    });
                };

            regionsTree.forEach(function(region) {
                trip(region);
            });

            geoValues = state.reduce(function(result, stateItem) {
                return result.concat(stateItem.regionId, stateItem.excluded);
            }, []);

            return geoValues;
        },

        /**
         * TODO DIRECT-67228: b-regions-tree:
         * TODO оптимизировать обход дерева и научится строить дерево по id-шникам регионов в любом порядке
         * эмитируем хеш(по аналогии с campaign.extended_geo)
         * @param {String} geoString
         * @returns {Object}
         */
        geoStringToFakeObject: function(geoString) {
            return geoString.split(/[\s|,]+/).reduce(function(result, regionId) {
                result[Math.abs(regionId)] = regionId > 0 ?
                    { all: 1 } :
                    { negative: { all: 1 } };

                return result;
            }, {});
        },

        /**
         * Возвращает информацию о ставках для регионов на основе данных о гео от сервера
         * @param {Object} extendedGeoData гео данные
         * @returns {Object}
         */
        getBidsFromExtendedData: function(extendedGeoData) {

            return Object.keys(extendedGeoData || {}).reduce(function(result, regionId) {
                var region = extendedGeoData[regionId];

                if (region.multiplier_pct != undefined) {
                    result[regionId] = region.multiplier_pct;
                }

                return result;
            }, {})
        }
    });
}());
