(function(u) {

    /**
     * Расчет значение 30-го перцентиля на основании ставок для фраз
     * Описание алгоритма расчета
     * https://wiki.yandex-team.ru/Direkt/TechnicalDesign/relevance-match/#izmenenijavjadre
     * @param {Array} pricesList - ставки на фразы
     * @param {String} currency - валюта
     * @returns {Number}
     */
    function calcPercentile(pricesList, currency) {
        var percentile,
            percentileIndex,
            prevIndex,
            prevWeighted,
            nextIndex,
            nextWeighted;

        // сортируем по возрастанию
        pricesList = u._.sortBy(pricesList, function(price) {
            return price;
        });
        // берем 30% от максимального индекса
        percentileIndex = (pricesList.length - 1) * 0.3;

        // если индекс целый, то это и есть перцентиль
        if (percentileIndex % 1 === 0) {
            percentile = pricesList[percentileIndex];
        } else {
            // иначе берем сумму взвешенных значений "сверху" и "снизу"
            // где вес - расстояние от индекса до индекса перцентиля второго значения
            prevIndex = Math.floor(percentileIndex);
            nextIndex = Math.ceil(percentileIndex);
            prevWeighted = (nextIndex - percentileIndex) * pricesList[prevIndex];
            nextWeighted = (percentileIndex - prevIndex) * pricesList[nextIndex];
            percentile = prevWeighted + nextWeighted;
        }
        // округляем до шага торгов
        return u.currencies.roundPriceToStep(percentile, currency, 'up' );
    }

    /**
     * Расчет среднего значения ставок по фразам, используется для автотаргетинга в сетях
     * https://a.yandex-team.ru/arc/trunk/arcadia/direct/core/src/main/java/ru/yandex/direct/core/entity/relevancematch/service/RelevanceMatchAutoPricesCalculator.java?rev=5664251#L249
     * @param {Array} pricesAndClicksList - ставки и клики по фразам
     * @param {String} currency - валюта
     * @returns {Number}
     */
    function calcAverage(pricesAndClicksList, currency) {
        var sumPriceAndClicks = pricesAndClicksList.reduce(function(acc, priceAndClicks) {
            var sumPrice = acc[0],
                sumClicks = acc[1],
                price = priceAndClicks.price,
                clicks = priceAndClicks.clicks + 1; // прибавляем 1, чтобы не оказаться с нулем в знаменателе

            return [sumPrice + price * (clicks), sumClicks + clicks];
        }, [0, 0]);

        // округляем до шага торгов
        return u.currencies.roundPriceToStep(sumPriceAndClicks[0] / sumPriceAndClicks[1], currency, 'up' );
    }

    u.register({
        'relevance-match': u._.extend({}, {
            calcPercentile: calcPercentile,
            calcAverage: calcAverage
        })
    })
})(u);
