(function() {

    u.register({

        /**
         * @typedef {Object} RetCondPerl
         *
         * @property {String} condition_name
         * @property {String} condition_desc
         * @property {String} retargeting_conditions_type
         * @property {Array} condition
         * @property {Number} is_accessible
         * @property {Number} ret_cond_id
         */

        retargeting: {

            ruleTypeNameMapping: {
                or: iget2('i-utils', 'retargeting-or', 'Выполнено хотя бы одно'),
                not: iget2('i-utils', 'retargeting-not', 'Не выполнено ни одного'),
                all: iget2('i-utils', 'retargeting-and', 'Выполнены все')
            },

            /**
             * Преобразовывает серверные данные условий ретаргетинга
             * @param {RetCondPerl} data серверные данные условий ретаргетинга
             * @returns {Array}
             */
            transformServerData: function(data) {
                var result = [];

                Object.keys(data || {}).map(function(conditionId) {
                    var item = data[conditionId],
                        condition = u._.pick(item, [
                            'ret_cond_id',
                            'condition_name',
                            'condition_desc',
                            'is_accessible',
                            'campaigns'
                        ]);

                    condition.is_negative = item.properties === 'negative';

                    condition.clicks = item.ctx_clicks || 0;
                    condition.shows = item.ctx_shows || 0;
                    condition.ctr = item.ctx_ctr || '0.00';

                    condition.campaigns = (condition.campaigns || []).map(function(campaign) {
                        return {
                            cid: campaign.cid,
                            name: campaign.name,
                            url: u.getUrl('showCamp', {
                                cid: campaign.cid,
                                ulogin: u.consts('ulogin')
                            })
                        }
                    });

                    condition.groups = item.condition.map(function(group) {

                        return {
                            type: group.type,
                            goals: (group.goals || []).map(function(goal) {
                                return {
                                    id: goal.goal_id,
                                    time: goal.time,
                                    //todo - пока не приходит goal_type
                                    //тикет на серверсайд - DIRECT-53726
                                    goalType: goal.goal_type || 'goal'
                                };
                            })
                        };
                    });

                    result.push(condition);
                });

                return result;
            },

            /**
             * Фильтрует условия ретаргетинга по типу
             * @param {Object} conditions - условия ретаргетинга
             * @param {String} [type] - тип условия
             */
            filterConditionsByType: function(conditions, type) {
                var filteredConditions = {};

                Object.keys(conditions).forEach(function(id) {
                    this.isConditionType(conditions[id], type) &&
                        (filteredConditions[id] = u._.extend({}, conditions[id]));
                }, this);

                return filteredConditions;
            },

            /**
             * Фильтрует условия ретаргетинга группы по типу
             * @param {Array<Object>} groupConditions - условия ретаргетинга группы
             * @param {Object} conditions - условия ретаргетинга
             * @param {String} [type] - тип условия
             */
            filterGroupConditionsByType: function(groupConditions, conditions, type) {
                var self = this;

                return groupConditions.reduce(function(result, groupCondition) {
                    var condition = conditions[groupCondition.ret_cond_id];

                    self.isConditionType(condition, type) && (
                        result.push(groupCondition)
                    );

                    return result;
                }, []);
            },

            /**
             * Проверяет, является ли условие конкретным типом
             * @param {Object} condition
             * @param {String} [type]
             * @return {*}
             */
            isConditionType: function(condition, type) {
                if (type === 'metrika_goals') {
                    return !!~['metrika_goals', null].indexOf(condition.retargeting_conditions_type);
                } if (type) {
                    return condition.retargeting_conditions_type === type;
                } else {
                    return true;
                }
            },

            /**
             * Проверяет, является ли условие конкретным типом
             * @param {Object} conditions
             * @param {String} [type]
             * @return {*}
             */
            hasRetargetingConditions: function(conditions, type) {
                return !u._.isEmpty(this.filterConditionsByType(conditions, type));
            },

            /**
             * Возвращает название для не найденной цели
             * @param {Object} goal
             * @returns {*|String|BEMJSON}
             */
            getLostGoalName: function(goal) {
                switch (goal.type) {
                    case 'audience':
                    case 'segment':
                        return iget2('b-retargetings-data', 'retargeting-lost-segment-with-id', 'Сегмент не найден (id:{id})', {
                            id: goal.id
                        });
                    default:
                        return iget2('b-retargetings-data', 'retargeting-lost-goal-with-id', 'Цель не найдена (id:{id})', {
                            id: goal.id
                        });
                }
            },

            getGoalName: function(goal) {

                if (!goal.name && !goal.allow_to_use) {
                    return this.getLostGoalName(goal);
                }

                var name = [goal.domain, goal.name]
                    .filter(function(val) { return val })
                    .join(' :: ');

                if (goal.type === 'lal_segment') {
                    name += ' — ' + iget2('b-retargetings-data', 'lal', 'похожие')
                }

                return name;
            },

            getRetargetingConditionsMixes: function() {
                return {
                    row: function(row) {
                        return [
                            { block: 'b-style-table', elem: 'row' },
                            {
                                block: 'p-retargeting-conditions',
                                elem: 'condition-row',
                                elemMods: { id: row.ret_cond_id }
                            }
                        ];
                    },
                    headerRow: [{
                        block: 'b-style-table',
                        elem: 'head'
                    }],
                    cell: function(key) {
                        return [
                            { block: 'b-style-table', elem: 'cell' },
                            {
                                block: 'p-retargeting-conditions',
                                elem: 'condition-cell',
                                elemMods: { key: key || 'num' }
                            }
                        ];
                    },
                    headerCell: function(key) {
                        return [
                            {
                                block: 'b-style-table',
                                elem: 'cell'
                            },
                            {
                                block: 'p-retargeting-conditions',
                                elem: 'condition-column',
                                elemMods: { key: key || 'num' }
                            }
                        ];
                    }
                };
            },

            getRetargetingConditionsColumns: function() {
                var CAMPS_MAX_COUNT = 3

                return [
                    {
                        key: 'checkbox',
                        transformData: function(row) {
                            return {
                                block: 'checkbox',
                                mods: { disabled: row.campaigns && row.campaigns.length ? 'yes' : '' },
                                mix: [{
                                    block: 'p-retargeting-conditions',
                                    elem: 'condition-checkbox',
                                    elemMods: { id: row.ret_cond_id }
                                }]
                            };
                        },
                        title: {
                            block: 'checkbox',
                            mix: [{
                                block: 'p-retargeting-conditions',
                                elem: 'select-all'
                            }]
                        }
                    },
                    {
                        key: 'name',
                        transformData: function(row, num) { return num; }
                    },
                    {
                        key: 'name',
                        title: iget2('p-retargeting-conditions', 'nazvanie', 'Название'),
                        transformData: function(row) {
                            return [
                                {
                                    block: 'b-group-retargeting-condition',
                                    mods: { 'without-remove': row.campaigns && row.campaigns.length ? 'yes' : '' },
                                    condition: row
                                },
                                row.is_accessible ?
                                    '' :
                                    {
                                        block: 'p-retargeting-conditions',
                                        elem: 'warning',
                                        conditionId: row.ret_cond_id
                                    }
                            ];
                        }
                    },
                    {
                        key: 'id',
                        title: iget2('p-retargeting-conditions', 'id', 'ID'),
                        transformData: function(row) { return row.ret_cond_id }
                    },
                    {
                        key: 'domains',
                        title: iget2('p-retargeting-conditions', 'sayt', 'Сайт')
                    },
                    {
                        key: 'campaigns',
                        title: iget2('p-retargeting-conditions', 'ispolzuetsya-v-kampaniyah', 'Используется в кампаниях'),
                        transformData: function(row) {
                            if (!row.campaigns || !row.campaigns.length) {
                                return {
                                    block: 'p-retargeting-conditions',
                                    elem: 'dash'
                                };
                            }

                            var campaigns = row.campaigns.map(function(campaign) {
                                    return {
                                        block: 'p-retargeting-conditions',
                                        elem: 'campaign',
                                        campaign: campaign,
                                        ulogin: u.consts('ulogin')
                                    };
                                }),
                                res = campaigns.slice(0, CAMPS_MAX_COUNT),
                                dropdownCampaigns = campaigns.slice(CAMPS_MAX_COUNT);

                            dropdownCampaigns.length && res.push({
                                block: 'dropdown',
                                content: [
                                    {
                                        block: 'link',
                                        mods: { pseudo: 'yes' },
                                        mix: [{
                                            block: 'dropdown',
                                            elem: 'switcher'
                                        }],
                                        content: iget2('p-retargeting-conditions', 'eshchyo', 'ещё') + '&nbsp;' +
                                            u.pluralize([iget2('p-retargeting-conditions', 'kampaniya', 'кампания'), iget2('p-retargeting-conditions', 'kampanii', 'кампании'), iget2('p-retargeting-conditions', 'kampaniy', 'кампаний')], dropdownCampaigns.length)
                                    },
                                    {
                                        elem: 'popup',
                                        content: dropdownCampaigns
                                    }
                                ]
                            });

                            return res;
                        }
                    },
                    {
                        key: 'comment',
                        title: iget2('p-retargeting-conditions', 'primechanie', 'Примечание'),
                        transformData: function(row) {
                            return row.condition_desc_escape || {
                                block: 'p-retargeting-conditions',
                                elem: 'dash'
                            };
                        }
                    },
                    {
                        key: 'shows',
                        title: iget2('p-retargeting-conditions', 'pokazy', 'Показы')
                    },
                    {
                        key: 'clicks',
                        title: iget2('p-retargeting-conditions', 'kliki', 'Клики')
                    },
                    {
                        key: 'ctr',
                        title: iget2('p-retargeting-conditions', 'ctr', 'CTR')
                    }
                ];
            }
        }
    });
})();
