describe('i-utils__strategy', function() {

    var currency = 'RUB',
        goal = {
            goal_id: '6002154',
            goal_name: 'Просмотреть более 7 страниц',
            goal_status: 'Active'
        },
        sandbox,
        constsStub;

    before(function() {
        sandbox = sinon.sandbox.create();
        constsStub = u.stubCurrencies2(sandbox);
    });

    after(function() {
        sandbox.restore();
    });

    describe('Тесты на формирование описания стратегии', function() {
        var strategyData;

        describe('Для раздельного управления', function() {
            describe('Показы на поиске разрешены', function() {
                it('Для стратегии DEFAULT', function() {
                    strategyData = {
                        options: {
                            net: { name: 'maximum_coverage' },
                            search: { name: 'default' }
                        },
                        name: 'different_places',
                        is_net_stopped: 0
                    };

                    expect(u.strategy.getHint({ strategyData: strategyData, currency: 'RUB' })).to.deep.equal(
                        ['Раздельно управлять ставками на поиске и в сетях']
                    );
                });
            });

            describe('Показы на поиске запрещены = стратегии "только в сетях"', function() {
                it('Для ручной стратегии', function() {
                    strategyData = {
                        options: {
                            search: { name: 'stop' },
                            net: { name: 'maximum_coverage' }
                        },
                        name: 'different_places',
                        is_net_stopped: 0
                    };

                    expect(JSON.stringify(u.strategy.getHint({ strategyData: strategyData, currency: currency })))
                        .to.equal('["На поиске: Показы запрещены","В сетях: Ручное управление ставками с оптимизацией в сетях"]');
                });

                it('Для автоматической стратегии', function() {
                    strategyData = {
                        options: {
                            search: { name: 'stop' },
                            net: {
                                name: 'autobudget_avg_click',
                                avg_bid: 12,
                                sum: ''
                            }
                        },
                        name: 'different_places',
                        is_net_stopped: 0
                    };

                    expect(JSON.stringify(u.strategy.getHint({ strategyData: strategyData, currency: currency })))
                        .to.equal('["На поиске: Показы запрещены","В сетях: Удерживать цену клика 12.00 руб. в среднем за неделю"]');//eslint-disable-line
                });
            });
        });

        describe('Для нераздельного управления', function() {
            describe('На всех площадках', function() {
                it('Для стратегии DEFAULT неприменимо', function() {
                    strategyData = {
                        name: 'default',
                        options: {
                            name: 'default'
                        },
                        is_net_stopped: 0
                    };

                    expect(u.strategy.getHint({ strategyData: strategyData, currency: currency })[0])
                        .to.equal(undefined);
                });

                it('Для автоматической стратегии', function() {
                    strategyData = {
                        name: 'autobudget_avg_click',
                        options: {
                            avg_bid: 12,
                            sum: '',
                            name: 'autobudget_avg_click'
                        },
                        is_net_stopped: 0
                    };

                    expect(u.strategy.getHint({ strategyData: strategyData, currency: currency })).to.deep.equal([
                        'Удерживать цену клика 12.00 руб. в среднем за неделю'
                    ]);
                });
            });

            describe('Только на поиске', function() {
                it('Для стратегии DEFAULT', function() {
                    strategyData = {
                        name: 'default',
                        options: { name: 'default' },
                        net: { name: 'stop' },
                        is_net_stopped: 1
                    };

                    expect(JSON.stringify(u.strategy.getHint({ strategyData: strategyData, currency: currency })))
                        .to.equal('["На поиске: Ручное управление ставками","В сетях: Показы запрещены"]');
                });

                it('Для автоматической стратегии', function() {
                    strategyData = {
                        name: 'autobudget_avg_click',
                        options: {
                            avg_bid: 12,
                            sum: '',
                            name: 'autobudget_avg_click'
                        },
                        is_net_stopped: 1
                    };

                    expect(JSON.stringify(u.strategy.getHint({ strategyData: strategyData, currency: currency })))
                        .to.equal('["На поиске: Удерживать цену клика 12.00 руб. в среднем за неделю","В сетях: Показы запрещены"]'); //eslint-disable-line
                });
            });
        });

        describe('Для Баннера на поиске — isOnlySearch', function() {
            it('Ручная сратегия', function() {
                strategyData = {
                    options: {
                        name: 'default'
                    },
                    name: 'search_maximum_coverage'
                };

                expect(u.strategy.getHint({ strategyData: strategyData, currency: 'RUB', isOnlySearch: true })).to.deep.equal(
                    ['Ручное управление ставками']
                );
            });

            it('Автоматическая стратегия', function() {
                strategyData = {
                    options: {
                        name: 'autobudget_avg_cpc_per_camp',
                        avg_bid: 300,
                        target: 'camp'
                    },
                    platform: 'search'
                };

                expect(u.strategy.getHint({ strategyData: strategyData, currency: 'RUB', isOnlySearch: true })).to.deep.equal(
                    ['Удерживать CPC на всю кампанию в среднем за неделю на уровне 300.00 руб.']
                );
            });
        });
    });

    describe('Тесты на хинты с опциями для автоматических стратегий', function() {
        // используем стратегию на всех площадках, нераздельное управление
        // тогда нас всегда интересует первый элемент getHint()
        var strategyData;

        describe('Хинты для стратегии AUTOBUDGET_AVG_CLICK', function() {
            it('Если указан недельный бюджет', function() {
                strategyData = {
                    name: 'autobudget_avg_click',
                    options: {
                        avg_bid: 12,
                        sum: '',
                        name: 'autobudget_avg_click'
                    },
                    is_net_stopped: 0
                };

                expect(u.strategy.getHint({ strategyData: strategyData, currency: currency })[0])
                    .to.equal('Удерживать цену клика 12.00 руб. в среднем за неделю');
            });

            it('Если не указан недельный бюджет', function() {
                strategyData = {
                    name: 'autobudget_avg_click',
                    options: {
                        avg_bid: 12,
                        sum: 25,
                        name: 'autobudget_avg_click'
                    },
                    is_net_stopped: 0
                };

                expect(u.strategy.getHint({ strategyData: strategyData, currency: currency })[0])
                    .to.equal('Удерживать цену клика 12.00 руб. в среднем за неделю при недельном бюджете не более 25.00 руб.'); //eslint-disable-line
            });
        });

        describe('Хинты для стратегии AUTOBUDGET_AVG_CPA', function() {
            var goalsVariants = [
                {
                    goalId: '6002154',
                    title: 'выбрана цель',
                    text: 'по цели «Просмотреть более 7 страниц»'
                },
                {
                    goalId: '0',
                    title: 'не выбрана цель',
                    text: 'по всем целям'
                }],
                weekBudgetVariants = [
                    {
                        sum: 0,
                        title: 'определен недельный бюджет',
                        text: 'тратить в недел'
                    },
                    {
                        sum: 310,
                        title: 'не определен недельный бюджет',
                        text: 'тратить в неделю не более 310.00 руб.'
                    }
                ],
                maxClickVariants = [
                    {
                        bid: 0,
                        title: 'не указана максимальная цена клика',
                        text: 'при максимальной цене клика'
                    },
                    {
                        bid: 2,
                        title: 'указана максимальная цена клика',
                        text: 'при максимальной цене клика не более 2.00 руб.'
                    }
                ];

            goalsVariants.forEach(function(goalVariant) {
                weekBudgetVariants.forEach(function(wbVariant) {
                    maxClickVariants.forEach(function(mcVariant) {
                        it('Если ' + [goalVariant.title, wbVariant.title, mcVariant.title]
                            .join(', ') + ' то хинт корректен', function() {
                            var hint;

                            strategyData = {
                                name: 'autobudget_avg_cpa',
                                options: {
                                    avg_cpa: 12,
                                    name: 'autobudget_avg_cpa'
                                },
                                is_net_stopped: 0
                            };

                            strategyData.options = u._.extend(strategyData.options, {
                                goal_id: goalVariant.goalId,
                                sum: wbVariant.sum,
                                bid: mcVariant.bid
                            });

                            hint = u.strategy.getHint({
                                strategyData: strategyData,
                                currency: currency,
                                metrika: { campaign_goals: [goal] }
                            })[0];

                            expect(hint).to.include('Удерживать цену конверсии 12.00 руб.');
                            // правильно указана цель
                            expect(hint).to.include(goalVariant.text);
                            // правильно (не) указан недельный бюджет
                            wbVariant.sum === 0 ?
                                expect(hint).not.to.include(wbVariant.text) :
                                expect(hint).to.include(wbVariant.text);
                            // правильно (не) указана максимальная цена клика
                            mcVariant.bid === 0 ?
                                expect(hint).not.to.include(mcVariant.text) :
                                expect(hint).to.include(mcVariant.text);
                        });
                    });
                });
            });

            it('Если в данных есть флаг metrika.cpa_deviation, то в хинте появляется сообщение про это', function() {
                strategyData = {
                    name: 'autobudget_avg_cpa',
                    options: {
                        avg_cpa: 12,
                        goal_id: '6002154',
                        name: 'autobudget_avg_cpa'
                    },
                    is_net_stopped: 0
                };

                expect(u.strategy.getHint({
                    strategyData: strategyData,
                    currency: currency,
                    metrika: { cpa_deviation: true, campaign_goals: [goal] }
                })[0][1].content[1].content)
                    .to.equal('Внимание! Стоимость конверсии значительно отличается от установленного значения');
            });

            it('Если в данных нет флага metrika.cpa_deviation, то в хинте не появляется сообщение про это', function() {
                var hint;

                strategyData = {
                    name: 'autobudget_avg_cpa',
                    options: {
                        avg_cpa: 12,
                        goal_id: '6002154',
                        name: 'autobudget_avg_cpa'
                    },
                    is_net_stopped: 0
                };
                hint = u.strategy.getHint({
                    strategyData: strategyData,
                    currency: currency,
                    metrika: { cpa_deviation: false, campaign_goals: [goal] }
                })[0];

                expect(typeof hint).to.equal('string');
            });

            describe('Если цель удалена', function() {
                var hint;

                before(function() {
                    strategyData = {
                        name: 'autobudget_avg_cpa',
                        options: {
                            avg_cpa: 12,
                            name: 'autobudget_avg_cpa',
                            goal_id: '6002154'
                        },
                        is_net_stopped: 0
                    };

                    goal['goal_status'] = 'Deleted';
                    hint = u.strategy.getHint({
                        strategyData: strategyData,
                        currency: currency,
                        metrika: { campaign_goals: [goal] }
                    })
                });

                after(function() {
                    goal['goal_status'] = 'Active';
                });

                it('Хинт возвращается массивом', function() {
                    expect(u._.isArray(hint[0])).to.equal(true);
                });

                it('В массиве есть объект с предупреждением', function() {
                    expect(u._.filter(hint[0], function(hintpart) {
                        return hintpart.block === 'b-help-link' ||
                            (u._.isArray(hintpart) && hintpart[1].block === 'b-help-link');
                    }).length).to.be.gt(0);
                });
            });

            describe('Если у цели нет имени ' +
                '(бывает, когда по цель, по которой настроили, не пришла в кампанию)', function() {
                var hint;

                before(function() {
                    strategyData = {
                        name: 'autobudget_avg_cpa',
                        options: {
                            avg_cpa: 12,
                            goal_id: '123456',
                            name: 'autobudget_avg_cpa'
                        },
                        is_net_stopped: 0
                    };

                    hint = u.strategy.getHint({
                        strategyData: strategyData,
                        currency: currency
                    })
                });

                it('Хинт возвращается массивом', function() {
                    expect(u._.isArray(hint[0])).to.equal(true);
                });

                it('В массиве хинт "цель не найдена"', function() {
                    expect(u._.filter(hint[0], function(hintpart) {
                        return u._.isArray(hintpart) && hintpart[1].content === 'цель не найдена'
                    }).length).to.be.gt(0);
                });
            });
        });

        describe('Хинты для стратегии AUTOBUDGET_AVG_CPI', function() {
            var weekBudgetVariants = [
                {
                    sum: 0,
                    title: 'определен недельный бюджет',
                    text: 'тратить в недел'
                },
                {
                    sum: 310,
                    title: 'не определен недельный бюджет',
                    text: 'тратить в неделю не более 310.00 руб.'
                }],
                maxClickVariants = [
                    {
                        bid: 0,
                        title: 'не указана максимальная цена клика',
                        text: 'максимальной цене клика'
                    },
                    {
                        bid: 2,
                        title: 'указана максимальная цена клика',
                        text: 'максимальной цене клика не более 2.00 руб.'
                    }
                ];

            weekBudgetVariants.forEach(function(wbVariant) {
                maxClickVariants.forEach(function(mcVariant) {
                    it('Если ' + [wbVariant.title, mcVariant.title].join(', ') + ' то хинт корректен', function() {
                        var hint;

                        strategyData = {
                            name: 'autobudget_avg_cpi',
                            options: {
                                avg_cpi: 12,
                                name: 'autobudget_avg_cpi'
                            },
                            is_net_stopped: 0
                        };

                        strategyData.options = u._.extend(strategyData.options, {
                            sum: wbVariant.sum,
                            bid: mcVariant.bid
                        });

                        hint = u.strategy.getHint({ strategyData: strategyData, currency: currency })[0];

                        expect(hint).to.include('Удерживать цену установки 12.00 руб. в среднем за неделю');
                        // правильно (не) указан недельный бюджет
                        wbVariant.sum === 0 ?
                            expect(hint).not.to.include(wbVariant.text) :
                            expect(hint).to.include(wbVariant.text);
                        // правильно (не) указана максимальная цена клика
                        mcVariant.bid === 0 ?
                            expect(hint).not.to.include(mcVariant.text) :
                            expect(hint).to.include(mcVariant.text);
                    });
                });
            });

            it('Если в данных есть флаг metrika.cpi_deviation, то в хинте появляется сообщение про это', function() {
                strategyData = {
                    name: 'autobudget_avg_cpi',
                    options: {
                        avg_cpi: 12,
                        name: 'autobudget_avg_cpi'
                    },
                    is_net_stopped: 0
                };

                expect(u.strategy.getHint({
                    strategyData: strategyData,
                    currency: currency,
                    metrika: { cpi_deviation: true }
                })[0][1].content[1].content)
                    .to.equal('Внимание! Стоимость установки значительно отличается от выбранного значения');
            });

            it('Если в данных нет флага metrika.cpi_deviation, то в хинте не появляется сообщение про это', function() {
                var hint;

                strategyData = {
                    name: 'autobudget_avg_cpi',
                    options: {
                        avg_cpi: 12,
                        name: 'autobudget_avg_cpi'
                    },
                    is_net_stopped: 0
                };
                hint = u.strategy.getHint({
                    strategyData: strategyData,
                    currency: currency,
                    metrika: { cpi_deviation: false }
                })[0];

                expect(typeof hint).to.equal('string');
            });
        });

        describe('Хинты для стратегии AUTOBUDGET_ROI', function() {
            var goalsVariants = [
                {
                    goalId: '6002154',
                    title: 'выбрана цель',
                    text: 'по цели «Просмотреть более 7 страниц»'
                },
                {
                    goalId: '0',
                    title: 'не выбрана цель',
                    text: 'по всем целям'
                }],
                weekBudgetVariants = [
                    {
                        sum: 0,
                        title: 'определен недельный бюджет',
                        text: 'при недельном бюджете не более '
                    },
                    {
                        sum: 310,
                        title: 'не определен недельный бюджет',
                        text: 'при недельном бюджете не более 310.00 руб.'
                    }
                ],
                maxClickVariants = [
                    {
                        bid: 0,
                        title: 'не указана максимальная цена клика',
                        text: 'максимальной ставке не более'
                    },
                    {
                        bid: 2,
                        title: 'указана максимальная цена клика',
                        text: 'максимальной ставке не более 2.00 руб.'
                    }
                ],
                profitabilityVariants = [
                    {
                        profitability: 7,
                        title: 'указана себестоимость ',
                        text: '7% доходов является себестоимостью'
                    },
                    {
                        profitability: 0,
                        title: 'не указана себестомость',
                        text: 'доходов является себестоимостью'
                    }
                ];

            goalsVariants.forEach(function(goalVariant) {
                weekBudgetVariants.forEach(function(wbVariant) {
                    maxClickVariants.forEach(function(mcVariant) {
                        profitabilityVariants.forEach(function(profitVariant) {
                            it('Если ' + [goalVariant.title, wbVariant.title, mcVariant.title, profitVariant.title]
                                .join(', ') +
                                ', то хинт корректен', function() {
                                var hint;

                                strategyData = {
                                    name: 'autobudget_roi',
                                    options: {
                                        roi_coef: 12,
                                        reserve_return: 24,
                                        name: 'autobudget_roi'
                                    },
                                    is_net_stopped: 0
                                };

                                strategyData.options = u._.extend(strategyData.options, {
                                    goal_id: goalVariant.goalId,
                                    sum: wbVariant.sum,
                                    bid: mcVariant.bid,
                                    profitability: profitVariant.profitability
                                });

                                hint = u.strategy.getHint({
                                    strategyData: strategyData,
                                    currency: currency,
                                    metrika: { campaign_goals: [goal] }
                                })[0];

                                expect(hint).to
                                    .include('Удерживать рентабельность инвестиций на уровне 12.00');
                                expect(hint).to.include('возвращать в рекламу 24% сэкономленного бюджета');
                                // правильно указана цель
                                expect(hint).to.include(goalVariant.text);
                                // правильно (не) указан недельный бюджет
                                wbVariant.sum === 0 ?
                                    expect(hint).not.to.include(wbVariant.text) :
                                    expect(hint).to.include(wbVariant.text);
                                // правильно (не) указана максимальная цена клика
                                mcVariant.bid === 0 ?
                                    expect(hint).not.to.include(mcVariant.text) :
                                    expect(hint).to.include(mcVariant.text);
                                // правильно (не) указана себестомость
                                profitVariant.profitability === 0 ?
                                    expect(hint).not.to.include(profitVariant.text) :
                                    expect(hint).to.include(profitVariant.text);
                            });
                        });
                    });
                });
            });

            describe('Если у цели нет имени ' +
                '(бывает, когда по цель, по которой настроили, не пришла в кампанию)', function() {
                var hint;

                before(function() {
                    strategyData = {
                        name: 'autobudget_roi',
                        options: {
                            goal_id: '123456',
                            roi_coef: 12,
                            reserve_return: 24,
                            name: 'autobudget_roi'
                        },
                        is_net_stopped: 0
                    };

                    hint = u.strategy.getHint({
                        strategyData: strategyData,
                        currency: currency,
                        metrika: { campaign_goals: [goal] }
                    })
                });

                it('Хинт возвращается массивом', function() {
                    expect(u._.isArray(hint[0])).to.equal(true);
                });

                it('В массиве хинт "цель не найдена"', function() {
                    expect(u._.filter(hint[0], function(hintpart) {
                        return u._.isArray(hintpart) && hintpart[1].content === 'цель не найдена'
                    }).length).to.be.gt(0);
                });
            });

            describe('Если цель удалена', function() {
                var hint;

                before(function() {
                    strategyData = {
                        name: 'autobudget_roi',
                        options: {
                            goal_id: '6002154',
                            roi_coef: 12,
                            reserve_return: 24,
                            name: 'autobudget_roi'
                        },
                        is_net_stopped: 0
                    };

                    goal['goal_status'] = 'Deleted';
                    hint = u.strategy.getHint({
                        strategyData: strategyData,
                        currency: currency,
                        metrika: { campaign_goals: [goal] }
                    })[0]
                });

                after(function() {
                    goal['goal_status'] = 'Active';
                });

                it('В хинте есть объект-предупреждение', function() {
                    expect(hint[3].length).to.equal(2);
                    expect(u._.isObject(hint[3][1])).to.equal(true);
                });

                it('В массиве есть объект с предупреждением', function() {
                    expect(u._.filter(hint[3], function(hintpart) {
                        return hintpart.block === 'b-help-link';
                    }).length).to.be.gt(0);
                });

                it('текст предупреждения - "Цель была удалена"', function() {
                    expect(u._.filter(hint[3], function(hintpart) {
                        return hintpart.block === 'b-help-link';
                    })[0].mix[0].js.hint).to.equal('Цель была удалена')
                });
            });
        });

        describe('Хинты для стратегии AUTOBUDGET', function() {
            var goalsVariants = [
                {
                    goalId: '',
                    title: 'не указана цель',
                    text: 'получать максимум кликов'
                },
                {
                    goalId: '6002154',
                    title: 'выбрана цель',
                    text: ['получать максимальную конверсию по цели', 'Просмотреть более 7 страниц']
                },
                {
                    goalId: '0',
                    title: 'по всем целям',
                    text: 'получать максимальную конверсию по всем целям'
                },
                {
                    goalId: '4',
                    title: 'указана мобильная цель',
                    text: 'получать максимальное количество установок'
                }],
                maxClickVariants = [
                    {
                        bid: 0,
                        title: 'не указана максимальная цена клика',
                        text: 'при максимальной ставке не более'
                    },
                    {
                        bid: 2,
                        title: 'указана максимальная цена клика',
                        text: 'при максимальной ставке не более'
                    }
                ];

            goalsVariants.forEach(function(goalVariant) {
                maxClickVariants.forEach(function(mcVariant) {
                    it('Если ' + [goalVariant.title, mcVariant.title].join(', ') +
                        ', то хинт корректен', function() {
                        var hint;

                        strategyData = {
                            name: 'autobudget',
                            options: {
                                sum: 12,
                                name: 'autobudget'
                            },

                            is_net_stopped: 0
                        };

                        strategyData.options = u._.extend(strategyData.options, {
                            goal_id: goalVariant.goalId,
                            bid: mcVariant.bid
                        });

                        hint = u.strategy.getHint({
                            strategyData: strategyData,
                            currency: currency,
                            metrika: { campaign_goals: [goal] }
                        })[0];

                        // цель может вернуться строкой или массивом
                        if (typeof hint === 'string') {
                            expect(hint).to.include('Тратить 12.00 руб.');
                            expect(hint).to.include(goalVariant.text);

                            // правильно (не) указана максимальная цена клика
                            mcVariant.bid === 0 ?
                               expect(hint).not.to.include(mcVariant.text) :
                               expect(hint).to.include(mcVariant.text);
                        } else {
                            expect(hint[0]).to.include('Тратить ');
                            expect(hint[1]).to.include('12.00 руб.');

                            // правильно указана цель (всегда массивом)
                            expect(hint[3][0][0]).to.include(goalVariant.text[0]);
                            expect(hint[3][0][1].content)
                                .to.include(goalVariant.text[1]);

                            // правильно (не) указана максимальная цена клика
                            if (mcVariant.bid === 0) {
                                expect(hint).not.to.include(mcVariant.text)
                            } else {
                                expect(hint[4]).to.include(mcVariant.text);
                            }
                        }
                    });
                });
            });

            describe('Если цель удалена', function() {
                var hint;

                before(function() {
                    strategyData = {
                        name: 'autobudget',
                        options: {
                            sum: 12,
                            name: 'autobudget',
                            goal_id: '6002154'
                        },
                        is_net_stopped: 0
                    };

                    goal['goal_status'] = 'Deleted';
                    hint = u.strategy.getHint({
                        strategyData: strategyData,
                        currency: currency,
                        metrika: { campaign_goals: [goal] }
                    })[0]
                });

                after(function() {
                    goal['goal_status'] = 'Active';
                });

                it('В хинте есть объект-предупреждение', function() {
                    expect(hint[3].length).to.equal(3);
                    expect(u._.isObject(hint[3][2])).to.equal(true);
                });

                it('В массиве есть объект с предупреждением', function() {
                    expect(u._.filter(hint[3], function(hintpart) {
                        return hintpart.block === 'b-help-link';
                    }).length).to.be.gt(0);
                });

                it('текст предупреждения - "Цель была удалена"', function() {
                    expect(u._.filter(hint[3], function(hintpart) {
                        return hintpart.block === 'b-help-link';
                    })[0].mix[0].js.hint).to.equal('Цель была удалена')
                });
            });

            describe('Если у цели нет имени ' +
                '(бывает, когда по цель, по которой настроили, не пришла в кампанию)', function() {
                var hint;

                before(function() {
                    strategyData = {
                        name: 'autobudget',
                        options: {
                            goal_id: '123456',
                            sum: 12,
                            name: 'autobudget'
                        },
                        is_net_stopped: 0
                    };

                    hint = u.strategy.getHint({
                        strategyData: strategyData,
                        currency: currency,
                        metrika: { campaign_goals: [goal] }
                    })
                });

                it('Хинт возвращается массивом', function() {
                    expect(u._.isArray(hint[0])).to.equal(true);
                });

                it('В массиве хинт "цель не найдена"', function() {
                    expect(u._.filter(hint[0], function(hintpart) {
                        return u._.isArray(hintpart) && hintpart[1][1].content === 'цель не найдена'
                    }).length).to.be.gt(0);
                });
            });
        });

        describe('Хинты для стратегии AUTOBUDGET_WEEK_BUNDLE', function() {
            before(function() {
                strategyData = {
                    name: 'autobudget_week_bundle',
                    options: {
                        bid: 24,
                        limit_clicks: '100',
                        name: 'autobudget_week_bundle'
                    },
                    net: { name: 'default' },
                    is_net_stopped: 0
                };
            });

            it('Если передан bid, то хинт корректен', function() {
                expect(u.strategy.getHint({
                    strategyData: strategyData,
                    currency: currency
                })[0]).to.equal('Получать 100 кликов в неделю при максимальной ставке не более 24.00&nbsp;руб.');
            });

            it('Если передан avg_bid, то хинт корректен', function() {
                strategyData.options.bid = 0;
                strategyData.options.avg_bid = 24;

                expect(u.strategy.getHint({
                    strategyData: strategyData,
                    currency: currency
                })[0]).to.equal('Получать 100 кликов в неделю при средней цене клика за неделю не более 24.00&nbsp;руб.');
            });

            it('Если не передано ничего, то хинт корректен', function() {
                strategyData.options.bid = 0;
                strategyData.options.avg_bid = 0;

                expect(u.strategy.getHint({
                    strategyData: strategyData,
                    currency: currency
                })[0]).to.equal('Получать 100 кликов в неделю');
            })
        });

        describe('Хинты для стратегии AUTOBUDGET_OPTIMIZATION_CPC', function() {
            var targetVariants = [
                {
                    title: 'на каждый фильтр',
                    text: 'на каждый фильтр',
                    target: 'filter'
                },
                {
                    title: 'на всю кампанию',
                    text: 'на всю кампанию',
                    target: 'camp'
                }],
                weekBudgetVariants = [
                    {
                        sum: 0,
                        title: 'определен недельный бюджет',
                        text: 'при недельном бюджете не более '
                    },
                    {
                        sum: 310,
                        title: 'не определен недельный бюджет',
                        text: 'при недельном бюджете не более 310.00 руб.'
                    }
                ],
                maxClickVariants = [
                    {
                        bid: 0,
                        title: 'не указана максимальная цена клика',
                        text: 'максимальной ставке не более'
                    },
                    {
                        bid: 2,
                        title: 'указана максимальная цена клика',
                        text: 'максимальной ставке не более 2.00 руб.'
                    }
                ];

            weekBudgetVariants.forEach(function(wbVariant) {
                maxClickVariants.forEach(function(mcVariant) {
                    targetVariants.forEach(function(targetVariant) {
                        it('Если ' + [targetVariant.title, wbVariant.title, mcVariant.title].join(', ') +
                            ', то хинт корректен', function() {
                            var target = targetVariant.target,
                                name = 'autobudget_avg_cpc_per_' + target,
                                hint;

                            strategyData = {
                                name: name,
                                options: {
                                    name: name
                                },
                                is_net_stopped: 0
                            };

                            strategyData.options = u._.extend(strategyData.options, {
                                sum: wbVariant.sum,
                                bid: mcVariant.bid,
                                target: target,
                                avg_bid: 12,
                                filter_avg_bid: 12
                            });

                            hint = u.strategy.getHint({ strategyData: strategyData, currency: currency })[0];

                            // правильно указано, по чему идет оптимизация
                            expect(hint).to.include(targetVariant.text);
                            expect(hint).to.include('в среднем за неделю на уровне 12.00 руб.');

                            // правильно (не) указан недельный бюджет
                            wbVariant.sum === 0 ?
                                expect(hint).not.to.include(wbVariant.text) :
                                expect(hint).to.include(wbVariant.text);
                            // правильно (не) указана максимальная цена клика
                            mcVariant.bid === 0 ?
                                expect(hint).not.to.include(mcVariant.text) :
                                expect(hint).to.include(mcVariant.text);
                        });
                    });
                });
            });

            ['camp', 'filter'].forEach(function(target) {
                it('Если ничего не передано, кроме target: ' + target + ', хинт корректен', function() {
                    var name = 'autobudget_avg_cpc_per_' + target;
                    strategyData = {
                        name: name,
                        options: {
                            name: name,
                            target: target
                        },
                        is_net_stopped: 0
                    };

                    expect(u.strategy.getHint({ strategyData: strategyData, currency: currency })[0]).to.equal(
                        target === 'camp' ?
                            'Удерживать CPC на всю кампанию' :
                            'Удерживать CPC на каждый фильтр')
                })
            })
        });

        describe('Хинты для стратегии AUTOBUDGET_OPTIMIZATION_CPA', function() {
            var goalsVariants = [
                {
                    goalId: '6002154',
                    title: 'выбрана цель',
                    text: 'по цели «Просмотреть более 7 страниц»'
                },
                {
                    goalId: '0',
                    title: 'не выбрана цель',
                    text: 'по всем целям'
                }],
                weekBudgetVariants = [
                    {
                        sum: 310,
                        title: 'определен недельный бюджет',
                        text: 'недельном бюджете не более 310.00 руб.'
                    },
                    {
                        sum: 0,
                        title: 'не определен недельный бюджет',
                        text: 'недельном бюджете не более'
                    }
                ],
                maxClickVariants = [
                    {
                        bid: 0,
                        title: 'не указана максимальная цена клика',
                        text: 'максимальной ставке не более'
                    },
                    {
                        bid: 2,
                        title: 'указана максимальная цена клика',
                        text: 'максимальной ставке не более 2.00 руб.'
                    }
                ],
                targetVariants = [
                    {
                        title: 'на каждый фильтр',
                        text: 'на каждый фильтр',
                        target: 'filter'
                    },
                    {
                        title: 'на всю кампанию',
                        text: 'на всю кампанию',
                        target: 'camp'
                    }
                ];

            function checkVariants(options) {
                var goalVariant = options.goalVariant,
                    mcVariant = options.mcVariant,
                    wbVariant = options.wbVariant,
                    targetVariant = options.targetVariant;

                it('Если ' + [goalVariant.title, mcVariant.title, wbVariant.title,
                    targetVariant.title].join(', ') +
                    ' то хинт корректен', function() {
                    var target = targetVariant.target,
                        name = 'autobudget_avg_cpa_per_' + target,
                        hint,
                        hintText,
                        hintObject;

                    strategyData = {
                        name: name,
                        options: {
                            avg_cpa: 10,
                            filter_avg_cpa: 20,
                            name: name
                        },
                        is_net_stopped: 0
                    };

                    strategyData.options = u._.extend(strategyData.options, {
                        sum: wbVariant.sum,
                        bid: mcVariant.bid,
                        target: target,
                        goal_id: goalVariant.goalId
                    });

                    hint = u.strategy.getHint({
                        strategyData: strategyData,
                        currency: currency,
                        metrika: { campaign_goals: [goal] }
                    })[0];

                    hintText = hint;
                    hintText = u._.isArray(hintText) ?
                        u._.map(hint, function(hintpart) {
                            return typeof hintpart === 'string' ? hintpart : hintpart.content;
                        }).join('') :
                        hintText;

                    expect(hintText).to.include('Удерживать CPA');
                    // правильный текст "по СРА/СРС"
                    expect(hintText).to.include(targetVariant.text);
                    // правильная цель
                    expect(hintText).to.include(goalVariant.text);
                    // правильное значение оптимизации
                    targetVariant.target === 'camp' ?
                        expect(hintText).to.include('на уровне 10.00 руб.') :
                        expect(hintText).to.include('на уровне 20.00 руб.');
                    // правильно (не) указан недельный бюджет
                    wbVariant.sum ?
                        expect(hintText).to.include(wbVariant.text) :
                        expect(hintText).not.to.include(wbVariant.text);
                    // правильно (не) указана максимальная цена клика
                    mcVariant.bid ?
                        expect(hintText).to.include(mcVariant.text) :
                        expect(hintText).not.to.include(mcVariant.text);
                });
            }

            goalsVariants.forEach(function(goalVariant) {
                maxClickVariants.forEach(function(mcVariant) {
                    weekBudgetVariants.forEach(function(wbVariant) {
                        targetVariants.forEach(function(targetVariant) {
                            checkVariants({
                                goalVariant: goalVariant,
                                mcVariant: mcVariant,
                                wbVariant: wbVariant,
                                targetVariant: targetVariant
                            });
                        });
                    });
                });
            });

            describe('Если у цели нет имени ' +
                '(бывает, когда по цель, по которой настроили, не пришла в кампанию)', function() {
                var hint;

                before(function() {
                    strategyData = {
                        name: 'autobudget_avg_cpa_per_camp',
                        options: {
                            goal_id: '123456',
                            target: 'camp',
                            avg_bid: 10,
                            name: 'autobudget_avg_cpa_per_camp'
                        },
                        is_net_stopped: 0
                    };

                    hint = u.strategy.getHint({
                        strategyData: strategyData,
                        currency: currency,
                        metrika: { campaign_goals: [goal] }
                    })
                });

                it('Хинт возвращается массивом', function() {
                    expect(u._.isArray(hint[0])).to.equal(true);
                });

                it('В массиве хинт "цель не найдена"', function() {
                    expect(u._.filter(hint[0], function(hintpart) {
                        return u._.isArray(hintpart) && hintpart[1].content === 'цель не найдена'
                    }).length).to.be.gt(0);
                });
            });

            describe('Если цель удалена', function() {
                var hint;

                before(function() {
                    strategyData = {
                        name: 'autobudget_avg_cpa_per_camp',
                        options: {
                            goal_id: '6002154',
                            target: 'camp',
                            avg_bid: 10,
                            name: 'autobudget_avg_cpa_per_camp'
                        }
                    };

                    goal['goal_status'] = 'Deleted';
                    hint = u.strategy.getHint({
                        strategyData: strategyData,
                        currency: currency,
                        metrika: { campaign_goals: [goal] }
                    })[0]
                });

                after(function() {
                    goal['goal_status'] = 'Active';
                });

                it('В хинте есть объект-предупреждение', function() {
                    expect(hint[3].length).to.equal(2);
                    expect(u._.isObject(hint[3][1])).to.equal(true);
                });

                it('В массиве есть объект с предупреждением', function() {
                    expect(u._.filter(hint[3], function(hintpart) {
                        return hintpart.block === 'b-help-link';
                    }).length).to.be.gt(0);
                });

                it('текст предупреждения - "Цель была удалена"', function() {
                    expect(u._.filter(hint[3], function(hintpart) {
                        return hintpart.block === 'b-help-link';
                    })[0].mix[0].js.hint).to.equal('Цель была удалена')
                });
            });
        });
    });

    describe('Тесты на хинт для дневного бюджета', function() {
        var dayBudgetData;
        var sandbox2;
        var spy;

        beforeEach(function() {
            sandbox2 = sinon.sandbox.create();
            sandbox2.stub(window, 'iget2');

            window.iget2
                .withArgs('i-utils', 'raspredelyonnyy')
                .returns('stretched');

            window.iget2.withArgs('i-utils', 'standartnyy')
                .returns('default');

            window.iget2.callThrough();
        });

        afterEach(function() {
            sandbox2.restore();
        });

        it('Если ДБ не установлен, возвращает пустую строку', function() {
            dayBudgetData = {
                set: false,
                sum: 300,
                show_mode: 'stretched'
            };

            expect(u.strategy.getDayBudgetHint({ dayBudget: dayBudgetData, currency: currency })).to.equal('');
        });

        it('Если ДБ установлен, но не задана сумма, возвращает пустую строку', function() {
            dayBudgetData = {
                set: true,
                sum: 0,
                show_mode: 'stretched'
            };

            expect(u.strategy.getDayBudgetHint({ dayBudget: dayBudgetData, currency: currency })).to.equal('');
        });

        it('Если ДБ установлен, возвращает текст со значением суммы', function() {
            dayBudgetData = {
                set: true,
                sum: 300,
                show_mode: 'stretched'
            };

            var result = u.strategy.getDayBudgetHint({ dayBudget: dayBudgetData, currency: currency });

            expect(result).to.include('300.00 руб.');
        });

        it('Если ДБ установлен режим показа стандартный, ' +
            'возвращает текст со значением суммы и текстом "стандартный"', function() {
                dayBudgetData = {
                    set: true,
                    sum: 300,
                    show_mode: 'default'
                };

                var result = u.strategy.getDayBudgetHint({ dayBudget: dayBudgetData, currency: currency });

                expect(result).to.include('default');
            });

        it('Если ДБ установлен режим показа распределенный, ' +
           'возвращает текст со значением суммы и текстом "распределенный"', function() {
               dayBudgetData = {
                   set: true,
                   sum: 300,
                   show_mode: 'stretched'
               };

               var result = u.strategy.getDayBudgetHint({ dayBudget: dayBudgetData, currency: currency });

               expect(result).to.include('stretched');
           });
    });

    describe('Тесты на описания стратегий (отображаются в попапе)', function() {

        var sandbox2;

        beforeEach(function() {
            sandbox2 = sinon.sandbox.create();
        });

        afterEach(function() {
            sandbox2.restore();
        });

        describe('Автоматические стратегии', function() {
            [
                {
                    name: 'autobudget_avg_cpa',
                    text: 'autobudget-avg-cpa-description'
                },
                {
                    name: 'autobudget_avg_click',
                    text: 'autobudget-avg-click-description'
                },
                {
                    name: 'autobudget_avg_cpi',
                    text: 'autobudget-avg-cpi-description'
                },
                {
                    name: 'autobudget_roi',
                    text: 'strategiya-pozvolyaet-poluchit-maksimalnuyu'
                },
                {
                    name: 'roi',
                    text: 'strategiya-pozvolyaet-poluchit-maksimalnuyu'
                },
                {
                    name: 'autobudget',
                    text: 'strategiya-pozvolyaet-effektivno-raspredelit'
                },
                {
                    name: 'autobudget_week_bundle',
                    text: 'strategiya-pozvolyaet-poluchit-fiksirovannoe'
                },
                {
                    name: 'autobudget_optimization_cpc',
                    text: 'strategiya-pozvolyaet-poluchit-maksimalnoe'
                },
                {
                    name: 'autobudget_optimization_cpa',
                    text: 'strategiya-pozvolyaet-poluchit-maksimum-106'
                }
            ].forEach(function(test) {
                it('Для стратегии ' + test.name + ' описание "' + test.text + '"', function() {

                    sandbox2.stub(window, 'iget2')
                        .withArgs('i-utils', test.text)
                        .returns(test.text);

                    expect(u.strategy.getStrategyDescription({ name: test.name })).to.equal(test.text);
                });
            });
        });

        describe('Ручные стратегии', function() {
            it('на всех площадках, нераздельное управление, стратегия DEFAULT', function() {
                sandbox2.stub(window, 'iget2')
                    .withArgs('i-utils', 'strategiya-pozvolyaet-poluchit-maksimalno-110')
                    .returns('strategiya-pozvolyaet-poluchit-maksimalno-110');

                expect(u.strategy.getStrategyDescription({ name: 'default', platform: 'all' }))
                    .to.equal('strategiya-pozvolyaet-poluchit-maksimalno-110');
            });

            it('на сетях (раздельное управление)', function() {
                sandbox2.stub(window, 'iget2')
                    .withArgs('i-utils', 'strategy_maximum_clicks_net')
                    .returns('strategy_maximum_clicks_net');

                expect(u.strategy.getStrategyDescription({ name: 'maximum_clicks', platform: 'net' }))
                    .to.equal('strategy_maximum_clicks_net');
            });

            it('на поиске (нераздельное управление)', function() {
                sandbox2.stub(window, 'iget2')
                    .withArgs('i-utils', 'strategiya-pozvolyaet-poluchit-maksimalno-109')
                    .returns('strategiya-pozvolyaet-poluchit-maksimalno-109');

                expect(u.strategy.getStrategyDescription({ name: 'default', platform: 'search' }))
                    .to.equal('strategiya-pozvolyaet-poluchit-maksimalno-109');
            });

            it('для ГО на поиске (нераздельное управление)', function() {
                sandbox2.stub(window, 'iget2')
                    .withArgs('i-utils', 'strategiya-pozvolyaet-poluchit-maksimalno-109')
                    .returns('strategiya-pozvolyaet-poluchit-maksimalno-109');

                expect(u.strategy.getStrategyDescription({ name: 'search_maximum_coverage', platform: 'search' }))
                    .to.equal('strategiya-pozvolyaet-poluchit-maksimalno-109');
            });
        });
    });
});
