(function(u) {

    // цели метрики
    var goals = {};

    /**
    * Возвращает текстовое описание настроек дневного бюджета
    * @param {Object} options – опции
    * @param {Object} options.dayBudget - дневной бюджет,
    * @param {String} options.currency - валюта кампании
    * @returns {String}
    * @private
    */
    function getDayBudgetHint(options) {
        var dayBudget = options.dayBudget || {},
            sum = dayBudget.sum,
            sumFormatted = u.currencies.formatSum(options.currency, sum);

        if (dayBudget.set && (sum > 0)) {
            return iget2(
                'i-utils',
                'dnevnoj-byudzhet-rezhim-pokazov',
                'Дневной бюджет {budget}, {mode} режим показов', {
                    budget: sumFormatted,
                    mode: dayBudget.show_mode === 'stretched' ?
                        iget2('i-utils', 'raspredelyonnyy', 'распределённый') :
                        iget2('i-utils', 'standartnyy', 'стандартный')
                }
            );
        }

        return '';
    }

    /**
     * Возвращает подпись - описание стратегии для _попапа_
     * @param {Object} strategy - опции стратегии
     * @returns {String}
     */
    function getStrategyDescription(strategy) {

        var isInAppEventsInRmpEnabled = u.consts('isTechInAppEventsInRmpEnabled') || u.consts('isUserInAppEventsInRmpEnabled'),
            maxCoverageHint = iget2(
                'i-utils',
                'strategy_search_maximum_coverage',
                'Стратегия позволяет получить максимально возможный объём трафика при выбранных для каждого условия показа ставках'
            ),
            descriptions = {
                autobudget_avg_click: iget2(
                    'i-utils',
                    'autobudget-avg-click-description',
                    'Стратегия позволяет получить максимальное количество кликов за неделю, учитывая ваши основные ограничения (недельный бюджет, средняя цена клика или целевое количество кликов)'
                ),
                autobudget_avg_cpa: iget2(
                    'i-utils',
                    'autobudget-avg-cpa-description',
                    'Стратегия позволяет получить максимум целевых визитов по заданной в Яндекс.Метрике цели на сайт, удерживая указанную среднюю цену конверсии или недельный бюджет'
                ),
                autobudget_avg_cpi: isInAppEventsInRmpEnabled ?
                    iget2(
                        'i-utils',
                        'cpi-conversions-description',
                        'Стратегия позволяет получить максимум конверсий, удерживая указанную среднюю цену конверсии или недельный бюджет'
                    ) : iget2(
                        'i-utils',
                        'autobudget-avg-cpi-description',
                        'Стратегия позволяет получить максимум установок, удерживая указанную среднюю цену одной установки или недельный бюджет'
                    ),
                autobudget_roi: iget2(
                    'i-utils',
                    'strategiya-pozvolyaet-poluchit-maksimalnuyu',
                    'Стратегия позволяет получить максимальную конверсию по заданной в Яндекс.Метрике цели и приблизить среднюю за неделю рентабельность инвестиций по этой цели к заданному вами значению'
                ),
                roi: iget2(
                    'i-utils',
                    'strategiya-pozvolyaet-poluchit-maksimalnuyu',
                    'Стратегия позволяет получить максимальную конверсию по заданной в Яндекс.Метрике цели и приблизить среднюю за неделю рентабельность инвестиций по этой цели к заданному вами значению'
                ),
                autobudget: iget2(
                    'i-utils',
                    'strategiya-pozvolyaet-effektivno-raspredelit',
                    'Стратегия позволяет эффективно распределить затраты на рекламную кампанию в рамках заданного бюджета'
                ),
                different_places: u.strategy.getPlatform(strategy) === 'net' ?
                    getStrategyDescription(strategy.net) :
                    '',
                autobudget_week_bundle: iget2(
                    'i-utils',
                    'strategiya-pozvolyaet-poluchit-fiksirovannoe',
                    'Стратегия позволяет получить фиксированное количество кликов по минимальной цене'
                ),
                autobudget_optimization_cpc: iget2(
                    'i-utils',
                    'strategiya-pozvolyaet-poluchit-maksimalnoe',
                    'Стратегия позволяет получить максимальное количество кликов за неделю, при этом в среднем клик будет стоить не больше суммы, которую назначил рекламодатель'
                ),
                autobudget_optimization_cpa: iget2(
                    'i-utils',
                    'strategiya-pozvolyaet-poluchit-maksimum-106',
                    'Стратегия позволяет получить максимум целевых визитов по заданной в Яндекс.Метрике цели на сайт и приблизить среднюю цену конверсии за неделю к заданному значению'
                ),
                search_maximum_coverage: maxCoverageHint,

                cpm_default: iget2(
                    'i-utils',
                    'strategiya-pozvolyaet-poluchit-maksimalno-cpm',
                    'Стратегия позволяет получить максимально возможный объём показов при выбранных для каждого условия показа максимальных ценах за тысячу показов'
                ),

                autobudget_max_reach: iget2(
                    'i-utils',
                    'pozvoljaet-poluchit-maksimalnoe-kolichestvo-pokazov-raspredelennyh-po-polzovateljam',
                    'За каждый следующий показ пользователю стратегия делает меньшую ставку, чтобы итоговая средняя цена тысячи показов была как можно меньше и не выше указанной. За счет этого повторные показы пользователю покупаются по эффективной цене и у кампании растёт охват. Задайте распределение бюджета в стратегии - еженедельное или за заданные даты.'
                ),

                autobudget_max_reach_custom_period: iget2(
                    'i-utils',
                    'pozvoljaet-poluchit-maksimalnoe-kolichestvo-pokazov-raspredelennyh-po-polzovateljam',
                    'За каждый следующий показ пользователю стратегия делает меньшую ставку, чтобы итоговая средняя цена тысячи показов была как можно меньше и не выше указанной. За счет этого повторные показы пользователю покупаются по эффективной цене и у кампании растёт охват. Задайте распределение бюджета в стратегии - еженедельное или за заданные даты.'
                ),

                autobudget_max_impressions: iget2(
                    'i-utils',
                    'kolichestvo-pokazov-s-rastjazheniem-bjudzheta-na-nedelju',
                    'Стратегия покупает максимальное количество показов за заданный период и бюджет. За каждый показ стратегия делает такую ставку, чтобы итоговая средняя цена тысячи показов была как можно меньше и не выше указанной. Задайте распределение бюджета в стратегии - еженедельное или за заданные даты.'
                ),

                autobudget_max_impressions_custom_period: iget2(
                    'i-utils',
                    'kolichestvo-pokazov-s-rastjazheniem-bjudzheta-na-nedelju',
                    'Стратегия покупает максимальное количество показов за заданный период и бюджет. За каждый показ стратегия делает такую ставку, чтобы итоговая средняя цена тысячи показов была как можно меньше и не выше указанной. Задайте распределение бюджета в стратегии - еженедельное или за заданные даты.'
                ),

                autobudget_avg_cpv: iget2(
                    'i-utils',
                    'kolichestvo-prosmotrov-s-rastjazheniem-bjudzheta-na-nedelju',
                    'Максимальное количество просмотров с распределением бюджета на неделю. Средняя стоимость просмотра за неделю будет не больше, чем заданное рекламодателем ограничение.'
                ),

                autobudget_avg_cpv_custom_period: iget2(
                    'i-utils',
                    'kolichestvo-prosmotrov-s-rastjazheniem-bjudzheta-na-nedelju',
                    'Максимальное количество просмотров с распределением бюджета на неделю. Средняя стоимость просмотра за неделю будет не больше, чем заданное рекламодателем ограничение.'
                ),

                net: maxCoverageHint,

                search: iget2(
                    'i-utils',
                    'strategiya-pozvolyaet-poluchit-maksimalno-109',
                    'Стратегия позволяет получить максимально возможный объём трафика при выбранных для каждого условия показа максимальных ценах клика'
                ),
                all: iget2(
                    'i-utils',
                    'strategiya-pozvolyaet-poluchit-maksimalno-110',
                    'Стратегия позволяет получить максимально возможный объём трафика при выбранных для каждого условия показа максимальных ценах клика. По умолчанию ставка в сетях устанавливается в зависимости от ставки на поиске'
                ),

                maximum_clicks_all: iget2('i-utils', 'strategy_maximum_clicks_all', 'Стратегия позволяет получить максимально возможный объём трафика при выбранных для каждого условия показа ставках. По умолчанию ставка в сетях устанавливается в зависимости от ставки на поиске'),
                maximum_clicks_search: iget2('i-utils', 'strategy_maximum_clicks_search', 'Стратегия позволяет получить максимально возможный объём трафика при выбранных для каждого условия показа ставках'),
                maximum_clicks_net: iget2('i-utils', 'strategy_maximum_clicks_net', 'Стратегия позволяет получить максимально возможный объём трафика при выбранных для каждого условия показа ставках')
            };

        return descriptions[strategy.name + '_' + strategy.platform] || descriptions[strategy.name] || descriptions[strategy.platform];
    }

    /**
     * Получает хинт для стратегии
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @param {String} options.metrika - данные о метрике кампании
     * @param {Boolean} options.noMetrika - флаг, что недоступны цели и данные метрики (при bemtree-шаблонизации)
     * @param {Boolean} options.hasNoMetrikaCounters - указаны ли счётчики метрики на кампанию
     * @param {Boolean} [options.onlyNet] - флаг компаний для стратегий только в Сетях, других площадок нет
     * @param {Boolean} [options.onlySearch] - флаг компаний для стратегий только на Поиске, других площадок нет
     * @param {Number} [options.contextLimit] - процент бюджета на показы в сетях
     * @param {String} [options.needAttributionModel] - доступно ли редактирование модели атрибуции в попапе стратегий
     * @param {boolean} [options.attributionModel] - значение модели атрибуции
     * @param {String} [options.mediaType] - тип кампании
     * @returns {Array}
     */
    function getHint(options) {
        var strategyData = options.strategyData,
            hints;

        goals = _prepareGoals(
            options.metrika && options.metrika.campaign_goals,
            (u.consts('isMobileAppGoalsForTextCampaignAllowed') || u.consts('isMobileAppGoalsForTextCampaignStrategyEnabled')) && options.mediaType === 'text',
            options.mediaType === 'mobile_content'
        );
        if (options.onlySearch) {
            // сейчас ['mcbanner']
            hints = _getHintForOnlySearchPlatform(options);
        } else if (strategyData.name === u.strategy.DIFFERENT_PLACES) {
            if (options.onlyNet) {
                // формат используют DIFFERENT_PLACES (так криво сложилось исторически, печалька)
                // сейчас ['cpm_banners', 'cpm_deals']
                hints = _getCpmHint(options);
            } else {
                // Раздельное управление: ручные стратегии на всех площадках с раздельным управлением
                // и стратегии "только в сети"
                hints = _getDifferentPlacesHint(options);
            }
        } else {
            // стратегии на всех площадках нераздельные
            // и стратегии только на поиске
            hints = _getNotDifferentPlacesHint(options);
        }

        if (options.needAttributionModel) {
            hints = hints.concat(_getAttributionModelHint(options.attributionModel));
        }

        return hints;
    }

    /**
     * Охватные кампании только в Сетях
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @returns {Array}
     */
    function _getCpmHint(options) {
        var strategyData = options.strategyData;

        if (strategyData.options.net.name === 'cpm_default') {
            // Ручная стратегия
            return _getNotDifferentPlacesHint(options);
        } else {
            // Автоматические стратегии
            return _getOptions({ strategyData: strategyData.options.net, currency: options.currency });
        }
    }

    /**
     * Возвращает период cpm стратегии
     * @param {String} start
     * @param {String} finish
     * @returns {String}
     */
    function getCpmPeriod(start, finish) {
        return iget2('i-utils', 'strategy-cpm-period', 'с {start} по {finish}', {
            start: u.moment(start).format('DD.MM.YYYY'),
            finish: u.moment(finish).format('DD.MM.YYYY'),
            context: 'с 10.09.17 по 12.09.17'
        });
    }

    /**
     * Проверяет закончился период или нет
     * @param {String} finish
     * @returns {Boolean}
     */
    function isExpiredCpmPeriod(finish) {
        return u.moment().isAfter(finish, 'd');
    }

    /**
     * Получает данные модели в JSON
     * @param {String} goalId - id цели
     * @returns {Object}
     * @private
     */
    function _getGoalData(goalId) {
        return goals[goalId] || {};
    }

    /**
     * Кеширует внутрь утилит данные по цели на кампанию
     * @param {Array} campaignGoals - список целей кампании
     * @param {Boolean} isMobileContentGoalsAvailable - доступность целей РМП в кампании
     * @param {Boolean} isMobileContent - кампания РМП
     * @returns {Object}
     * @private
     */
    function _prepareGoals(campaignGoals, isMobileContentGoalsAvailable, isMobileContent) {
        var res = {};

        campaignGoals && campaignGoals.reduce(function(res, goal) {
            res[goal.goal_id] = goal;

            return res;
        }, res);

        if (isMobileContentGoalsAvailable || isMobileContent) {
            // дополнительные цели по РМП объявлены в bricks
            // source.blocks/b-strategy2-settings/_name/b-strategy2-settings_name_cpi.utils.js

            Object.keys(u['goalsIdName']).reduce(function(res, goalId) {
                res[goalId] = { goal_name: u['goalsIdName'][goalId] };

                return res;
            }, res);

            Object.keys(u['techGoalsIdName']).reduce(function(res, goalId) {
                res[goalId] = { goal_name: u['techGoalsIdName'][goalId] };

                return res;
            }, res);

        }

        return res;
    }

    /**
     * Возвращает true, если цель удалена
     * @param {String|null|undefined} goalId - id цели
     * @returns {Boolean}
     * @private
     */
    function _goalIsDeleted(goalId) {
        var goal = _getGoalData(goalId);

        return goalId !== '' && goalId !== null && goalId !== undefined && goalId !== '0' &&
            (goal.goal_status === 'Deleted' || goal.counter_status === 'Deleted');
    }

    /**
     * Получение предупреждения автобюджета
     * @param {String} goalId - id цели
     * @param {String} platform - платформа стратегии
     * @param {String} url - ссылка на хинт
     * @returns {Object}
     * @private
     */
    function _getGoalWarning(goalId, platform, url) {
        // флаг, что цель удалена
        var goalIsDeleted = _goalIsDeleted(goalId);

        if (goalIsDeleted) {
            return {
                block: 'b-help-link',
                mix: [
                    {
                        block: 'b-hintable',
                        js: {
                            hint: iget2('i-utils', 'tsel-byla-udalena', 'Цель была удалена')
                        }
                    }
                ],
                url: u.getHelpUrl(url)
            }
        }
        return '';
    }

    /**
     * Получение хинта для ненайденной цели
     * DIRECT-64320#1489761538000
     * такая ситуация возникает, когда по цели давно не было достижений:
     * в данных стратегии goalId есть, а в данных кампании этой цели нет
     * @param {Boolean} dontNeedVisits - визиты по цели не требуются (холодный пуск автостратегии)
     * @returns {Array}
     * @private
     */
    function _getGoalNotFoundHint(dontNeedVisits) {
        return [
            iget2('i-utils', 'po-tseli', 'по цели: '),
            {
                tag: 'span',
                elem: 'warning',
                content: iget2('i-utils', 'tsel-ne-naidena', 'цель не найдена')
            },
            '&nbsp;',
            {
                block: 'icon',
                mods: { 'size-12': 'alert' },
                mix: [
                    {
                        block: 'b-hintable',
                        js: {
                            hint: dontNeedVisits ? iget2(
                                'i-utils',
                                'vnimanie-vybrannaya-tsel-udalena2',
                                'Внимание! Выбранная цель удалена, для эффективной работы выберите другую цель.'
                            ) : iget2(
                                'i-utils',
                                'vnimanie-vybrannaya-tsel-udalena',
                                'Внимание! Выбранная цель удалена, для эффективной работы выберите цель с достаточным количеством целевых визитов.'
                            )
                        }
                    }
                ]
            }
        ];
    }

    /**
     * Создает текст "по всем целям" или "по цели «Имя цели»"
     * @param {String} goalId - id цели
     * @param {''|Object} warning - предупреждение о проблемах с целью
     * @param {Boolean} dontNeedVisits - визиты по цели не требуются (холодный пуск стратегии)
     * @returns {String|Array}
     * @private
     */

    function _getGoalText(goalId, warning, dontNeedVisits, isCpaPayForConversionEnabled, goals) {
        var goal = _getGoalData(goalId),
            goalName = u.escapeHTML(goal.goal_name),
            goalText;

        if (u.consts('failedToFetchMetrika')) {
            return iget2('i-utils', 'for-goal', 'по цели {goalName}', {
                goalName: {
                    elem: 'warning-text-standalone',
                    content: iget2('i-utils', 'goal-unavailable', 'Цель временно недоступна')
                }
            });
        }

        switch (goalId) {
            case '0':
                goalText = iget2('i-utils', 'po-vsem-tselyam', 'по всем целям');
                break;

            case '13':
                goalText = iget2('i-utils', 'by-key-goals', 'по всем ключевым целям');
                break;

            default:
                if (isCpaPayForConversionEnabled) {
                    var goalData = u._.find(goals, function(c) {
                        return c.id === goalId;
                    });

                    if (goalData) {
                        goalName = goalData.name;
                    }

                }

                if (goalName === undefined) {
                    return _getGoalNotFoundHint(dontNeedVisits);
                }

                goalText = iget2('i-utils', 'po-tseli-goalname', 'по цели «{goalName}»', { goalName: goalName });
                break;
        }

        return warning ? [goalText, warning] : goalText;
    }

    /**
     * Возвращаем JSON с хэлпом
     * @param {String} url - ссылка на хэлп
     * @returns {{block: String, mods: {type: String}, stretched: Boolean, url: String}}
     * @private
     */
    function _getHelpJSON(url) {
        return {
            block: 'b-help-link',
            mods: { type: 'modal' },
            stretched: true,
            url: u.getHelpUrl(url)
        }
    }

    /**
     * Получает предупреждение метрики
     * @param {Object} metrika - данные метрики
     * @param {String} key - ключ, по каким данным метрики смотреть предупреждение
     * @param {Object} helpIconJSON - JSON с хэлпом
     * @returns {Object|String}
     * @private
     */
    function _getDeviationHintJSON(metrika, key, helpIconJSON) {
        if (!metrika || !metrika[key]) return '';

        return {
            tag: 'div',
            elem: 'warning',
            content: [
                {
                    elem: 'warning-icon',
                    content: {
                        block: 'icon',
                        mods: { 'size-12': 'alert' }
                    }
                },
                {
                    elem: 'warning-text',
                    content: key === 'cpi_deviation' ?
                        iget2(
                            'i-utils',
                            'vnimanie-stoimost-ustanovki-znachitelno',
                            'Внимание! Стоимость установки значительно отличается от выбранного значения') :
                        iget2(
                            'i-utils',
                            'vnimanie-stoimost-konversii-znachitelno',
                            'Внимание! Стоимость конверсии значительно отличается от установленного значения')
                },
                helpIconJSON && [
                    ' ',
                    {
                        elem: 'warning-help',
                        content: helpIconJSON
                    }
                ]
            ]
        }
    }

    /**
     * Обрабатывает получившийся хинт, убирает лишние пробелы в массиве
     * @param {String|Array} hint - хинт в "сыром" виде
     * @param {Object} warning - дополнительное предупреждение
     * @returns {Array|String}
     * @private
     */
    function _postProcessHint(hint, warning) {
        var result = hint,
            getNextPart = function(index) {
                if (index < hint.length - 1) {
                    return hint[index + 1];
                }
                return null;
            },
            getPrevPart = function(index) {
                if (index > 1) {
                    return hint[index - 1];
                }
            },
            getPartString = function(part, first) {
                if (u._.isArray(part)) {
                    return part[first ? 0 : part.length - 1];
                }
                return part;
            };

        if (u._.isArray(hint)) {
            result = u._.filter(hint, function(part, index) {
                var isOnlySpace = $.trim(part) === '', // в строке есть только пробелы
                    nextPart = getPartString(getNextPart(index), true), // начало следующего кусочка
                    prevPart = getPartString(getPrevPart(index), false); // конец предыдущего кусочка

                // кусочек непустой и либо состоит не из пробелов, либо пробел между 2 строками
                return part !== '' &&
                    (!isOnlySpace || (typeof nextPart === 'string' && typeof prevPart === 'string'));
            }); // убираем элементы массива, состоящие из одних пробелов
        }
        return warning ? [result].concat(warning) : result; // дописываем warning, если передан
    }

    /**
     * Получение точного имени стратегии (используется в хинте)
     * @param {Object} options - данные стратегии
     * @returns {String}
     */
    function _getHandStrategyTitle(options) {
        var name = options.name || options.strategyData.name,
            platform = options.platform,
            hints = {
                default: options.isChangeManualStrategyName ? iget2('i-utils', 'strategy_hint_default2', 'Ручное управление ставками с оптимизацией') : iget2('i-utils', 'strategy_hint_default', 'Ручное управление ставками с оптимизацией в сетях'),
                search_default: options.isChangeManualStrategyName ? iget2('i-utils', 'strategy_hint_search_default2', 'Ручное управление ставками с оптимизацией') : iget2('i-utils', 'strategy_hint_search_default', 'Ручное управление ставками'),
                net_default: options.isChangeManualStrategyName ? iget2('i-utils', 'strategy_hint_net_default2', 'Ручное управление ставками с оптимизацией') : iget2('i-utils', 'strategy_hint_net_default', 'Ручное управление ставками с оптимизацией в сетях'),
                maximum_coverage: iget2('i-utils', 'maksimalnyy-dostupnyy-ohvat', 'Максимальный доступный охват'),
                search_maximum_coverage: iget2('i-utils', 'maksimalnyy-dostupnyy-ohvat', 'Максимальный доступный охват'),
                different_places: iget2('i-utils', 'razdelno-upravlyat-stavkami-na', 'Раздельно управлять ставками на поиске и в сетях'),
                with_context_limit: options.contextLimit === 100 ?
                    iget2('i-utils', 'context-limit-not-set', 'Расход в сетях не ограничен') :
                    iget2('i-utils', 'aplly-context-limit', 'Расход в сетях удерживать в пределах {limit}% от общего расхода кампании', { limit: options.contextLimit || 0 }),
                with_cpc_hold: iget2('i-utils', 'cpc-hold-enabled', 'Удерживать среднюю стоимость клика в сетях ниже, чем на поиске')
            };

        return hints[platform + '_' + name] || hints[name];
    }

    /**
     * Получает хинт для стратегий с раздельным управлением:
     * а) таб "на всех площадках", ручная стратегия, стоит галочка "раздельное управление"
     * (name: different_plces, search.name: 'default', net.name: 'maximum_coverage')
     * б) таб "только в сетях", все стратегии
     * (name: different_places, search.name: 'stop', net.name: 'maximum_coverage|autobudget|autobudget_roi...')
     * TODO: надо перенести эту логику в маппинг buildStrategy
     * https://st.yandex-team.ru/DIRECT-70262
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @param {String} options.metrika - данные о метрике кампании
     * @returns {Array}
     */
    function _getDifferentPlacesHint(options) {
        var res = [],
            strategyData = options.strategyData,
            search = strategyData.options.search,
            searchName = search.name,
            net = strategyData.options.net,
            netName = net.name;

        if (searchName === u.strategy.STOP) {
            res = _getHintForBothPlatforms(
                // на поиске показы запрещены
                _getOptions(u._.extend(options,{ strategyData: { name: u.strategy.STOP }, platform: 'search' })),
                // в сетях для ручной стратегии пишем только имя, для автоматических - описание
                netName === u.strategy.MAXIMUM_COVERAGE ?
                    _getOptions(u._.extend(options,{ strategyData: { name: u.strategy.DEFAULT }, platform: 'net' })) :
                    _getOptions(u._.extend(options, { strategyData: net, platform: 'net' }))
            );
        } else {
            // ручные стратегии с раздельным управлением.
            // единственная поисковая стратегия - default
            // единственная сетевая стратегия - maximum_coverage
            if (searchName === u.strategy.DEFAULT) {
                // для стратегии "раздельное управление - дефолт" особый вид подписи, без
                // разделения на поиске и в сетях
                res.push(_getHandStrategyTitle(u._.extend(options,{ name: u.strategy.DIFFERENT_PLACES })));
            } else {
                res = _getHintForBothPlatforms(
                    _getOptions(u._.extend(options,{ strategyData: { name: searchName }, platform: 'search' })),
                    _getOptions(u._.extend(options,{ strategyData: { name: netName }, platform: 'net' }))
                );
            }
        }
        return res;
    }

    /**
     * Получает хинт для стратегий без раздельного управления:
     * а) таб "на всех площадках", не стоит галочка "раздельное управление"
     * (name: '', search.name: 'autobudget|autobudget_roi...', net.name:'maximum_coverage')
     * б) таб "только на поиске", все стратегии
     * (name: '', search.name: 'autobudget|autobudget_roi...', net.name:'stop', is_net_stop: 1)
     * TODO: надо перенести эту логику в маппинг buildStrategy
     * https://st.yandex-team.ru/DIRECT-70262
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @param {String} options.metrika - данные о метрике кампании
     * @returns {Array}
     */
    function _getNotDifferentPlacesHint(options) {
        var res = [],
            strategyData = options.strategyData,
            name = strategyData.name;

        // поисковые стратегии
        if (strategyData.is_net_stopped) {
            // для ручной стратегии пишем точное имя стратегии, для автоматических - описание
            res = _getHintForBothPlatforms(
                // на поиске пишем текст хинта
                _getOptions(u._.extend(options, { strategyData: strategyData.options, platform: 'search' })),
                // в сетях показы запрещены
                _getOptions({ strategyData: { name: u.strategy.STOP }, platform: 'net' })
            );

            // данный if нужен только для смарт кампаний, чтобы для них показывать место размещения показов после включения смартов на поиск.
            // и только под фичей (смарт открыт на поиск)
        } else if (
            strategyData.is_search_stopped &&
            // исключаем охватные(медийные) кампании
            !(strategyData.options.net && strategyData.options.net.name === u.strategy.CPM_DEFAULT)
        ) {
            res = _getHintForBothPlatforms(
                // на поиске показы запрещены
                _getOptions({ strategyData: { name: u.strategy.STOP }, platform: 'search' }),
                // в сетях пишем текст хинта
                _getOptions(u._.extend(options, { strategyData: strategyData.options, platform: 'net' }))
            );
        // на всех площадках
        } else {
            // для дефолтной стратегии опять нет уточнений, оставляем только описание
            if (name !== u.strategy.DEFAULT) {
                // для ручной стратегии пишем точное имя стратегии, для автоматических - описание
                res.push(_getOptions(u._.extend(options, { strategyData: strategyData.options, platform: 'search' })));
            } else if (options.contextLimit) {
                options.contextLimit && res.push(_getOptions(u._.extend(options, { name: 'with_context_limit' })));
                options.enable_cpc_hold && res.push(_getOptions(u._.extend(options, { name: 'with_cpc_hold' })));
            }
        }
        return res;
    }

    /**
     * Получение хинта для платформ "на поиске" и "в сетях"
     * @param {String|Array|Object} searchHint - хинт на поиске
     * @param {String|Array|Object} netHint - хинт в сети
     * @returns {Array}
     * @private
     */
    function _getHintForBothPlatforms(searchHint, netHint) {
        var res = [];

        res.push(iget2(
            'i-utils',
            'na-poiske-search-hint',
            'На поиске:&nbsp;{searchHint}', { searchHint: searchHint }));
        res.push(iget2(
            'i-utils',
            'v-setyah-net-hint',
            'В сетях:&nbsp;{netHint}', { netHint: netHint }));

        return res;
    }

    /**
     * Получает описание для стратегии на Поиске
     * @param {Object} options
     * @returns {Array}
     * @private
     */
    function _getHintForOnlySearchPlatform(options) {
        var res = [],
            strategyData = options.strategyData;

        if (strategyData.name !== u.strategy.SEARCH_MAXIMUM_COVERAGE) {
            res.push(_getOptions(u._.extend(options, { strategyData: strategyData.options, platform: 'search' })));
        }

        return res;
    }

    /**
     * Получает описание для автоматических стратегий
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @param {String} options.metrika - данные о метрике кампании
     * @param {'net'|undefined} options.platform - платформа показа
     * @returns {Array|String}
     */
    function _getOptions(options) {
        var strategyData = options.strategyData,
            name = strategyData && strategyData.name,
            isUserInAppEventsInRmpEnabled = u.consts('isUserInAppEventsInRmpEnabled'),
            methods = {
                autobudget_avg_click: _getAvgClickHint,
                autobudget_avg_cpa: _getAvgCPAHint,
                autobudget_avg_cpi: isUserInAppEventsInRmpEnabled ? _getAvgCPAHint : _getCPIHint,
                autobudget_roi: _getROIHint,
                autobudget: _getAutobudgetHint,
                autobudget_week_bundle: _getWeekBundleHint,
                autobudget_optimization_cpc: _getOptCPCHint,
                autobudget_avg_cpc_per_camp: _getOptCPCHint,
                autobudget_avg_cpc_per_filter: _getOptCPCHint,
                autobudget_optimization_cpa: _getOptCPAHint,
                autobudget_avg_cpa_per_camp: _getOptCPAHint,
                autobudget_avg_cpa_per_filter: _getOptCPAHint,
                default: _getHandStrategyTitle,
                maximum_clicks: _getHandStrategyTitle,
                maximum_coverage: _getHandStrategyTitle,
                search_maximum_coverage: _getHandStrategyTitle,
                stop: _getStopHint,
                autobudget_max_impressions_custom_period: _getOptCpmHintForPeriod,
                autobudget_max_impressions: _getOptCpmHintForWeek,
                autobudget_max_reach_custom_period: _getOptCpmHintForPeriod,
                autobudget_max_reach: _getOptCpmHintForWeek,
                autobudget_avg_cpv_custom_period: _getOptAvgCpvHintForPeriod,
                autobudget_avg_cpv: _getOptAvgCpvHintForWeek
            };
        return methods[name] ? methods[name](options) : '';
    }

    /**
     * Возвращает хинт на Автоматической стратегии
     * для Охватной кампании За Неделю
     * @param {Object} options
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @returns {Array}
     * @private
     */
    function _getOptCpmHintForWeek(options) {
        var strategyData = options.strategyData;

        return [
            iget2('i-utils', 'strategy-preview-week', 'Тратить за неделю'),
            iget2('i-utils', 'strategy-preview-budget', 'Бюджет {currency}', {
                currency: u.currencies.formatSum(options.currency, strategyData.sum),
                context: 'Бюджет 4 650.00 руб.'
            }),
            iget2('i-utils', 'strategy-preview-cpm', 'Средняя цена не более {currency} за тысячу показов', {
                currency: u.currencies.formatSum(options.currency, strategyData.avg_cpm),
                context: 'Средняя цена не более 5.00 руб за тысячу показов'
            })
        ];
    }

    /**
     * Возвращает хинт на "Оптимизация просмотров"
     * для Охватной кампании За Неделю
     * @param {Object} options
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @returns {Array}
     * @private
     */
    function _getOptAvgCpvHintForWeek(options) {
        var strategyData = options.strategyData;

        return [
            iget2('i-utils', 'strategy-preview-week', 'Тратить за неделю'),
            iget2('i-utils', 'strategy-preview-budget', 'Бюджет {currency}', {
                currency: u.currencies.formatSum(options.currency, strategyData.sum),
                context: 'Бюджет 4 650.00 руб.'
            }),
            iget2('i-utils', 'strategy-preview-avg-cpv', 'Средняя цена {currency} за просмотр', {
                currency: u.currencies.formatSum(options.currency, strategyData.avg_cpv),
                context: 'Средняя цена не более 5.00 руб за тысячу показов'
            })
        ];
    }

    /**
     * Возвращает хинт на Автоматической стратегии
     * для Охватной кампании За Период
     * @param {Object} options
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @returns {Array}
     * @private
     */
    function _getOptCpmHintForPeriod(options) {
        var strategyData = options.strategyData,
            res = [
                iget2('i-utils', 'strategy-preview-period', 'Тратить в период с {start} по {finish}', {
                    start: u.moment(strategyData.start).format('DD.MM.YYYY'),
                    finish: u.moment(strategyData.finish).format('DD.MM.YYYY'),
                    context: 'Тратить в период с 25.01.2018 по 24.02.2018'
                }),
                iget2('i-utils', 'strategy-preview-budget', 'Бюджет {currency}', {
                    currency: u.currencies.formatSum(options.currency, strategyData.budget),
                    context: 'Бюджет 4 650.00 руб.'
                }),
                iget2('i-utils', 'strategy-preview-cpm', 'Средняя цена не более {currency} за тысячу показов', {
                    currency: u.currencies.formatSum(options.currency, strategyData.avg_cpm),
                    context: 'Средняя цена не более 5.00 руб за тысячу показов'
                })
            ];

        (strategyData.auto_prolongation == 1) &&
            res.push(iget2('i-utils', 'strategy-preview-prolongation', 'Продлевать автоматически'));

        // в рамках DIRECT-71249
        // при создании cpm кампании, сервер присылает дефолтную невалидную стратегию, пользователь должен ее настроить
        if (!strategyData.budget || !strategyData.avg_cpm) {
            return [{
                elem: 'hint-warning',
                content: iget2('i-utils', 'not-specified-budget-and-prices', 'Не заданы ограничения бюджета и цены')
            }];
        }

        return res;
    }

    /**
     * Возвращает хинт для стратегии "Оптимизация просмотров"
     * для Охватной кампании За Период
     * @param {Object} options
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @returns {Array}
     * @private
     */
    function _getOptAvgCpvHintForPeriod(options) {
        var strategyData = options.strategyData,
            res = [
                iget2('i-utils', 'strategy-preview-period', 'Тратить в период с {start} по {finish}', {
                    start: u.moment(strategyData.start).format('DD.MM.YYYY'),
                    finish: u.moment(strategyData.finish).format('DD.MM.YYYY'),
                    context: 'Тратить в период с 25.01.2018 по 24.02.2018'
                }),
                iget2('i-utils', 'strategy-preview-budget', 'Бюджет {currency}', {
                    currency: u.currencies.formatSum(options.currency, strategyData.budget),
                    context: 'Бюджет 4 650.00 руб.'
                }),
                iget2('i-utils', 'strategy-preview-avg-cpv', 'Средняя цена {currency} за просмотр', {
                    currency: u.currencies.formatSum(options.currency, strategyData.avg_cpv)
                })
            ];

        (strategyData.auto_prolongation == 1) &&
        res.push(iget2('i-utils', 'strategy-preview-prolongation', 'Продлевать автоматически'));

        if (!strategyData.budget || !strategyData.avg_cpv) {
            return [{
                elem: 'hint-warning',
                content: iget2('i-utils', 'not-specified-budget-and-prices', 'Не заданы ограничения бюджета и цены')
            }];
        }

        return res;
    }

    /**
     * Возвращает хинт про запрещение показов
     * @returns {String}
     * @private
     */
    function _getStopHint() {
        return iget2('i-utils', 'pokazy-zapreshcheny', 'Показы запрещены');
    }

    /**
     * Возвращает хинт для стратегии "Средняя цена клика"
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @private
     */
    function _getAvgClickHint(options) {
        var strategyData = options.strategyData,
            currency = options.currency,
            avgBid = u.currencies.formatSum(currency, strategyData.avg_bid);

        return strategyData.sum ?
            iget2(
                'i-utils',
                'uderzhivat-tsenu-clicka',
                'Удерживать цену клика {avgBid} в среднем за неделю при недельном бюджете не более {weekSum}',{
                    avgBid: avgBid,
                    weekSum: u.currencies.formatSum(currency, strategyData.sum)
                }
            ) : iget2(
                'i-utils',
                'uderzhivat-tsenu-clicka-1',
                'Удерживать цену клика {avgBid} в среднем за неделю', {
                    avgBid: avgBid,
                }
            );
    }

    /**
     * Возвращает хинт для стратегии "средняя цена конверсии"
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @param {String} options.metrika - данные о метрике кампании
     * @param {String} options.mediaType - данные о метрике кампании
     * @returns {Array|String}
     * @private
     */
    function _getAvgCPAHint(options) {
        var currency = options.currency,
            goals = (options.metrika || {}).campaign_goals || [],
            strategyData = options.strategyData,
            isCpaPayForConversionEnabled = options.isCpaPayForConversionEnabled,
            goalId = strategyData.goal_id,
            helpIconJSON = _getHelpJSON('average-cpa'),
            goalsPlatform = options.platform === 'net' ? 'context_' : '',
            warning = _getGoalWarning(goalId, goalsPlatform, 'average-cpa'),
            goalText = _getGoalText(goalId, warning, true, isCpaPayForConversionEnabled, goals),
            deviationWarning = _getDeviationHintJSON(options.metrika, 'cpa_deviation', helpIconJSON),
            isMobileContent = options.mediaType === 'mobile_content',
            values = {
                goal: goalText,
                avgCPA: u.currencies.formatSum(currency, isMobileContent ? strategyData.avg_cpi : strategyData.avg_cpa),
                weekSum: u.currencies.formatSum(currency, strategyData.sum),
                maxClick: u.currencies.formatSum(currency, strategyData.bid)
            },
            keys = [],
            labels,
            result;

        strategyData.sum && keys.push('sum');
        strategyData.bid && keys.push('bid');

        if (!goalId) {
            return [{
                elem: 'hint-warning',
                content: iget2('i-utils', 'invalid-avg-cpa-strategy', 'Перед настройкой стратегии укажите счетчик Метрики и задайте ключевые цели кампании')
            }];
        }

        labels = {
            common: isCpaPayForConversionEnabled ? iget2(
                'i-utils',
                'uderzhivat-tsenu-conversii-4',
                'Удерживать цену конверсии {avgCPA} {goal}',
                values) : iget2(
                'i-utils',
                'uderzhivat-tsenu-conversii',
                'Удерживать цену конверсии {avgCPA} в среднем за неделю {goal}',
                values),
            bid: iget2(
                'i-utils',
                'uderzhivat-tsenu-conversii-1',
                'Удерживать цену конверсии {avgCPA} в среднем за неделю {goal} при максимальной цене клика не более {maxClick}',
                values),
            sum: isCpaPayForConversionEnabled ? iget2(
                'i-utils',
                'uderzhivat-tsenu-conversii-5',
                'Удерживать цену конверсии {avgCPA} {goal}, тратить в неделю не более {weekSum}',
                values) : iget2(
                'i-utils',
                'uderzhivat-tsenu-conversii-2',
                'Удерживать цену конверсии {avgCPA} в среднем за неделю {goal}, тратить в неделю не более {weekSum}',
                values),
            'sum-bid': isCpaPayForConversionEnabled ? iget2(
                'i-utils',
                'uderzhivat-tsenu-conversii-6',
                'Удерживать цену конверсии {avgCPA} {goal}, тратить в неделю не более {weekSum} при максимальной цене клика не более {maxClick}',
                values) : iget2(
                'i-utils',
                'uderzhivat-tsenu-conversii-3',
                'Удерживать цену конверсии {avgCPA} в среднем за неделю {goal}, тратить в неделю не более {weekSum} при максимальной цене клика не более {maxClick}',
                values)
        };

        result = labels[keys.join('-') || 'common'];

        return _postProcessHint(result, deviationWarning);
    }

    /**
     * Возвращает хинт для стратегии "средняя цена установки приложения"
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @param {Object} options.metrika - данные о метрике кампании и РМП-счетчиках
     * @returns {Array|String}
     * @private
     */
    function _getCPIHint(options) {
        var currency = options.currency,
            strategyData = options.strategyData,
            warning = _getDeviationHintJSON(options.metrika, 'cpi_deviation'),
            values = {
                avgCPI: u.currencies.formatSum(currency, strategyData.avg_cpi),
                weekSum: u.currencies.formatSum(currency, strategyData.sum),
                maxClick: u.currencies.formatSum(currency, strategyData.bid)
            },
            keys = [],
            labels,
            result;

        strategyData.sum && keys.push('sum');
        strategyData.bid && keys.push('bid');

        labels = {
            common: iget2(
                'i-utils',
                'uderzhivat-tsenu-ustanovki',
                'Удерживать цену установки {avgCPI} в среднем за неделю', values),
            bid: iget2(
                'i-utils',
                'uderzhivat-tsenu-ustanovki-1',
                'Удерживать цену установки {avgCPI} в среднем за неделю при максимальной цене клика не более {maxClick}',
                values),
            sum: iget2(
                'i-utils',
                'uderzhivat-tsenu-ustanovki-2',
                'Удерживать цену установки {avgCPI} в среднем за неделю, тратить в неделю не более {weekSum}',
                values),
            'sum-bid': iget2(
                'i-utils',
                'uderzhivat-tsenu-ustanovki-3',
                'Удерживать цену установки {avgCPI} в среднем за неделю, тратить в неделю не более {weekSum} при максимальной цене клика не более {maxClick}',
                values)
        };
        result = labels[keys.join('-') || 'common'];

        return _postProcessHint(result, warning);
    }

    /**
     * Возвращает хинт для стратегии "средняя рентабельность инвестиций"
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {Boolean} options.hasNoMetrikaCounters - указаны ли счётчики метрики на кампанию
     * @param {String} options.currency - валюта кампании
     * @returns {Array|String}
     * @private
     */
    function _getROIHint(options) {
        var currency = options.currency,
            strategyData = options.strategyData,
            goalId = strategyData.goal_id,
            goalsPlatform = options.platform === 'net' ? 'context_' : '',
            warning = _getGoalWarning(goalId, goalsPlatform, 'average-roi'),
            goalText = _getGoalText(goalId, warning),
            values = {
                goal: goalText,
                roiCoef: parseFloat(strategyData.roi_coef).toFixed(2),
                resReturn: strategyData.reserve_return + '%',
                weekSum: u.currencies.formatSum(currency, strategyData.sum || ''),
                maxClick: u.currencies.formatSum(currency, strategyData.bid || ''),
                profitability: strategyData.profitability + '%'
            },
            keys = [],
            labels,
            result = [];

        strategyData.sum && keys.push('sum');
        strategyData.bid && keys.push('bid');
        strategyData.profitability && keys.push('profitability');

        if (options.hasNoMetrikaCounters && goalId === undefined) {
            return [{
                elem: 'hint-warning',
                content: iget2('i-utils', 'invalid-avg-cpa-strategy', 'Перед настройкой стратегии укажите счетчик Метрики и задайте ключевые цели кампании')
            }];
        }

        if (strategyData.roi_coef == null) {
            return [{
                elem: 'hint-warning',
                content: iget2('i-utils', 'invalid-roi-coef', 'Задайте целевой ROI')
            }];
        }

        labels = {
            common: iget2(
                'i-utils',
                'uderzhivat-rentablenost',
                'Удерживать рентабельность инвестиций на уровне {roiCoef} {goal}, возвращать в рекламу {resReturn} сэкономленного бюджета',
                values),
            sum: iget2(
                'i-utils',
                'uderzhivat-rentablenost-1',
                'Удерживать рентабельность инвестиций на уровне {roiCoef} {goal}, возвращать в рекламу {resReturn} сэкономленного бюджета при недельном бюджете не более {weekSum}',
                values),
            profitability: iget2(
                'i-utils',
                'uderzhivat-rentablenost-2',
                'Удерживать рентабельность инвестиций на уровне {roiCoef} {goal}, возвращать в рекламу {resReturn} сэкономленного бюджета, учитывать, что {profitability} доходов является себестоимостью товаров или услуг',
                values),
            bid: iget2(
                'i-utils',
                'uderzhivat-rentablenost-3',
                'Удерживать рентабельность инвестиций на уровне {roiCoef} {goal}, возвращать в рекламу {resReturn} сэкономленного бюджета при максимальной ставке не более {maxClick}',
                values),
            'sum-profitability': iget2(
                'i-utils',
                'uderzhivat-rentablenost-4',
                'Удерживать рентабельность инвестиций на уровне {roiCoef} {goal}, возвращать в рекламу {resReturn} сэкономленного бюджета при недельном бюджете не более {weekSum}, учитывать, что {profitability} доходов является себестоимостью товаров или услуг',
                values),
            'bid-profitability': iget2(
                'i-utils',
                'uderzhivat-rentablenost-5',
                'Удерживать рентабельность инвестиций на уровне {roiCoef} {goal}, возвращать в рекламу {resReturn} сэкономленного бюджета при максимальной ставке не более {maxClick}, учитывать, что {profitability} доходов является себестоимостью товаров или услуг',
                values),
            'sum-bid': iget2(
                'i-utils',
                'uderzhivat-rentablenost-6',
                'Удерживать рентабельность инвестиций на уровне {roiCoef} {goal}, возвращать в рекламу {resReturn} сэкономленного бюджета при недельном бюджете не более {weekSum}, максимальной ставке не более {maxClick}',
                values),
            'sum-bid-profitability': iget2(
                'i-utils',
                'uderzhivat-rentablenost-7',
                'Удерживать рентабельность инвестиций на уровне {roiCoef} {goal}, возвращать в рекламу {resReturn} сэкономленного бюджета при недельном бюджете не более {weekSum}, максимальной ставке не более {maxClick}, учитывать, что {profitability} доходов является себестоимостью товаров или услуг',
                values)
        };

        result = labels[keys.join('-') || 'common'];

        return _postProcessHint(result);
    }

    /**
     * Получение текста цели для стратегии "недельный бюджет"
     * @param {String} goalId - id цели
     * @param {String | undefined} sum - сумма бюджета
     * @param {Object} warning - предупреждение автобюджета
     * @returns {String|Array}
     * @private
     */
    function _getAutobudgetGoalText(goalId, sum, warning) {
        var goalData = _getGoalData(goalId);

        switch (goalId) {
            case '': // максимум кликов
            case null: // максимум кликов
            case undefined: // вообще недоступна метрика
                return iget2('i-utils','poluchat-maksimum-klikov', 'получать максимум кликов');
            case u.strategy.MOBILE_CPA_GOAL_ID:
                return iget2('i-utils', 'poluchat-maksimalnoe-kolichestvo-ustanovok', 'получать максимальное количество установок');
            case '0':
                return iget2('i-utils', 'poluchat-maksimalnuyu-konversiyu-po', 'получать максимальную конверсию по всем целям');
            case '13':
                return iget2('i-utils', 'get-maximum-conversions-for-all-key-goals', 'получать максимум конверсий по всем ключевым целям');
            default:
                if (goalData.goal_name === undefined) {
                    return iget2(
                        'i-utils',
                        'poluchat-max-conversiu-po-tseli-ne-naidena',
                        'получать максимальную конверсию {goal}', { goal: _getGoalNotFoundHint(true) }
                        );
                }
                return [
                    iget2(
                        'i-utils',
                        'poluchat-max-conversiu-po-tseli',
                        'получать максимальную конверсию по цели&nbsp;«{goalName}»',
                        {
                            goalName: function() {
                                return {
                                    tag: 'span',
                                    elem: 'text',
                                    elemMods: { warning: warning ? 'yes' : '' },
                                    content: u.escapeHTML(goalData.goal_name)
                                }
                            }()
                        }
                    ),
                    '&nbsp;',
                    warning
                ]
        }
    }

    /**
     * Возвращает хинт для стратегии "недельный бюджет"
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @param {'net'|undefined} options.platform - платформа показа
     * @returns {Array|String}
     * @private
     */
    function _getAutobudgetHint(options) {
        var currency = options.currency,
            strategyData = options.strategyData,
            goalId = strategyData.goal_id,
            goalsPlatform = options.platform === 'net' ? 'context_' : '',
            warning = _getGoalWarning(goalId, goalsPlatform, 'weekly-budget'),
            values = {
                sum: u.currencies.formatSum(currency, strategyData.sum),
                bid: u.currencies.formatSum(currency, strategyData.bid),
                goal: _getAutobudgetGoalText(goalId, strategyData.sum, warning)
            },
            labels = {
                common: iget2(
                        'i-utils',
                        'spend-weekly',
                        'Тратить {sum} за неделю, {goal}',
                        values),
                bid: iget2(
                        'i-utils',
                        'spend-weekly-with-max-bid',
                        'Тратить {sum} за неделю, {goal} при максимальной ставке не более {bid}',
                        values)
            },
            result;

        result = labels[strategyData.bid ? 'bid' : 'common'];

        if (!strategyData.sum) {
            return [{
                elem: 'hint-warning',
                content: iget2('i-utils', 'invalid-autobudget-strategy', 'Укажите, сколько бы вы хотели тратить в неделю')
            }];
        }

        return _postProcessHint(result);
    }

    /**
     * Возвращает хинт для стратегии "недельный пакет кликов"
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @returns {Array|String}
     * @private
     */
    function _getWeekBundleHint(options) {
        var currency = options.currency,
            strategyData = options.strategyData,
            limit = strategyData.limit_clicks,
            values = {
                bid: u.currencies.formatSum(currency, strategyData.bid, { delim: '&nbsp;' }),
                avgBid: u.currencies.formatSum(currency, strategyData.avg_bid, { delim: '&nbsp;' }),
                clicks: u.pluralizeWord([
                    iget2('i-utils', 'klik', 'клик'),
                    iget2('i-utils', 'klika', 'клика'),
                    iget2('i-utils', 'klikov', 'кликов')
                ], limit),
                limit: limit
            },
            labels = {
                common: iget2(
                        'i-utils',
                        'get-clicks-limit',
                        'Получать {limit} {clicks} в неделю',
                        values),
                bid: iget2(
                        'i-utils',
                        'get-clicks-limit-with-max-bid',
                        'Получать {limit} {clicks} в неделю при максимальной ставке не более&nbsp;{bid}',
                        values),
                avg: iget2(
                        'i-utils',
                        'get-clicks-limit-with-avg-bid',
                        'Получать {limit} {clicks} в неделю при средней цене клика за неделю не более&nbsp;{avgBid}',
                        values)
            },
            keys = [];

        strategyData.bid && keys.push('bid');
        strategyData.avg_bid && keys.push('avg');

        return labels[keys.join('-') || 'common'];
    }

    /**
     * Возвращает хинт для стратегии "оптимизация количества кликов" (в смартах и МКБаннере)
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @returns {Array|String}
     * @private
     */
    function _getOptCPCHint(options) {
        var currency = options.currency,
            strategyData = options.strategyData,
            values = {
                sum: u.currencies.formatSum(currency, strategyData.sum || ''),
                bid: u.currencies.formatSum(currency, strategyData.bid || ''),
                campBid: u.currencies.formatSum(currency, strategyData.avg_bid || ''),
                filterBid: u.currencies.formatSum(currency, strategyData.filter_avg_bid || '')
            },
            keys = [],
            labels;

        strategyData.target && keys.push(strategyData.target);
        (strategyData.avg_bid || strategyData.filter_avg_bid) && keys.push('cpc');

        strategyData.sum && keys.push('sum');
        strategyData.bid && keys.push('bid');

        labels = {
            camp: iget2(
                'i-utils',
                'uderzhivat-cpc-na-vsyu-kampaniyu',
                'Удерживать CPC на всю кампанию'),
            'camp-cpc': iget2(
                'i-utils',
                'uderzhivat-cpc-na-vsyu-kampaniyu-1',
                'Удерживать CPC на всю кампанию в среднем за неделю на уровне {campBid}',
                values),
            'camp-cpc-sum': iget2(
                'i-utils',
                'uderzhivat-cpc-na-vsyu-kampaniyu-2',
                'Удерживать CPC на всю кампанию в среднем за неделю на уровне {campBid} при недельном бюджете не более {sum}',
                values),
            'camp-cpc-bid': iget2(
                'i-utils',
                'uderzhivat-cpc-na-vsyu-kampaniyu-3',
                'Удерживать CPC на всю кампанию в среднем за неделю на уровне {campBid} при максимальной ставке не более {bid}',
                values),
            'camp-cpc-sum-bid': iget2(
                'i-utils',
                'uderzhivat-cpc-na-vsyu-kampaniyu-4',
                'Удерживать CPC на всю кампанию в среднем за неделю на уровне {campBid} при недельном бюджете не более {sum}, максимальной ставке не более {bid}',
                values),
            filter: iget2(
                'i-utils',
                'uderzhivat-cpc-na-kazhdyy-filtr',
                'Удерживать CPC на каждый фильтр'),
            'filter-cpc': iget2(
                'i-utils',
                'uderzhivat-cpc-na-kazhdyy-filtr-1',
                'Удерживать CPC на каждый фильтр в среднем за неделю на уровне {filterBid}', values),
            'filter-cpc-sum': iget2(
                'i-utils',
                'uderzhivat-cpc-na-kazhdyy-filtr-2',
                'Удерживать CPC на каждый фильтр в среднем за неделю на уровне {filterBid} при недельном бюджете не более {sum}',
                values),
            'filter-cpc-bid': iget2(
                'i-utils',
                'uderzhivat-cpc-na-kazhdyy-filtr-3',
                'Удерживать CPC на каждый фильтр в среднем за неделю на уровне {filterBid} при максимальной ставке не более {bid}',
                values),
            'filter-cpc-sum-bid': iget2(
                'i-utils',
                'uderzhivat-cpc-na-kazhdyy-filtr-4',
                'Удерживать CPC на каждый фильтр в среднем за неделю на уровне {filterBid} при недельном бюджете не более {sum}, максимальной ставке не более {bid}',
                values)
        };

        return labels[keys.join('-')];
    }

    /**
     * Возвращает хинт для стратегии "оптимизация количества конверсий" (в смартах)
     * @param {Object} options - опции
     * @param {Object} options.strategyData - данные стратегии
     * @param {String} options.currency - валюта кампании
     * @returns {Array|String}
     * @private
     */
    function _getOptCPAHint(options) {
        var currency = options.currency,
            strategyData = options.strategyData,
            specialGoals = (options.metrika || {}).campaign_goals || [],
            goalId = strategyData.goal_id,
            isCpaPayForConversionEnabled = options.isCpaPayForConversionEnabled,
            goalsPlatform = options.platform === 'net' ? 'context_' : '',
            warning = _getGoalWarning(goalId, goalsPlatform, 'optimize-number-of-conversions'),
            goalText = _getGoalText(goalId, warning, false, isCpaPayForConversionEnabled, specialGoals),
            target = strategyData.target,
            values = {
                sum: u.currencies.formatSum(currency, strategyData.sum || ''),
                bid: u.currencies.formatSum(currency, strategyData.bid || ''),
                campCpa: u.currencies.formatSum(currency, strategyData.avg_cpa || ''),
                filterCpa: u.currencies.formatSum(currency, strategyData.filter_avg_cpa || ''),
                CPA: 'CPA',
                goal: goalText
            },
            labels,
            keys = [],
            result;

        isCpaPayForConversionEnabled ? keys.push('conversion') : target && keys.push(target);

        strategyData.sum && keys.push('sum');
        !isCpaPayForConversionEnabled && strategyData.bid && keys.push('bid');

        labels = {
            camp: iget2(
                'i-utils',
                'uderzhivat-cpa-na-vsyu-kampaniyu-1',
                'Удерживать CPA на всю кампанию в среднем за неделю на уровне {campCpa} {goal}',
                values),
            'camp-sum': iget2(
                'i-utils',
                'uderzhivat-cpa-na-vsyu-kampaniyu-2',
                'Удерживать CPA на всю кампанию в среднем за неделю на уровне {campCpa} {goal} при недельном бюджете не более {sum}',
                values),
            'camp-bid': iget2(
                'i-utils',
                'uderzhivat-cpa-na-vsyu-kampaniyu-3',
                'Удерживать CPA на всю кампанию в среднем за неделю на уровне {campCpa} {goal} при максимальной ставке не более {bid}',
                values),
            'camp-sum-bid': iget2(
                'i-utils',
                'uderzhivat-cpa-na-vsyu-kampaniyu-4',
                'Удерживать CPA на всю кампанию в среднем за неделю на уровне {campCpa} {goal} при недельном бюджете не более {sum}, максимальной ставке не более {bid}',
                values),
            filter: iget2(
                'i-utils',
                'uderzhivat-cpa-na-kazhdyy-filtr-1',
                'Удерживать CPA на каждый фильтр в среднем за неделю на уровне {filterCpa} {goal}', values),
            'filter-sum': iget2(
                'i-utils',
                'uderzhivat-cpa-na-kazhdyy-filtr-2',
                'Удерживать CPA на каждый фильтр в среднем за неделю на уровне {filterCpa} {goal} при недельном бюджете не более {sum}',
                values),
            'filter-bid': iget2(
                'i-utils',
                'uderzhivat-cpa-na-kazhdyy-filtr-3',
                'Удерживать CPA на каждый фильтр в среднем за неделю на уровне {filterCpa} {goal} при максимальной ставке не более {bid}',
                values),
            'filter-sum-bid': iget2(
                'i-utils',
                'uderzhivat-cpa-na-kazhdyy-filtr-4',
                'Удерживать CPA на каждый фильтр в среднем за неделю на уровне {filterCpa} {goal} при недельном бюджете не более {sum}, максимальной ставке не более {bid}',
                values),
            conversion: iget2(
                'i-utils',
                'uderzhivat-cpa-conversion-enabled',
                'Удерживать цену конверсии {campCpa} {goal}',
                values),
            'conversion-sum': iget2(
                'i-utils',
                'uderzhivat-cpa-conversion-enabled-sum',
                'Удерживать цену конверсии {campCpa} {goal}, тратить в неделю не более {sum}',
                values)
        };

        result = labels[keys.join('-')];

        return _postProcessHint(result);
    }

    function _getAttributionModelHint(attributionModel) {
        var description;

        switch (attributionModel) {
            case 'last_yandex_direct_click':
                description = iget2('i-utils', 'last-direct-click', 'Последний переход из Яндекс.Директа');
                break;
            case 'last_significant_click':
                description = iget2('i-utils', 'last-significant-click', 'последний значимый переход');
                break;
            case 'last_click':
                description = iget2('i-utils', 'last-click', 'последний переход');
                break;
            case 'first_click':
                description = iget2('i-utils', 'first-click', 'первый переход');
                break;
        }

        return description && iget2('i-utils', 'attribution-model-hint', 'Модель атрибуции: {description}', { description: description });
    }

    u.register({
        strategy: u._.extend({}, {
            getDayBudgetHint: getDayBudgetHint,
            getStrategyDescription: getStrategyDescription,
            getHint: getHint,
            getCpmPeriod: getCpmPeriod,
            isExpiredCpmPeriod: isExpiredCpmPeriod
        })
    });

})(u);
