describe('i-utils__table', function() {

    describe('.getSortHeaderLink', function() {

        it('должен вернуть ссылку', function() {
            expect(u.table.getSortHeaderLink()).to.match(/^<a\s[^>]+><\/a>/);
        });

        it('должен вернуть html entity код стрелки &uarr обернутый в тег small', function() {
            expect(u.table.getSortHeaderLink()).to.match(/<small\s[^>]+>\&uarr\;<\/small>$/);
        });

        it('должен вернуть html entity код стрелки &darr обернутый в тег small' +
            'если указана обратная (reverse) сортировка', function() {

                expect(u.table.getSortHeaderLink({ options: { defaultReverse: true } })).to.match(/<small\s[^>]+>\&darr\;<\/small>$/);
            });

        it('должен подставить параметр col в href ссылки', function() {
            expect(u.table.getSortHeaderLink({ col: 'login' })).to.match(/href="[^"]+sort=login[^"]*"/);
        });

        it('если параметр col массив должен подставить значения col в href ссылки через запятую', function() {
            expect(decodeURIComponent(u.table.getSortHeaderLink({ col: ['login', 'login2'] })))
                .to.match(/href="[^"]+sort=login,login2[^"]*"/);
        });

        it('должен подставить параметр caption в текст ссылки', function() {
            expect(u.table.getSortHeaderLink({ caption: 'MyLogin' })).to.match(/^<a\s[^>]+>MyLogin<\/a>/);
        });

        it('должен выставлять по умолчанию reverse=1 в href ссылки', function() {
            expect(u.table.getSortHeaderLink()).to.match(/href="[^"]+reverse=1"/);
        });

        it('должен выставлять reverse=0 в href ссылки, если параметры col и defCol не равны', function() {
            expect(u.table.getSortHeaderLink({ col: 'noLogin', defCol: 'login' })).to.match(/href="[^"]+reverse=0"/);
        });

        it('должен выставлять reverse=1 в href ссылки, если параметры col и defCol равны', function() {
            expect(u.table.getSortHeaderLink({ col: 'login', defCol: 'login' })).to.match(/href="[^"]+reverse=1"/);
        });

        it('должен выставлять значение script в href ссылки', function() {
            var script = 'https://10011.beta3.direct.yandex.ru/registered/main.WAgHxUyM-GwNnAZd.pl';

            expect(u.table.getSortHeaderLink({ script: script })).to.have.string(script);
        });

        it('должен вставлять значения параметров в href ссылки из FORM', function() {
            expect(u.table.getSortHeaderLink({}, {p1: 'v1', p2: 'v2'})).to.match(/href="[^"]+p1=v1&p2=v2\b/);
        });

        ['ncrnd', 'UID', 'ouid_url', 'page'].forEach(function(param) {
            it('не должен вставлять значение параметра ' + param + ' в href ссылки из FORM', function() {
                var form = {},
                    value = param + 'Value';

                form[param] = value;

                expect(u.table.getSortHeaderLink({}, form)).not.to.have.string(value);
            });
        });



        it('должен добавлять в ссылку переданный якорь', function() {
            expect(u.table.getSortHeaderLink({ options: { anchor: 'anc' } })).to.match(/href="[^"]+#anc"/);
        });

        it('должен подставить параметр page из формы в урл ссылки ' +
            'если options.dontRemovePageFromUrl=true', function() {
                expect(u.table.getSortHeaderLink({ options: { dontRemovePageFromUrl: true } }, { page: 'pageValue' })).to.have.string('page=pageValue');
            });

        context('если options.defaultReverse=true', function() {
            it('должен выставлять по умолчанию reverse=0 в href ссылки', function() {
                expect(u.table.getSortHeaderLink({ options: { defaultReverse: true } })).to.match(/href="[^"]+reverse=0"/);
            });

            it('должен выставлять reverse=1 в href ссылки, если параметры col и defCol не равны', function() {
                expect(u.table.getSortHeaderLink({ col: 'noLogin', defCol: 'login', options: { defaultReverse: true } })).to.match(/href="[^"]+reverse=1"/);
            });

            it('должен выставлять reverse=0 в href ссылки, если параметры col и defCol равны', function() {
                expect(u.table.getSortHeaderLink({ col: 'login', defCol: 'login', options: { defaultReverse: true } })).to.match(/href="[^"]+reverse=0"/);
            });
        });

        context('если задан options.urlParam', function() {
            context('если "sort" + options.urlParam присутствует в FROM', function() {
                var sortHeaderLinkParams;

                beforeEach(function() {
                    sortHeaderLinkParams = {
                        col: 'login',
                        defCol: 'noLogin',
                        options: { urlParam: 'urlParamLogin' }
                    };
                });

                it('если значение FORM["sort" + options.urlParam] равно значению col, ' +
                    'должен выставлять "reverse" + FORM["sort" + options.urlParam]=0 в href ссылки', function() {

                    expect(u.table.getSortHeaderLink(
                        sortHeaderLinkParams,
                        {
                            sorturlParamLogin: 'urlParamLogin'
                        })).to.match(/href="[^"]+reverseurlParamLogin=0"/);
                });

                it('если значение FORM["sort" + options.urlParam] не равно значению col, ' +
                    'должен выставлять reverse=1 в href ссылки', function() {

                    expect(u.table.getSortHeaderLink(
                        sortHeaderLinkParams,
                        {
                            sorturlParamLogin: 'login'
                        })).to.match(/href="[^"]+reverseurlParamLogin=1"/);
                });

                context('если options.defaultReverse=true', function() {

                    beforeEach(function() {
                        sortHeaderLinkParams.options.defaultReverse = true;
                    });

                    afterEach(function() {
                        delete sortHeaderLinkParams.options.defaultReverse;
                    });

                    it('если значение FORM["sort" + options.urlParam] равно значению col, ' +
                        'должен выставлять "reverse" + FORM["sort" + options.urlParam]=1 в href ссылки', function() {

                        expect(u.table.getSortHeaderLink(
                            sortHeaderLinkParams,
                            {
                                sorturlParamLogin: 'urlParamLogin'
                            })).to.match(/href="[^"]+reverseurlParamLogin=1"/);
                    });

                    it('если значение FORM["sort" + options.urlParam] не равно значению col, ' +
                        'должен выставлять reverse=0 в href ссылки', function() {

                        expect(u.table.getSortHeaderLink(
                            sortHeaderLinkParams,
                            {
                                sorturlParamLogin: 'login'
                            })).to.match(/href="[^"]+reverseurlParamLogin=0"/);
                    });
                })
            });
        });
    });


    describe('.sortData', function() {
        var sortData,
            form,
            result;

        beforeEach(function() {
            sortData = [
                {
                    "status": { "text": "б" },
                    "ManagerUID": "1",
                    "AgencyUID": 1,
                    "name": "b",
                    "fio": "а"
                },
                {
                    "status": { "text": "аб" },
                    "ManagerUID": "3",
                    "AgencyUID": 11,
                    "name": "b",
                    "fio": "b"
                },
                {
                    "status": { "text": "ав" },
                    "ManagerUID": "2",
                    "AgencyUID": 2,
                    "name": "а",
                    "fio": "b"
                }
            ];
            form = {};
            result = undefined;

        });

        afterEach(function() {
            sortData = [];
            form = {};
            result = undefined;
        });

        it('должен вернуть не измененной data если data не массив', function() {
            expect(u.table.sortData({})).deep.equal({});
        });

        it('должен вернуть не измененной data если form.sort и defaultColumn пустые', function() {
            form.sort = '';

            expect(u.table.sortData(sortData, form, '')).deep.equal(sortData);
        });

        it('при сортировке data должен сравнивать значения как числа если поля не указали как строковые', function() {

            expect(u.table.sortData(sortData.slice(0), form, 'AgencyUID'))
                .deep.equal(sortData.sort(function(a, b) {
                    return a.AgencyUID - b.AgencyUID;
                }));
        });

        it('при сортировке data должен сравнивать значения как строки если поля указали как строковые', function() {
            expect(u.table.sortData(sortData.slice(0), form, 'AgencyUID', ['AgencyUID']))
                .deep.equal(sortData.sort());
        });

        context('если поля для сортировки заданы в поле form.sort', function() {

            it('должен сортировать data только по ManagerUID или AgencyUID если для сортировки передали mauid', function() {
                form.sort = 'mauid';

                result = sortData.sort(function(a, b) {

                    return (a.ManagerUID || a.AgencyUID || 0) - (b.ManagerUID || b.AgencyUID || 0);
                });

                expect(u.table.sortData(sortData, form)).deep.equal(result);
            });

            it('должен сортировать data по вложенным полям (например status.text)', function() {
                form.sort = 'status.text';

                result = sortData.slice(0).sort(function(a, b) {
                    return a.status.text.localeCompare(b.status.text);
                });

                expect(u.table.sortData(sortData, form, '', ['status.text'])).deep.equal(result);
            });

            it('должен сортировать data по нескольким полям если они перечислены в форме через запятую', function() {
                form.sort = 'name,fio';

                result = sortData.slice(0).sort(function(a, b) {
                    return a.name.localeCompare(b.name) || a.fio.localeCompare(b.fio);
                });

                expect(u.table.sortData(sortData, form, '', ['fio', 'name'])).deep.equal(result);
            });

            it('должен сортировать data по нескольким полям в том же порядке в котором они перечислены в form.sort',
                function() {
                    form.sort = 'fio,name';

                    result = sortData.slice(0).sort(function(a, b) {
                        return a.fio.localeCompare(b.fio) || a.name.localeCompare(b.name);
                    });

                    expect(u.table.sortData(sortData, form, '', ['fio', 'name'])).deep.equal(result);
                });
        });

        context('если поля для сортировки перечислены в колонках для сортировки по умолчанию', function() {

            it('должен сортировать data по полю в form.sort если оно не пустое', function() {
                    form.sort = 'name';

                    result = sortData.slice(0).sort(function(a, b) {
                        return a.name.localeCompare(b.name);
                    });

                    expect(u.table.sortData(sortData, form, ['fio'], ['fio', 'name'])).deep.equal(result);
                });

            it('должен сортировать data по полю в указанному в колонке ' +
                'для сортировки по умолчанию если form.sort пустое', function() {
                    form.sort = '';

                    result = sortData.slice(0).sort(function(a, b) {
                        return a.fio.localeCompare(b.fio);
                    });

                    expect(u.table.sortData(sortData, form, ['fio'], ['fio', 'name'])).deep.equal(result);
                });

            it('должен сортировать data по нескольким полям если они перечислены', function() {
                    result = sortData.slice(0).sort(function(a, b) {
                        return a.name.localeCompare(b.name) || a.fio.localeCompare(b.fio);
                    });

                    expect(u.table.sortData(sortData, form, ['name', 'fio'], ['fio', 'name'])).deep.equal(result);
                });



            it('должен сортировать data по нескольким полям в том же порядке в котором они перечислены', function() {
                    result = sortData.slice(0).sort(function(a, b) {
                        return a.fio.localeCompare(b.fio) || a.name.localeCompare(b.name);
                    });

                    expect(u.table.sortData(sortData, form, ['fio', 'name'], ['fio', 'name'])).deep.equal(result);
                });
        });

    });

});
