(function() {

    var MAX_REMAINING_SUM = 999999.99; //максимальная сумма остатка, если max_remaining_sum отсутствует у карты/кошелька

    BEM.MODEL.decl('m-autopay-settings', {
        // readonly режим
        read_only: 'boolean',

        // отвязывать/привязывать карты может только клиент
        allow_bind_unbind: 'boolean',

        // Список привязанных карт
        // @param {Object[]} cards
        //  @param {String} cards.paymethod_id - id привязанной карты/кошелька в балансе
        //  @param {String} cards.number - Номер для отображения на странице
        //  @param {Number} cards.allowed_set_autopay - Может ли пользователь назначить данный вид платежа на кампанию
        //  @param {Number} cards.min_payment_sum - минимальная сумма платежа
        //  @param {Number} cards.max_payment_sum - максимальная сумма платежа
        //  @param {Number} cards.max_remaining_sum - максимальная сумма остатка
        cards: {
            type: 'array',
            default: []
        },

        // Список привязанных аккаунтов ЯД
        // @param {Object[]} yandex_money
        //  @param {String} yandex_money.paymethod_id - id привязанной карты/кошелька в балансе
        //  @param {String} yandex_money.number - Номер для отображения на странице
        //  @param {Number} yandex_money.allowed_set_autopay - Может ли пользователь назначить данный вид платежа на кампанию
        //  @param {Number} yandex_money.min_payment_sum - минимальная сумма платежа
        //  @param {Number} yandex_money.max_payment_sum - максимальная сумма платежа
        //  @param {Number} yandex_money.max_remaining_sum - максимальная сумма остатка
        yandex_money: {
            type: 'array',
            default: []
        },

        // доступные способы оплаты
        available_paymethod_types: {
            type: 'array',
            default: []
        },

        // Тип авто платежа
        autopay_mode: {
            type: 'string',
            preprocess: function(value) {
                if (typeof value == 'boolean') {
                    return value ? 'min_balance' : 'none';
                } else {
                    return value;
                }
            }
        },

        // Тип оплаты: карта или ЯД
        paymethod_type: {
            type: 'enum',
            enum: [
                'card',
                'yandex_money'
            ]
        },

        // Номер кампании Общего счета для автоплатежа
        wallet_cid: {
            type: 'string'
        },

        // Валюта клиента
        currency: {
            type: 'string'
        },

        // id привязанной карты в балансе
        paymethod_id: {
            type: 'string',
            validation: function() {
                return {
                    rules: {
                        required: {
                            text: '', // текст не нужен
                            validate: function(val) {
                                return !!val;
                            }
                        }
                    },
                    needToValidate: function() {
                        return this.get('autopay_mode') == 'min_balance';
                    }
                }
            }
        },

        // объект текущего способа оплаты (карта или ЯД)
        // от способа оплаты зависит максимальная/минимальная сумма платежа
        paymethod_object: {
            type: 'object',
            dependsFrom: 'paymethod_id',
            calculate: function() {
                var paymethod,
                    paymethodId = this.get('paymethod_id');

                // ищем текущий объект способ оплаты по id
                paymethod = u._.find(
                    this.get('paymethod_type') == 'yandex_money' ?
                        this.get('yandex_money') :
                        this.get('cards'),
                    function(item) {
                        return item.paymethod_id == paymethodId;
                    });

                return paymethod || {};
            }
        },

        // максимальная сумма остатка
        max_remaining: {
            internal: true,
            type: 'number',
            dependsFrom: 'paymethod_object',
            calculate: function() {
                return this.get('paymethod_object').max_remaining_sum || MAX_REMAINING_SUM;
            }
        },

        // максимальная сумма платежа
        max_payment: {
            internal: true,
            type: 'number',
            dependsFrom: 'paymethod_object',
            calculate: function() {
                return this.get('paymethod_object').max_payment_sum || +Infinity;
            }
        },

        // минимальная сумма платежа
        min_payment: {
            internal: true,
            type: 'number',
            dependsFrom: 'paymethod_object',
            calculate: function() {
                return this.get('paymethod_object').min_payment_sum || 0;
            }
        },

        // несгораемая сумма
        remaining_sum: {
            type: 'blank-number',
            validation: function() {
                return {
                    rules: {
                        required: {
                            text: iget2('m-autopay-settings', 'ne-ukazana-summa-ostatka', 'Не указана сумма остатка'),
                            validate: function(value) {
                                return value === 0 || !!value;
                            }
                        },
                        gte: {
                            value: this.get('max_remaining'),
                            text: iget2('m-autopay-settings', 'maksimalnaya-summa-ostatka-s', 'Максимальная сумма остатка {foo}', {
                                foo: u.currencies.formatSum(this.get('currency'), this.get('max_remaining'))
                            }),
                            needToValidate: function(value) {
                                return !!value;
                            }
                        }
                    },
                    needToValidate: function() {
                        return !!(this.get('paymethod_id') && this.get('autopay_mode') == 'min_balance');
                    }
                };
            }
        },

        // сумма платежа
        payment_sum: {
            type: 'blank-number',
            validation: function() {
                return {
                    rules: {
                        required: {
                            text: iget2('m-autopay-settings', 'ne-ukazana-summa-popolneniya', 'Не указана сумма пополнения'),
                            validate: function(value) {
                                return !!value;
                            }
                        },
                        lte: {
                            value: this.get('min_payment'),
                            text: iget2('m-autopay-settings', 'summa-popolneniya-ne-mozhet', 'Сумма пополнения не может быть меньше {foo}', {
                                foo: u.currencies.formatSum(this.get('currency'), this.get('min_payment'))
                            }),
                            needToValidate: function(value) {
                                return !!value;
                            }
                        },
                        gte: {
                            value: this.get('max_payment'),
                            text: iget2('m-autopay-settings', 'summa-popolneniya-ne-mozhet-202', 'Сумма пополнения не может быть больше {foo}', {
                                foo: u.currencies.formatSum(this.get('currency'), this.get('max_payment'))
                            }),
                            needToValidate: function(value) {
                                return !!value;
                            }
                        }
                    },
                    needToValidate: function() {
                        return !!(this.get('paymethod_id') && this.get('autopay_mode') == 'min_balance');
                    }
                };
            }
        },

        // сумма платежа с НДС
        payment_sum_nds: {
            type: 'string',
            dependsFrom: ['payment_sum', 'client_nds'],
            calculate: function() {
                var value = (+this.get('payment_sum') || 0) * (100 + this.get('client_nds')) / 100;

                return u.numberFormatter.roundFormatted(value);
            }
        },

        // процент НДС
        client_nds: {
            type: 'number',
            default: 0
        },

        disabledModeCheckbox: 'string',

        // состояние disabled контролов режима «Неснижаемый остаток»
        disabledAutopaySettings: {
            type: 'string',
            internal: true,
            dependsFrom: 'autopay_mode',
            calculate: function() {
                return this.get('autopay_mode') == 'none' ? 'yes' : '';
            }
        },

        // состояние disabled контролов ЯД кошелька
        disabledYandexMoneys: {
            type: 'string',
            internal: true,
            dependsFrom: ['autopay_mode', 'paymethod_type'],
            calculate: function() {
                if (this.get('autopay_mode') == 'none') {
                    return 'yes';
                } else if (!this.get('yandex_money').length) {
                    return '';
                } else {
                    return this.get('paymethod_type') == 'card' ? 'yes' : ''
                }
            }
        },

        // состояние disabled контролов банковских карт
        disabledCards: {
            type: 'string',
            internal: true,
            dependsFrom: ['autopay_mode', 'paymethod_type'],
            calculate: function() {
                if (this.get('autopay_mode') == 'none') {
                    return 'yes';
                } else {
                    return this.get('paymethod_type') == 'yandex_money' ? 'yes' : ''
                }
            }
        },

        // id плательщика
        person_id: {
            type: 'string'
        },

        // тип плательщика - юридическое или физическое лицо
        payerType: {
            type: 'enum',
            enum: [
                'natural',
                'legal'
            ]
        },

        disabledPayer: {
            type: 'string',
            internal: true,
            dependsFrom: ['autopay_mode'],
            calculate: function() {
                if (this.get('autopay_mode') == 'none') {
                    return 'yes';
                } else {
                    return '';
                }
            }
        }

    }, {

        value: function() {
            var mode = this.get('autopay_mode'),
                isLegal = this.get('payerType') === 'legal';

            if (mode == 'min_balance') {
                return {
                    wallet_cid: this.get('wallet_cid'),
                    autopay_mode: mode,
                    payment_sum: this.get('payment_sum'),
                    paymethod_id: this.get('paymethod_id'),
                    remaining_sum: this.get('remaining_sum'),
                    paymethod_type: this.get('paymethod_type'),
                    ur_person: isLegal ? 1 : 0,
                    person_id: isLegal ? this.get('person_id') : undefined
                };
            } else if (mode == 'none') {
                return {
                    wallet_cid: this.get('wallet_cid'),
                    autopay_mode: mode
                }
            }
        }

    });

})();
