describe('m-banner', function() {
    var bannerData,
        bannerModelData,
        constStub,
        sandbox;

    beforeEach(function() {
        bannerData = u.bannerData({
            banner: {
                modelId: 'banner-id',
                title: 'title',
                body: 'body',
                href: 'www.href.ru',
                domain: 'www.href.ru',
                vcard: { 'city': 'bla' },
                has_site_enum: 'yes',
                sitelinks: [
                    {
                        href: "ya.ru?q=1",
                        url_protocol: "http: //",
                        title: 'title',
                        description: 'description'
                    },
                    {
                        href: "ya.ru?q=2",
                        url_protocol: "http: //",
                        title: 'title2',
                        description: 'description2'
                    }
                ]
            }
        });

        bannerModelData = u.bannerModelData({ banner: bannerData });
        sandbox = sinon.sandbox.create({
            useFakeTimers: true
        });
        constStub = sandbox.stub(u, 'consts');

        constStub.withArgs('NEW_MAX_TITLE_LENGTH').returns(35);
        constStub.withArgs('MAX_BODY_LENGTH').returns(81);
        constStub.withArgs('MAX_NUMBER_OF_NARROW_CHARACTERS').returns(15);
        constStub.withArgs('NARROW_SYMBOLS').returns('.,!:;"');
        constStub.withArgs('ALLOW_BANNER_LETTERS').returns("abcdefghijklmonpqrstuvwxyzABCDEFGHIJKLMONPQRSTUVWXYZабвгдеёжзийклмнопрстуфхцчшщъыьэюяАБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯЄІЇєіїҐґ'ҰҒғӨҢҚҺҮұөңқһүҖҲҳҗӘәÄäßẞğĞşŞçÇıİ̇âÂîÎûÛéüÜöÖ\[\]0123456789.-+,. \"!?\()%$€;:/&#38;'*_=#№«» –—−™®©’°⁰¹²³⁴⁵⁶⁷⁸⁹₽");
        constStub.withArgs('MAX_BODY_UNINTERRUPTED_LENGTH').returns(23);
        constStub.withArgs('MAX_TITLE_UNINTERRUPTED_LENGTH').returns(22);
        constStub.withArgs('rights').returns({ ajaxAdGroupsMultiSave: false });
    });

    afterEach(function() {
        sandbox.restore();
    });

    describe('Валидация: ', function() {
        var banner;

        beforeEach(function() {
            banner = BEM.MODEL.create({ name: 'm-banner', id: 'banner-id' }, bannerModelData);
        });

        afterEach(function() {
            banner.destruct();
        });

        function checkUnapplicableForImageAd(field, value) {
            it('Для ГО не проверяется', function() {
                banner.set('ad_type', 'image_ad');

                var spy = sandbox.spy(banner, 'validate');
                banner.set(field, value);

                expect(banner.validate.called).to.be.false;
            });
        }

        function expectFieldToBeValid(field) {
            expect(banner.validate(field).valid).to.be.true;
        }

        ['title', 'body', 'display_href'].forEach(function(value) {

            var valueText = {
                title: 'Заголовок',
                body: 'Текст',
                display_href: 'Отображаемая ссылка'
            }[value],
                lengthConst = {
                    title: u.consts('MAX_TITLE_LENGTH'),
                    body: +u.consts('MAX_BODY_LENGTH'),
                    display_href: 20
                }[value],
                wordLengthConst = {
                    title: u.consts('MAX_TITLE_UNINTERRUPTED_LENGTH'),
                    body: u.consts('MAX_BODY_UNINTERRUPTED_LENGTH')
                }[value];

            describe(valueText + ' баннера', function() {

                checkUnapplicableForImageAd(value, 'someText');

                describe('Для текстового баннера: ', function() {
                    beforeEach(function() {
                        banner.set('ad_type', 'text');
                    });

                    it('при пустом поле "' + valueText + '" баннер ' + value == 'display_href' ? 'валидный' : 'невалидный', function() {
                        banner.set(value, null);

                        if(value == 'display_href') {
                            expectFieldToBeValid(value);
                        } else {
                            expect(banner.validate(value).errors[0].text.indexOf('Необходимо указать ')).not.to.equal(-1);
                        }
                    });

                    it('при длине поля "' + valueText + '" меньше ' + lengthConst + ' баннер валидный', function() {
                        banner.set(value, new Array(20).join('a'));

                        expectFieldToBeValid(value);
                    });

                    it('при длине поля "' + valueText + '" больше ' + lengthConst + ' баннер невалидный', function() {
                        banner.set(value, new Array(120).join('a'));

                        expect(banner.validate(value).errors[0].text.length).to.be.above(0);
                    });

                    it('Значки "#" внутри поля "' + valueText + '" игнорируются', function() {
                        var extra = lengthConst - 10,
                            extrapart = extra > u.consts('MAX_BODY_UNINTERRUPTED_LENGTH') ?
                                new Array(Math.floor(extra / 2)).join('b ') :
                                new Array(extra - 2).join('b'),
                            expression = new Array(10).join('a') + ' #c# ' + extrapart;
                        banner.set(value, expression);

                        expect(expression.length).to.be.above(lengthConst);
                        expectFieldToBeValid(value);
                    });

                    if (value != 'display_href') {
                        it('При длине слова в поле ' + valueText + ' больше ' + wordLengthConst + ' символов баннер невалидный', function() {
                            banner.set(value, new Array(wordLengthConst + 2).join('a'));

                            expect(banner.get(value).length).to.be.above(wordLengthConst);
                            expect(banner.validate(value).errors[0].text.length).to.be.above(0);
                        });
                    }
                });
            });
        });

        describe('Модель ссылки на сайт', function() {

            checkUnapplicableForImageAd('href_model', {});

            it('При установленной визитке ссылка не проверяется', function() {
                banner.set('ad_type', 'text');
                banner.set('has_vcard', true);

                var spy = sandbox.spy(banner, 'validate');
                banner.set('href_model', {});

                expect(banner.validate.called).to.be.false;
            });

            it('При непустой визитке ссылка не проверяется', function() {
                banner.set('ad_type', 'text');
                banner.set('has_vcard', true);
                banner.set('isVCardEmpty', false);

                var spy = sandbox.spy(banner, 'validate');
                banner.set('href_model', {});

                expect(banner.validate.called).to.be.false;
            });

            it('При установленной отображаемой ссылке ссылка не проверяется', function() {
                banner.set('ad_type', 'text');
                banner.set('has_vcard', false);
                banner.set('isVCardEmpty', true);
                banner.set('display_href', 'vk.com');

                var spy = sandbox.spy(banner, 'validate');
                banner.set('href_model', {});

                expect(banner.validate.called).to.be.false;
            });

            it('Пустая ссылка невалидна', function() {
                banner.set('ad_type', 'text');
                banner.set('has_vcard', false);
                banner.set('isVCardEmpty', true);
                banner.set('display_href', '');
                banner.set('href_model', BEM.MODEL.create({ name: 'm-banner-href', id: 'banner-href-id' }));

                expect(banner.validate('href_model').errors[0].text).to.equal('Не введена ссылка на сайт');
            });

            it('Введенная ссылка проходит валидацию', function() {
                banner.set('ad_type', 'text');
                banner.set('has_vcard', false);
                banner.set('isVCardEmpty', true);
                banner.set('display_href', '');
                banner.set('href_model', BEM.MODEL.create({ name: 'm-banner-href', id: 'banner-href-id' }).set('href', 'vk.com'));

                expectFieldToBeValid('href_model');
            });
            //TODO: тесты на m-banner-href
        });

        describe('Быстрые ссылки', function() {

            checkUnapplicableForImageAd('sitelinks', {});

            it('При пустых быстрых ссылках может быть не заполнена ссылка на сайт', function() {
                banner.set('ad_type', 'text');

                banner.set('sitelinks', {});
                banner.set('href_model', {});

                expectFieldToBeValid('sitelinks');
            });

            it('При пустых быстрых ссылках может быть заполнена ссылка на сайт', function() {
                banner.set('ad_type', 'text');
                banner.set('sitelinks', {});
                banner.set('href_model', BEM.MODEL.create({ name: 'm-banner-href', id: 'banner-href-id' }).set('href', 'vk.com'));

                expectFieldToBeValid('sitelinks');
            });

            it('При непустых быстрых ссылках не может быть пустой ссылка на сайт', function() {
                banner.set('ad_type', 'text');
                banner.set('sitelinks', BEM.MODEL.create({ name: 'm-banner-sitelinks', id: 'banner-sitelinks-id' }).set('href0', 'vk.com'));
                banner.get('href_model').set('href', '');

                expect(banner.validate('sitelinks').errors[0].text).to.equal('Если заполнены быстрые ссылки, то должна быть заполнена основная ссылка на сайт');
            });

            it('При заполненных быстрых ссылках и ссылке на сайт баннер валидный', function() {
                banner.set('ad_type', 'text');
                banner.set('sitelinks', BEM.MODEL.create({ name: 'm-banner-sitelinks', id: 'banner-sitelinks-id' }).set('href0', 'vk.com'));
                banner.get('href_model').set('href', 'example.com');

                expectFieldToBeValid('sitelinks');
            })
        });

        describe('Модель изображения для ТГО', function() {

            checkUnapplicableForImageAd('image_model', {});

            it('Если картинка еще обрабатывается, валидация дает ошибку', function() {
                banner.set('ad_type', 'text');

                banner.set('image_model', BEM.MODEL.create({ name: 'm-banner-pic', id: 'banner-pic-id' }).set('image_processing_state', 'pending'));

                expect(banner.validate('image_model').errors[0].text).to.equal('Необходимо дождаться окончания обработки изображения');
            });

            it('Если картинка обрабаталась, баннер валидный', function() {
                banner.set('ad_type', 'text');
                banner.set('image_model', BEM.MODEL.create({ name: 'm-banner-pic', id: 'banner-pic-id' }).set('image_processing_state', ''));

                expectFieldToBeValid('image_model');
            });
        });

        describe('Модель изображения для ГО', function() {
            it('Для ТГО не проверяется', function() {
                banner.set('ad_type', 'text');

                var spy = sandbox.spy(banner, 'validate');
                banner.set('image_ad', {});

                expect(banner.validate.called).to.be.false;
            });

            it('При незагруженных изображениях и креативах баннер невалидный', function() {
                banner.set('ad_type', 'image_ad');

                banner.set('image_ad', null);
                banner.set('creative', null);

                expect(banner.validate('image_ad').errors[0].text).to.equal('Необходимо загрузить креатив');
            });

            it('При загруженном изображении баннер валидный', function() {
                banner.set('ad_type', 'image_ad');
                banner.set('image_ad', BEM.MODEL.create({ name: 'dm-image-ad', id: 'banner-image-ad' }).set('hash', 'aaa'));

                expectFieldToBeValid('image_ad');
            });

            it('При загруженном креативе баннер валидный', function() {
                banner.set('ad_type', 'image_ad');

                banner.set('image_ad', null);
                banner.set('creative', BEM.MODEL.create({ name: 'dm-creative', id: 'dm-creative-id' }).set('creative_id', 'bbb'));

                expectFieldToBeValid('image_ad');
            });

            // В интерфейсе нет возможности поставить и изображение, и креатив
            // на сервере есть отдельная валидация, что не выставлено и изображение, и креатив
            // то есть 2 изображений быть не может, однако валидация это допускает.
            it('При загруженных обоих изображениях баннер валидный', function() {
                banner.set('ad_type', 'image_ad');

                banner.set('image_ad', BEM.MODEL.create({ name: 'dm-image-ad', id: 'banner-image-ad' }).set('hash', 'aaa'));
                banner.set('creative', BEM.MODEL.create({ name: 'dm-creative', id: 'dm-creative-id' }).set('creative_id', 'bbb'));

                expectFieldToBeValid('image_ad');
            });
        });
    });

    describe('Зависимости полей: ', function() {
        var banner;

        beforeEach(function() {
            banner = BEM.MODEL.create({ name: 'm-banner', id: 'banner-id' }, bannerModelData);
        });

        afterEach(function() {
            banner.destruct();
        });

        describe('От поля href_model и связанных', function() {
            beforeEach(function() {
                banner.set('href_model', BEM.MODEL.create({ name: 'm-banner-href', id: 'banner-href-id' }).set('href', 'vk.com'));
            });

            it('Поле display_domain пправильно ставит значение', function() {
                expect(banner.get('display_domain')).to.equal('vk.com');
            });

            it('При установленном href_model поле has_display_domain возвращает true', function() {
                expect(banner.get('has_display_domain')).to.be.true;
            });

            it('При установленном href_model поле has_href возвращает 1', function() {
                expect(banner.get('has_href')).to.equal(1);
            });

            it('При неустановленном href_model поле has_display_domain возвращает false', function() {
                banner.get('href_model').set('href', '');
                expect(banner.get('has_display_domain')).to.be.false;
            });

            it('При неустановленном href_model поле has_href возвращает 0', function() {
                banner.get('href_model').set('href', '');
                expect(banner.get('has_href')).to.equal(0);
            });
        });

        describe('Поля has_vcard', function() {
            it('При пустой и закрытой визитке has_vcard равно 0', function() {
                banner.set('isVCardEmpty', true);
                banner.set('is_vcard_open', false);

                expect(banner.get('has_vcard')).to.equal(0);
            });

            it('При пустой и открытой визитке has_vcard равно 0', function() {
                banner.set('isVCardEmpty', true);
                banner.set('is_vcard_open', true);

                expect(banner.get('has_vcard')).to.equal(0);
            });

            it('При непустой и закрытой визитке has_vcard равно 0', function() {
               banner.set('isVCardEmpty', false);
               banner.set('is_vcard_open', false);

               expect(banner.get('has_vcard')).to.equal(0);
            });

            it('При непустой и открытой визитке has_vcard равно 1', function() {
                banner.set('isVCardEmpty', false);
                banner.set('is_vcard_open', true);

                expect(banner.get('has_vcard')).to.equal(1);
            });

            it('При наличии визитки и отсутствии href_model.href has_href возвращает 0', function() {
                banner.set('isVCardEmpty', false);
                banner.set('is_vcard_open', true);
                banner.get('href_model').set('href', '');

                expect(banner.get('has_href')).to.equal(0);
            });

            it('При отсутствии визитки и отсутствии href_model.href has_href возвращает 1', function() {
                banner.set('isVCardEmpty', true);
                banner.set('is_vcard_open', true);
                banner.get('href_model').set('href', '');

                expect(banner.get('has_href')).to.equal(1);
            })
        });
    });

    describe('Формирование хэша данных: ', function() {
        var data,
            banner = BEM.MODEL.create({ name: 'm-banner', id: 'banner-id' }, bannerModelData);

        describe('Визитка', function() {
            it('При неустановленной визитке в выдаче нет полей vcard и vcard_id', function() {
                banner.set('isVCardEmpty', true);
                banner.set('is_vcard_open', false);

                data = banner.provideData();

                expect('vcard' in data).to.be.false;
                expect('vcard_id' in data).to.be.false;
            });

            it('При установленной визитке в выдаче есть поля vcard и vcard_id', function() {
                banner.set('isVCardEmpty', false);
                banner.set('is_vcard_open', true);

                data = banner.provideData();

                expect('vcard' in data).to.be.true;
                expect('vcard_id' in data).to.be.true;
            });
        });

        describe('Поля из модели ссылки перенесены в выдачу:', function() {
            banner.set('href_model', BEM.MODEL.create({ name: 'm-banner-href', id: 'banner-href-id' }).set('href', 'vk.com'));

            var hrefObject = banner.get('href_model').toJSON();
            data = banner.provideData();

            Object.keys(hrefObject).forEach(function(key) {
                it('Поле '+ key + ' есть в выдаче', function() {
                    expect(key in data).to.be.true;
                })
            });
        });

        describe('Поля из модели изображения перенесены в выдачу:', function() {
            banner.set('image_model', BEM.MODEL.create({ name: 'm-banner-pic', id: 'banner-pic-id' }).set('image_processing_state', ''));

            var imageObject = banner.get('image_model').toJSON();
            data = banner.provideData();

            Object.keys(imageObject).forEach(function(key) {
                it('Поле '+ key + ' есть в выдаче', function() {
                    expect(key in data).to.be.true;
                })
            });
        });

        describe('Наличие поля age', function() {
            it('Если возвраст установлен, то поле есть в выдаче', function() {
                banner.set('age', 10);

                data = banner.provideData();

                expect('age' in data).to.be.true;
            });

            it('Если возвраст не установлен, то поля нет в выдаче', function() {
                banner.set('ageInstalled', false);

                data = banner.provideData();

                expect('age' in data).to.be.false;
            })
        });

        describe('Прочие поля', function() {
            it('Если нет ссылки, то нет и отображаемого домена', function() {
                banner.get('href_model').set('href', '');

                data = banner.provideData();

                expect('domain' in data).to.be.false;
            });

            it('Если баннер графический, то объявление десктопное', function() {
                banner.set('ad_type', 'image_ad');

                data = banner.provideData();

                expect(data.banner_type).to.equal('desktop');
            })
        });

        describe('Поле turbolanding', function() {
            var turbolandingModel = BEM.MODEL.create({ name: 'dm-turbo-landing' }, {
                id: '1',
                name: 'name',
                href: 'http://qwe.ru'
            });

            it('Если нет turbolanding, то нет поля', function() {
                data = banner.provideData();

                expect('turbolanding' in data).to.be.false;
            });

            it('Если задан turbolanding, то поле есть', function() {
                banner.set('turbolanding', turbolandingModel);
                banner.get('href_model').set('href', 'https://ya.ru');
                data = banner.provideData();

                expect(data.turbolanding.id).to.be.equal('1');
            });

            it('Если задан turbolanding но нет ссылки у объявления, то нет поля', function() {
                banner.set('turbolanding', turbolandingModel);
                banner.get('href_model').set('href', '');

                data = banner.provideData();

                expect('turbolanding' in data).to.be.false;
            });
        });

        banner.destruct();
    });

    describe('Данные для отрисовки ошибок', function() {
        var banner,
            errorsList;

        banner = BEM.MODEL.create({ name: 'm-banner', id: 'banner-id' }, bannerModelData);
        errorsList = banner.toJSONForErrors();

        ['isNewBanner', 'newBannerIndex', 'bid', 'modelId'].forEach(function(name) {
            it('Поле ' + name + 'есть в списке полей с ошибками', function() {
                expect(name in errorsList).to.be.true;
            })
        }, this);


    });

    describe('Внешний API', function() {
        var banner = BEM.MODEL.create({ name: 'm-banner', id: 'banner-id' }, bannerModelData),
            data = banner.toJSONForCopy();

        it('Метод getVCardData возвращает данные визитки', function() {
            banner.set('vcard', { vcard_id: 'blabla' });
            var vcard = banner.get('vcard');

            expect(banner.getVCardData()).to.deep.equal(vcard);
        });

        describe('Коллауты', function() {
            banner.set('callouts', [{ additions_item_id: 1 }, { additions_item_id: 2 }]);

            it('При существуеющем флаге isCalloutInUse возвращет true', function() {
                expect(banner.isCalloutInUse(1)).to.be.true;
            });

            it('При несуществуеющем флаге isCalloutInUse возвращет false', function() {
                expect(banner.isCalloutInUse(4)).to.be.false;
            });

            it('deleteCalloutById удаляет флаг', function() {
                banner.deleteCalloutById(1);

                expect(banner.isCalloutInUse(1)).to.be.false;
            });
        });

        describe('toJSONForCopy работает корректно', function() {

            banner.PARAMS_FOR_COPY.text.map(function(name) {
                it('toJSONForCopy возвращает поле ' + name, function() {
                    expect(data).to.have.property(name);
                });
            });

            Object.keys(banner.fieldsDecl).filter(function(key) {
                return banner.PARAMS_FOR_COPY.text.indexOf(key) == -1;
            }).map(function(name) {
                it('toJSONForCopy не возвращает поле ' + name, function() {
                    expect(data).not.to.have.property(name);
                });
            });

        });

        banner.destruct();
    });
});
