BEM.MODEL.decl({ name: 'm-money-transfer-to', baseModel: 'm-money-transfer' }, {
    /**
     * Кол-во кампаний с флагом isEqualReceive: true
     * нужно для расчета равной суммы
     */
    numOfEqualReceivingCampaigns: {
        type: 'number',
        default: 0,
        dependsFrom: ['campaigns'],
        calculate: function() {
            return this.getCampaignsWithEqualReceive().length
        }
    },

    /**
     * Кол-во кампаний с которых выполняется перенос
     * нужно для включения/ выключения возможности равного переноса
     */
    numOfTransferringCamps: {
        type: 'number',
        default: 0
    },

    /**
     * Сумма к переносу
     */
    receivingSum: {
        type: 'number',
        default: 0
    },

    /**
     * Список кампаний
     */
    campaigns: {
        type: 'models-list',
        modelName: 'm-money-transfer-campaign-to'
    }
}, {
    /**
     * Сбрасывает кампании campaigns, если campaigns не указан сбрасывает все кампании списка
     * @param {BEM.MODEL[]} campaigns массив моделей 'm-money-transfer-campaign'
     */
    resetCampaigns: function(campaigns) {
        (campaigns || this.get('campaigns')).forEach(function(camp) {
            camp.update({
                amount: 0,
                isEqualReceive: false,
                isReceiving: false
            })
        })
    },

    /**
     * Возвращает массив кампаний на которые выполняется перенос
     * @returns {Array} массив моделей 'm-money-transfer-campaign'
     */
    getReceivingCampaigns: function() {
        return this.get('campaigns').where({ isReceiving: true });
    },

    /**
     * Распределяет сумму receivingSum поровну по кампаниям campaigns.
     * Если campaigns не задан, распределяет по все кампания без флага isDisabled
     * @param {BEM.MODEL[]} campaigns массив моделей кампании типа m-money-transfer-campaign
     */
    allocateSumEqually: function(campaigns) {
        var camps = campaigns || this.getNotDisabledCampaigns(),
            calculation = this._calculateEqualSum(camps);

        camps.forEach(function(camp, i) {
            camp.update({
                amount: i < calculation.numOfRemains ? calculation.sumWithRemain : calculation.equalSum,
                isEqualReceive: true
            });
        }, this);
    },

    /**
     * Распределяет сумму receivingSum поровну по кампаниям campaigns.
     * Если campaigns не задан, распределяет по всем кампания без флага isDisabled
     * @param {BEM.MODEL[]} camps массив моделей кампании типа m-money-transfer-campaign
     * @private
     * @returns {Object} объект расчета
     * {
     *     equalSum - сумма переноса,
     *     numOfRemains - кол-во остатков от деления,
     *     sumWithRemain - величина остатка
     * }.
     */
    _calculateEqualSum: function(camps) {

        this._calculateEqualSum.cache = this._calculateEqualSum.cache || {};

        var campaignsLng = camps.length,
            transferValue = this.get('receivingSum'),
            key = campaignsLng + '_' + transferValue;

        if (!this._calculateEqualSum.cache[key]) {
            var equalSum = u.numberFormatter.round(transferValue / campaignsLng, { roundType: 'floor' });

            this._calculateEqualSum.cache[key] = {
                equalSum: equalSum,
                numOfRemains: Math.round((transferValue - equalSum * campaignsLng) * 100),
                sumWithRemain: parseFloat((equalSum + 0.01).toFixed(2))
            }
        }

        return this._calculateEqualSum.cache[key];
    },

    /**
     * Возвращает массив кампаний на которое выполняется перенос средств в равном размере
     * @returns {BEM.MODEL[]} массив моделей 'm-money-transfer-campaign'
     */
    getCampaignsWithEqualReceive: function() {
        return this.get('campaigns').where({ isEqualReceive: true });
    },

    /**
     * Устанавливает флаг isEqualReceive: true кампании campaign
     * @param {BEM.MODEL} campaign модель кампании типа m-money-transfer-campaign
     * @param {Boolean} flag флаг-значение
     */
    setOneCampEqualReceive: function(campaign, flag) {
        campaign.set('isEqualReceive', flag);
    },

    /**
     * Устанавливает флаг isEqualReceive: true кампаний campaigns
     * @param {BEM.MODEL[]} campaigns массив моделей кампаний типа m-money-transfer-campaign
     * @param {Boolean} flag флаг-значение
     */
    setAllCampsEqualReceive: function(campaigns, flag) {
        campaigns.forEach(function(camp) {
            camp.set('isEqualReceiveMode', flag)
        });
    },

    /**
     * Устанавливает флаг isReceiving: true кампании campaign, если кампании не isDisabled
     * @param {BEM.MODEL} campaign модель кампании типа m-money-transfer-campaign
     */
    selectOneCamp: function(campaign) {
        !campaign.get('isDisabled') && campaign.set('isReceiving', true);
    }
});
