/**
 * Событие первоначального обновления VM.
 * Триггерится после обновления VM
 * @event sync:init
 */

/**
 * Событие успешной синхронизации в DM.
 * Триггерится после обновления DM
 * @event sync:ok
 * @type {object}
 */

/**
 * Событие успешной синхронизации из DM.
 * Триггерится после обновления VM
 * @event sync:from
 */

/**
 * Событие ошибки при синхронизации в DM.
 * Триггерится при ошибке обновления DM данными VM
 * @event sync:error
 * @type {object}
 */

BEM.MODEL.decl({ model: 'vm-auto-sync-from-dms', baseModel: 'vm-sync-dms' }, {

}, {
    /**
     * Находит DM с которым синхронизируется VM
     * Наполняет данными VM
     * @fires sync:init
     */
    init: function() {
        if (this._inited) return false;

        var dms = this.getDM();

        Object.keys(dms, function(name) {
            if (typeof dms[name] === 'undefined') {
                throw new Error('model ' + this.name + 'cant find DM ' + JSON.stringify(this.get('_dmDecl')));
            }
        }, this);

        this.syncFromDM({ inited: true });

        this.initAutoSync();

        this.trigger('sync:init');

        this._inited = true;
    },

    /**
     * Возвращает объект зависимостей полей view-модели от data-моделей, в виде
     * {
         dmKey1: {
             dmField1: ['vmField1'],
             dmField2: ['vmField2']
         },
         dmKey2: {
             dmField1: ['vmField1', 'vmField2', 'vmField3'],
             dmField2: ['vmField3'],
             dmField3: ['vmField5']
         }
     }
     * @returns {Object}
     * @private
     */
    _dependsFromDMs: function() {
        return {};
    },

    /**
     * Инициализирует автоматическую синхронизацию dm -> vm
     */
    initAutoSync: function() {
        this.syncFromDM = $.debounce(this.syncFromDM, 50);

        Object.keys(this._dependsFromDMs()).forEach(this._listenDMfields, this);

        this.on('destruct', this._onDestruct, this);
    },

    _isDestructed: null,

    _onDestruct: function() {
        this.un('destruct', this._onDestruct, this);
        this._isDestructed = true;

        Object.keys(this._dependsFromDMs()).forEach(this._stopListenDMfields, this);
    },

    /**
     * Подписываемся на изменение полей data-модели,
     * при изменении - обновляем поля view-модели, описанные в `_dependsFromDMs`
     * @param {String} modelKey ключ data-модели
     * @returns {BEM.MODEL}
     * @private
     */
    _listenDMfields: function(modelKey) {
        var dependsFromDM = this._dependsFromDMs()[modelKey],
            modelFields = Object.keys(dependsFromDM),
            model = this.getDM()[modelKey],
            listenArgs;

        if (model) {
            listenArgs = [
                modelFields.join(' '),
                'change',
                function(e, data) {
                    this
                        ._addFieldsToSync(dependsFromDM[data.field])
                        .syncFromDM();
                },
                this
            ];

            var cacheListens = this._cacheListens || (this._cacheListens = {});

            cacheListens[modelKey] = { model: model, args: listenArgs };

            model.on.apply(model, listenArgs);
        }

        return this;
    },

    /**
     * Отписываемся от изменений полей data-модели
     * @param {String} modelKey ключ data-модели
     * @returns {BEM.MODEL}
     * @private
     */
    _stopListenDMfields: function(modelKey) {
        var listensByModelKey = (this._cacheListens || {})[modelKey];

        listensByModelKey && listensByModelKey.model.un.apply(listensByModelKey.model, listensByModelKey.args);

        this._cacheListens[modelKey] = null;

        return this;
    },

    /**
     * Получает данные из DM
     * Отличается от базового метода тем, что в data находятся модели вместо объектов
     * @param {String[]} [required] список необходимых полей
     * @returns {Object} - обработанные данные из DM
     * @private
     */
    _getFromDM: function(required) {
        var dms = this.getDM();

        return this.prepareDataFromDM(dms, required);
    },

    /**
     * Подготавливает данные для обновления VM
     * Обрабатывает данные DM перед обновлением VM
     * @param {Object} data необработанные данные из DM
     * @param {String[]} [required] список необходимых полей, если параметр задан - вернется объект только с указанными
     * полями, иначе - объект со всеми данными
     * @returns {Object} - обработанные данные модели
     */
    prepareDataFromDM: function(data, required) {
        return data;
    },

    _fieldsToSync: null,

    /**
     * Добавляет поля в список полей, которые необходимо обновить
     * @param {String[]} fields
     * @returns {BEM.DOM}
     * @private
     */
    _addFieldsToSync: function(fields) {
        this._fieldsToSync = fields.reduce(function(fieldsForSync, field) {
            fieldsForSync.indexOf(field) === -1 && fieldsForSync.push(field);

            return fieldsForSync;
        }, this._fieldsToSync || []);

        return this;
    },

    /**
     * Ресет из DM
     * @param {Object} [options] Набор параметров
     * @param {Boolean} [options.inited] Флаг говорящий о том, что вызов произошел при инициализации модели
     * @fires sync:from
     */
    syncFromDM: function(options) {
        // Если debounce решит вызваться после того как мы удалим модель
        // ничего такого не случится :)
        if (this._isDestructed) return;
        this.update(this._getFromDM(this._fieldsToSync), (options || {}).inited && { inited: true });

        this._fieldsToSync = null;

        this.trigger('sync:from');
    },

    /**
     * Возвращает функцию необходимости полей (по сути !!~indexOf)
     * @param {String[]} [required]
     * @returns {Function}
     */
    getIsNeedFunction: function(required) {
        var requiredHash = required && u._.indexBy(required);

        return required ?
            function(key) { return requiredHash[key] } :
            function() { return true };
    },

    /**
     * Подменяет одну из data-моделей на новую
     * @param {String} modelKey ключ модели
     * @param {BEM.DOM} dataModel новая модель
     */
    changeDM: function(modelKey, dataModel) {
        var _dmDecl = this.get('_dmDecl'),
            dependsFromOneDM = this._dependsFromDMs()[modelKey];

        // отписываемся от изменений которые слушались на старой модели
        this._stopListenDMfields(modelKey);

        // меняем data-модель на новую
        _dmDecl[modelKey] = {
            name: dataModel.name,
            path: dataModel.path()
        };

        this.set('_dmDecl', _dmDecl);

        // подписываемся на изменения новой модели
        this._listenDMfields(modelKey);

        // Добавляем в список полей, которые нужно обновить, все поля связанные с новой моделью
        Object.keys(dependsFromOneDM).forEach(function(field) {
            this._addFieldsToSync(dependsFromOneDM[field]);
        }, this);

        // Обновляем view-модель
        this.syncFromDM();
    }

});
