BEM.MODEL.decl({ name: 'vm-feed', baseModel: 'vm-sync-dm' }, {

    // идентификатор фида
    feedId: 'id',

    // название фида
    name: {
        type: 'string',
        preprocess: function(value) { return value.trim(); },
        validation: {
            rules: { required: true },
            validate: function(name) {
                return !!name;
            }
        }
    },

    // тип фида
    type: {
        type: 'string'
    },

    // тип бизнеса фида
    business_type: {
        type: 'enum',
        enum: u.feedData.businessType,
        default: u.feedData.defaultBusinessType
    },

    // дата последнего изменения вида 01.01.2014 в 01:01
    lastChange: 'string',

    // имя файла фида
    fileName: {
        type: 'string',
        validation: {
            text: iget2('vm-feed', 'podderzhivayutsya-tolko-fayly-zip', 'Поддерживаются только файлы: zip, gz, yml, csv, tsv, xml, xls, xlsx'),
            validate: function(name) {
                return name && /\.(zip|gz|yml|csv|tsv|xml|xls|xlsx)$/.test(name);
            },
            needToValidate: function() {
                return this.get('source') === 'file';
            }
        }
    },

    // источник файла фида
    source: {
        type: 'enum',
        enum: u.feedData.source
    },

    // url файла фида
    url: {
        type: 'string',
        preprocess: function(value) { return value.trim(); },
        validation: {
            validate: function(url) {
                // TODO после поддержки бэкэндом IPv6 ссылок, добавить также u.ipV6DomainRegExpStr
                var feedDomainRegExpStr = '((' + u.domainRegExpString + ')|(' + u.ipV4RegExpStr + '))';

                // требуется кастомная валидация url с включением протокола ftp (которой не проходит при валидации утилитами)
                return (new RegExp('^\\s*(?:(http(s)?|ftp):\/\/)?' + feedDomainRegExpStr +
                    '(?:\\s*$|\/|:\\d+|\\?|#)', 'i')).test(url);
            },
            needToValidate: function() {
                return this.get('source') === 'url';
            }
        }
    },

    // логин
    login: {
        type: 'string',
        preprocess: function(value) { return value.trim(); },
        validation: {
            validate: function(login) {
                return !!login;
            },
            needToValidate: function() {
                return this.get('source') === 'url' && this.get('authRequired');
            }
        }
    },

    // пароль
    password: {
        type: 'string',
        format: function(value) {
            // либо введенный, либо пробелы для звездочек, если пароль уже есть
            return value || (this.model.get('hasPassword') ? '       ' : '');
        },
        preprocess: function(value) {
            return value.trim();
        },
        dependsFrom: ['hasPassword'],
        validation: {
            validate: function(password) {
                return !!password || this.get('hasPassword');
            },
            needToValidate: function() {
                return this.get('source') === 'url' && this.get('authRequired');
            }
        }
    },

    // уже имеется пароль (флаг для того, чтобы не вводить его при редактировании)
    hasPassword: 'boolean',

    isReadOnly: 'boolean',

    // нужна ли авторизация с помощью логина/пароля
    authRequired: {
        type: 'boolean',
        default: false
    },

    // элементы фида
    offersCount: 'number',

    // количество ошибок при скачивании файла по ссылке
    fetchErrorsCount: 'number',

    // количество ошибок парсинга
    errorsCount: 'number',

    // количество предупреждений при парсинге
    warningsCount: 'number',

    // статус фида
    status: {
        type: 'enum',
        enum: u.feedData.status
    },

    // статус фида как текс
    statusText: 'string',

    // флаг требования автоматически удалять UTM-метки
    removeUTM: 'boolean',

    // список кампаний, в которых используется фид
    campaigns: {
        type: 'models-list',
        modelName: 'vm-feed-campaign'
    },

    /**
     * декларация DM модели
     */
    _dmDecl: {
        type: 'object',
        internal: true,
        calculate: function() {
            return {
                name: 'dm-feed',
                id: this.id
            };
        }
    }

}, {

    /**
     * Проверяет новая ли модель (у новой нет DM)
     * @returns {Boolean}
     */
    isNew: function() {
        return !this.getDM();
    },

    /**
     * Возвращает значение поля в DM по полю в VM
     * @param {String} field
     * @returns {*}
     */
    getFieldFromDM: function(field) {
        return this.prepareDataFromDM(this.getDM().toJSON())[field];
    },

    /**
     * Синхронизация из DM
     */
    syncFromDM: function() {
        this.__base.apply(this, arguments);

        this.get('campaigns').forEach(function(campaign) {
            campaign.syncFromDM();
        });
    },

    /**
     * Подготовка DM данных для применения их в VM
     * @param {Object} data
     * @returns {Object}
     */
    prepareDataFromDM: function(data) {
        var hasPassword = data.hasPassword;

        return {
            feedId: data.feed_id,
            isReadOnly: data.isReadOnly,
            name: data.name,
            type: data.feed_type,
            business_type: data.business_type,
            lastChange: u.moment(data.last_change).format(iget2('vm-feed', 'dd-mm-yyyy-hh', 'DD.MM.YYYY HH:mm')),
            fileName: data.filename,
            source: data.source,
            url: data.url,
            login: data.login,
            hasPassword: hasPassword,
            offersCount: data.offers_count,
            fetchErrorsCount: data.fetch_errors_count,
            errorsCount: data.errors_count,
            warningsCount: data.warnings_count,
            removeUTM: data.is_remove_utm === '1',
            status: data.display_status,
            authRequired: hasPassword,
            statusText: ({
                New: iget2('vm-feed', 'novyy', 'Новый'),
                Outdated: iget2('vm-feed', 'nuzhdaetsya-v-obnovlenii', 'Нуждается в обновлении'),
                Updating: iget2('vm-feed', 'na-validacii', 'На валидации'),
                Done: iget2('vm-feed', 'zagruzka-uspeshna', 'Загрузка успешна'),
                Error: iget2('vm-feed', 'oshibka-zagruzki', 'Ошибка загрузки')
            })[data.display_status],
            campaigns: data.campaigns.map(function(data) { return { cid: data.id }; })
        };
    }

});
