'use strict';

const fs = require('fs');
const techs = {
    dev: require('direct-dev').techs,
    bem: require('enb-bem-techs'),
    enb: {
        provideFile: require('enb/techs/file-provider'),
        fileMerge: require('enb/techs/file-merge'),
        browserJs: require('enb-js/techs/browser-js'),
        borschik: require('enb-borschik/techs/borschik'),
        stylus: require('enb-stylus/techs/stylus')
    },
    chromeTesting: require('enb-headless-chrome-testing'),
    xjst: {
        bemhtml: require('enb-bemxjst/techs/bemhtml'),
        bemjsonToHtml: require('enb-bemxjst/techs/bemjson-to-html')
    }
};
const transporterPlugins = require('direct-dev').transporterPlugins;

const BlockFilter = require('direct-dev').BlockFilter;

const levels = [
    { path: 'node_modules/@yandex-lego/ui.i-bem/common.blocks', check: false },
    { path: 'node_modules/islands/common.blocks', check: false },
    { path: 'node_modules/@yandex-lego/ui.islands-deprecated/common.blocks', check: false },
    { path: 'node_modules/islands/desktop.blocks', check: false },
    { path: 'node_modules/bem-mvc-direct/blocks', check: false },
    { path: 'node_modules/bem-mvc-direct/override.blocks', check: false },
    { path: 'source.blocks', check: true }
];

const bundleConfig = require('../bundles.json');

function configureNode(bundle, rootPath, nodeConfig) {
    let hasTests = bundle.hasTests;             // bool
    let needCoverage = bundle.coverage;         // bool
    let entities = bundle.entities;             // string[]
    let buildLevels = [].concat(bundle.baseLevels || []).concat(levels || []);

    let filter = new BlockFilter(
        { targetBlock: bundle.block, targetLevels: ['source.blocks'] },
        { rootPath: rootPath }
    );

    nodeConfig.addTechs([
        // bundle endpoint
        entities
            ? [techs.dev.devDeclaration, { target: '?.bemdecl.js', entities: entities }]
            : [techs.enb.provideFile, { target: '?.bemdecl.js' }],

        // essential
        [techs.bem.levels, { levels: buildLevels }],
        [techs.bem.files],
        [techs.bem.depsOld],

        // css
        [techs.enb.stylus, { url: 'inline' }],

        // bemhtml
        [techs.xjst.bemhtml, { engineOptions: { elemJsInstances: true } }],

        // js
        [techs.dev.transporter('js'), {
            target: '?.browser.js',
            sourceSuffixes: ['utils.js', 'vm.js', 'js'],
            apply: [
                needCoverage && transporterPlugins.coverage({ filter: filter.vinyl }),
                transporterPlugins.wrap()
            ]
        }],

        [techs.enb.fileMerge, { target: '?.merged.js', sources: ['?.bemhtml.js', '?.browser.js'] }],
        [techs.enb.borschik, { sourceTarget: '?.merged.js', destTarget: '?.js', minify: false }],

        // dev bundles
        [techs.dev.sandbox, { filter: filter }],
        [techs.dev.jsTest, { filter: filter }],
        [techs.dev.devPageBemjson, { target: '?.test.bemjson', type: 'test', js: '?.js', devJs: '?.test.js', css: '?.css' }],
        [techs.dev.devPageBemjson, { target: '?.sandbox.bemjson', type: 'sandbox', js: '?.js', devJs: '?.sandbox.js', css: '?.css' }],
        [techs.xjst.bemjsonToHtml, { target: '?.test.html', bemjsonFile: '?.test.bemjson' }],
        [techs.xjst.bemjsonToHtml, { target: '?.sandbox.html', bemjsonFile: '?.sandbox.bemjson' }],
        hasTests
            ? [techs.chromeTesting, { html: '?.test.html' }]
            : [techs.dev.emptyTestResult(needCoverage), {
                sourceSuffixes: ['utils.js', 'vm.js', 'js'],
                filter: filter
            }]
    ]);

    bundle.target && nodeConfig.addTargets([bundle.target]);
}

module.exports = function(config) {
    var targetBlock = process.env.block;

    bundleConfig
        .filter(bundle => !targetBlock || bundle.block === targetBlock)
        .forEach(function(bundle) {
            config.nodes(bundle.path, configureNode.bind(this, bundle, config.getRootPath()));
        });
};
