/**
 * @fires change – триггерится при изменении активного элемента
 * @fires select-item – триггерится всегда при клике на элемент, даже если он уже был выбран (в отличие от change)
 */
BEM.DOM.decl('b-chooser', {

    onSetMod: {
        js: function() {
            var selected = this.findElem('item', 'selected', 'yes');

            if (selected.length) {
                this._prevSelectedName = this.elemInstance(selected).getItemParams().name;
            }
        },

        multi: function(modName, modVal, prevVal) {
            // если в мультирежиме был выбран один элемент, затем
            // блок переключили в режим одиночного выбора
            // нужно запомнить выбранный элемент, чтобы отключить его при выборе другого
            if(modVal == '' && this.getAll('selected').length == 1) {
               this._prevSelectedName = this.getAll('selected')[0].name;
            }
        }
    },

    /**
     * Выставляет на элемент c именем itemName модификатор hovered
     * @param {String} itemName
     */
    hoverItem: function(itemName) {
        if (this._prevSelectedName) {
            this.delMod(this.elem('item', 'name', this._prevSelectedName), 'hovered', 'yes')
        }

        if (itemName) {
            this.setMod(this.elem('item', 'name', itemName), 'hovered', 'yes');
        }
    },

    /**
     * Удаляет с элемента itemName модификатор hovered
     * @param {String} itemName
     */
    unHoverItem: function(itemName) {
        this.delMod(this.elem('item', 'name', itemName), 'hovered', 'yes');
    },

    /**
     * name последнего выбранного элемента item
     */
    _prevSelectedName: null,

    /**
     * В зависимости от data.selected добавляет/удаляет item в хранилище
     * @param {Object} data
     * @param {String} data.name - имя элемента item
     * @param {Boolean} data.selected - выбран/не выбран
     * @param {*} [data.extraParams] - дополнительные параметры
     * @returns {BEM}
     * @private
     */
    _onItemChange: function(data) {
        if (data.selected) {
            // делаем предыдущий item не выбранным
            this._prevSelectedName && this._uncheckOne(this._prevSelectedName);
            this._prevSelectedName = data.name;
        } else {
            this._prevSelectedName = null;
        }

        return this.trigger('change', data);
    },

    /**
     * @private
     */
    _getRowHeight: function() {
        return this._rowHeight || (this._rowHeight = this.findElem('item').outerHeight());
    },

    /**
     * Действия, которые выполняются при показе контрола
     */
    onControlShowed: function() {
        if (this._prevSelectedName) {
            var viewport = this.elem('wrap').length ? this.elem('wrap') : this.domElem;

            u.scrollNodeTo(this.elem('item', 'name', this._prevSelectedName), viewport);

            this.hoverItem(this._prevSelectedName);
        }
    },

    /**
     * Вызывается после удаления элемента item из DOM-дерева
     * @param {Object} data
     * @param {String} data.name - имя элемента
     * @param {Boolean} data.selected - выбран/не выбран
     * @param {*} [data.extraParams] - дополнительные параметры
     * @returns {BEM}
     * @private
     */
    _onItemRemove: function(data) {
        this.trigger('remove', data);

        return this;
    },

    /**
     * Вызывается при попытке удаления элемента item из DOM-дерева
     * @param {Object} data
     *  @param {String} data.name - имя элемента item
     *  @param {Boolean} data.selected - выбран/не выбран
     *  @param {*} [data.extraParams] - дополнительные параметры
     * @returns {BEM}
     * @private
     */
    _onItemRemoving: function(data) {
        return this.trigger('removing', data);
    },

    /**
     * Делает item не выбранным
     * @param {String} name - имя элемента item
     * @param {*} [triggerParams]
     * @private
     */
    _uncheckOne: function(name, triggerParams) {
        var instance = this.elemInstance('item', 'name', name);

        if (instance && instance.hasMod('selected', 'yes')) {
            instance.toggle(triggerParams);
        }
    },

    /**
     * Возвращает выбранный item
     * @returns {Object}
     */
    getSelected: function() {
        return this.getAll('selected')[0];
    },

    /**
     * Возвращает значение выбранного пункта
     * @returns {String}
     */
    val: function() {
        var selected = this.getSelected();

        return selected ? selected.name : '';
    },

    /**
     * Возвращает все элементы
     * @param {String} [type] - название модификатора элемента
     * @returns {Array}
     */
    getAll: function(type) {
        var instances = type ? this.elemInstances('item', type, 'yes') : this.elemInstances('item');

        return instances.reduce(function(prev, current) {
            current.hasMod('name') && prev.push(current.getItemParams());

            return prev;
        }, []);
    },

    /**
     * Делает элемент или массив элементов не выбранными
     * @param {String|Array} name - имя элемента item или массив имен
     * @param {*} [triggerParams]
     * @returns {BEM}
     */
    uncheck: function(name, triggerParams) {
        if ($.isArray(name)) {
            name.map(function(name) {
                this._uncheckOne(name, triggerParams);
            }, this);
        } else {
            this._uncheckOne(name, triggerParams);
        }

        return this;
    },

    /**
     * Делает item выбранным
     * @param {String} name - имя элемента item
     * @param {*} [triggerParams]
     * @returns {BEM}
     */
    check: function(name, triggerParams) {
        var instance = this.elemInstance('item', 'name', name);

        if (instance && !instance.hasMod('selected', 'yes')) {
            instance.toggle(triggerParams);
        }

        return this;
    },

    /**
     * Удаляет item из DOM-дерева
     * @param {String} name - имя элемента item
     * @param {*} [triggerParams]
     * @returns {BEM}
     */
    remove: function(name, triggerParams) {
        var instance = this.elemInstance('item', 'name', name);

        instance && instance.remove(triggerParams);

        return this;
    },

    /**
     * Удаляет все item из DOM-дерева
     * @param {*} [triggerParams]
     * @returns {BEM}
     */
    removeAll: function(triggerParams) {
        this.elemInstances('item').forEach(function(instance) {
            instance.remove(triggerParams);
        }, this);

        return this;
    },

    /**
     * Создает и добавляет новый item в DOM-дерево
     * @param {Object} item
     * @param {String} item.name - имя элемента item
     * @param {*} item.content
     * @param {Object} [item.js]
     * @param {Object} [item.elemMods]
     * @returns {BEM}
     */
    add: function(item) {
        this.elemInstance('item', 'name', item.name) || BEM.DOM.append(this.domElem, BEMHTML.apply({
            block: 'b-chooser',
            elem: 'item',
            elemMods: item.elemMods,
            mix: item.mix,
            js: item.js,
            name: item.name,
            content: item.content
        }));


        return this;
    },

    /**
     * Заменяет item в DOM-дереве
     * @param {String} name
     * @param {Object} item
     * @param {String} item.name - имя элемента item
     * @param {*} item.content
     * @param {Object} [item.js]
     * @param {Object} [item.elemMods]
     * @returns {BEM}
     */
    replace: function(name, item) {
        var replacedInstance = this.elemInstance('item', 'name', name);

        if (replacedInstance && !this.elemInstance('item', 'name', item.name)) {
            BEM.DOM.replace(replacedInstance.domElem, BEMHTML.apply({
                block: 'b-chooser',
                elem: 'item',
                elemMods: item.elemMods,
                mix: item.mix,
                js: item.js,
                name: item.name,
                content: item.content
            }));
        }

        return this;
    },

    /**
     * Скрывает item от поиска
     * @param {String} name - имя элемента item
     * @returns {BEM}
     */
    disable: function(name) {
        this.elemInstance('item', 'name', name).disable();

        return this;
    },

    /**
     * Возвращает все элементы (без групп)
     * @returns {BEM[]}
     * @private
     */
    _getAllItems: function() {
        return this._allItems || (this._allItems = [].concat(this.elemInstances('item')));
    },

    /**
     * Возвращает все группы
     * @return {BEM[]}
     * @private
     */
    _getAllGroups: function() {
        return this._allGroups || (this._allGroups = [].concat(this.elemInstances('group')));
    },
    /**
     * Возвращает все группы и элементы вместе параметрами
     * @typedef { item: BEM; params: object } AnyItem
     * @return { groups: AnyItem[]; items: AnyItem[] }
     */
    getFlatTree: function() {
        if (!this._flatTree) {
            this._flatTree = {
                groups: this._getAllGroups().map(function(group) {
                    return { item: group, params: group.val() };
                }),
                items: this._getAllItems().map(function(item) {
                    return { item: item, params: item.val() };
                })
            };
        }

        return this._flatTree;
    },

    setStates: function(states) {
        var flatTree = this.getFlatTree();

        [].concat(
            flatTree.items,
            flatTree.groups
        ).forEach(function(container) {
            var item = container.item,
                name = container.params.name;

            item.setMod('hidden', states.hidden[name] ? 'yes' : '');
        });
    },

    /**
     * Открывает item для поиска
     * @param {String} name - имя элемента item
     * @returns {BEM}
     */
    enable: function(name) {
        this.elemInstance('item', 'name', name).enable();

        return this;
    },

    /**
     * Обрабатывает ошибку
     * @param {Object} error
     * @returns {BEM}
     */
    onError: function(error) {
        return this.trigger('error', error);
    },

    _chunkSize: 80,

    _chunkDelay: 0

}, {

    live: function() {

        this
            .liveInitOnBlockInsideEvent('change', 'b-chooser__item', function(e, data) {
                this._onItemChange(data);
            })
            .liveInitOnBlockInsideEvent('select-item', 'b-chooser__item', function(e, data) {
                this.trigger('select-item', data);
            })
            .liveInitOnBlockInsideEvent('remove', 'b-chooser__item', function(e, data) {
                this._onItemRemove(data);
            })
            .liveInitOnBlockInsideEvent('removing', 'b-chooser__action', function(e, data) {
                this._onItemRemoving(data);
            })
            .liveInitOnBlockInsideEvent('unselect-all', 'b-chooser__unselect-all', function(e, data) {
                this._uncheckOne(this._prevSelectedName);
            });;
    }

});
