BEM.DOM.decl('b-errors-presenter2', {

    onSetMod: {
        js: function() {
            this._usedErrorsObj = {};
        }
    },

    /**
     * Отображает ошибки в блоках с интерфейсом `i-error-view-interface` с наиболее подходящим `path`
     * @param {{ path: string, description: [string], text: [string] }[]} errors
     * @returns {BEM.DOM[]}
     */
    showErrors: function(errors) {
        this._resetErrors();

        var errorsViewMap = this._createMap(),
            unUsedErrors = [];

        errors.forEach(function(error) {
            var matchedPath = this._getMatchedPath(u['b-errors-presenter2'].toPath(error.path), errorsViewMap);

            if (u._.isUndefined(matchedPath)) {
                unUsedErrors.push(error);
            } else {
                this._usedErrorsObj[matchedPath] = (this._usedErrorsObj[matchedPath] || []).concat(error);
            }
        }, this);

        u._.forEach(this._usedErrorsObj, function(errors, matchedPath) {
            errorsViewMap[matchedPath].forEach(function(errorView) {
                errorView.showErrors(errors);
            });
        });

        this._initCleaners();

        this.trigger('show', {
            used: this._getUsedList(),
            unUsed: unUsedErrors
        });
    },

    /**
     * Скрывает ошибки которые подходят по переданному `path`
     * @param {String} [path='']
     */
    clearErrors: function(path) {
        var normalizedClearPath = u['b-errors-presenter2'].formatPath(path || ''),
            errorsViewMap = this._createMap(),
            matchedPathes = Object.keys(errorsViewMap).reduce(function(res, path) {
                return path.indexOf(normalizedClearPath) === 0 ? res.concat(path) : res;
            }, []),
            cleared = [];

        matchedPathes.forEach(function(path) {
            errorsViewMap[path].forEach(function(errorView) {
                if (!errorView._isDestructing) {
                    errorView.clearErrors();
                }
            });

            cleared = cleared.concat(this._usedErrorsObj[path] || []);
            delete this._usedErrorsObj[path];
        }, this);

        this.trigger('clear', {
            cleared: cleared,
            used: this._getUsedList()
        });
    },

    _resetErrors: function() {
        if (u._.size(this._usedErrorsObj)) {
            var errorsViewMap = this._createMap();

            u._.forEach(this._usedErrorsObj, function(errors, matchedPath) {
                errorsViewMap[matchedPath].forEach(function(errorView) {
                    errorView.clearErrors();
                });
            });

            this._usedErrorsObj = {};

            this.trigger('reset');
        }
    },

    /**
     * Возвращает наиболее подходящий `path` из имеющихся в `errorsMap`
     * @param {String[]} pathArray
     * @param {Object} errorsMap
     * @returns {String|undefined}
     * @private
     */
    _getMatchedPath: function(pathArray, errorsMap) {
        var path = u['b-errors-presenter2'].joinPath(pathArray);

        if (path in errorsMap) {
            return path;
        } else if (pathArray.length) {
            return this._getMatchedPath(pathArray.slice(0, -1), errorsMap);
        }
    },

    /**
     * Согласно DOM (конкатенирует data-атрибуты `err-path` родительских DOM-нод) формирует объект,
     * ключами которого являются `path`,
     * значениям - экземпляры блоков с интерфейсом `i-error-view-interface`
     * @returns {{string: BEM.DOM[]}}
     * @private
     */
    _createMap: function() {
        var collectPath = this._collectPath.bind(this);

        return this.findBlocksByInterfaceInside('i-error-view-interface')
            .reduce(function(result, errorViewBlock) {
                var normalizedPath = collectPath(errorViewBlock.domElem);

                result[normalizedPath] = (result[normalizedPath] || []).concat(errorViewBlock);

                return result;
            }, {});
    },

    /**
     * Инициализирует чистильщиков ошибок
     * @private
     */
    _initCleaners: function() {
        var cleaners = this.dropElemCache('cleaner').elem('cleaner');

        if (cleaners.length) {
            this._cleanSubMan || (this._cleanSubMan = BEM.create('i-subscription-manager'));
            this._cleanSubMan.dispose();

            cleaners.each(this._initCleaner.bind(this));
        }
    },

    /**
     * Инициализирует чистильщика ошибок
     * @param {number} i
     * @param {DOMNode} cleanerNative
     * @private
     */
    _initCleaner: function(i, cleanerNative) {
        var cleaner = $(cleanerNative),
            params = this.elemParams(cleaner),
            blockName = u._.get(params, 'block');

        if (blockName) {
            var block = this.findBlockOn(cleaner, blockName);

            if (block) {
                var event = u._.get(params, 'event', 'change');

                this._cleanSubMan.onFirst(block, event, { cleaner:  cleaner }, this._onClean, this);
            }
        }
    },

    /**
     * Обработчик события очистки, вызванное чистильщиком
     * @param {jQuery.Event} e
     * @param {Object} e.data
     * @param {jQuery} e.data.cleaner
     * @private
     */
    _onClean: function(e) {
        var path = this._collectPath(e.data.cleaner);

        this.clearErrors(path);
    },

    /**
     * Возвращает список отображаемых ошибок
     * @return {{ path: string, description: [string], text: [string] }[]}
     * @private
     */
    _getUsedList: function() {
        return u._.reduce(this._usedErrorsObj, function(res, errors) {
            return res.concat(errors);
        }, []);
    },

    /**
     *
     * @param {jQuery} item
     * @return {string}
     * @private
     */
    _collectPath: function(item) {
        return u['b-errors-presenter2'].formatPath(
            u['b-errors-presenter2'].joinPath(
                u['b-errors-presenter2'].getPathFromDom(item.get(0), this.domElem.get(0))
            )
        );
    }

}, {

    live: true

});
