/**
 * @typedef {Object} BEMJSON
 * @property {String} block
 * @property {Object|Object[]|String|String[]} [content]
 */

/**
 * Блок-декоратор для модального попапа
 * @fires b-modal-popup-decorator#outside-click при клике вне попапа
 * @fires b-modal-popup-decorator#close-blocked при блокировке закрытия при наличии изменений во внутреннем блоке
 * @fires b-modal-popup-decorator#close при полном закрытии попапа
 */
BEM.DOM.decl('b-modal-popup-decorator', {

    onSetMod: {
        js: function() {
            this._outsideMargin = this.params.offset || 20;
        }
    },

    /**
     * Показывает попап
     * @returns {BEM.DOM<b-modal-popup-decorator>}
     */
    show: function() {
        var controlToFocus = this.findBlockInside({ block : 'input', modName : 'autofocus', modVal : 'yes' });

        this.getPopup().show();

        controlToFocus && controlToFocus.setMod('focused', 'yes');

        this._maxHeight || this._fitPopupSize();

        return this;
    },

    /**
     * Прячет попап
     * Если есть внутренний блок и в нём есть изменения, то закрытия попапа не будет
     * @param {Object} [params] Параметры
     * @param {Boolean} [params.force] при наличии параметра принудительно закрываем попап, не смотря на изменения во внутреннем блоке
     * @returns {BEM.DOM<b-modal-popup-decorator>}
     * @fires b-modal-popup-decorator#close-blocked при наличии изменений во внутреннем блоке (кроме случаев когда метод вызван при клике снаружи попапа)
     */
    hide: function(params) {
        this.getPopup().hide(u._.get(params, 'force') ? { source: 'force' } : undefined);

        return this;
    },

    /**
     * Переключает состояние видимости/скрытия попапа
     * @returns {BEM.DOM<b-modal-popup-decorator>}
     */
    toggle: function() {
        this.getPopup().toggle();

        return this;
    },

    /**
     * Возвращает состояние видимости попапа
     * @returns {Boolean}
     */
    isShown: function() {
        return this.getPopup().isShown();
    },

    /**
     * Перерисовывает попап.
     * @returns {BEM.DOM}
     */
    repaint: function() {
        this.getPopup().repaint();

        return this;
    },

    /**
     * Удаляет блок
     * @override переопределён с целью подготовки блока попапа к удалению
     */
    destruct: function() {
        var args = arguments,
            base = this.__base;

        // нужно дождаться завершения асинхронных процессов попапа
        this.getPopup().afterCurrentEvent(function() {
            BEM.DOM.destruct(this.domElem, true);
            base.apply(this, args);
        }, this);
    },

    /**
     * Возвращает блок попапа
     * @returns {BEM.DOM<popup>}
     */
    getPopup: function() {
        if (!this._popup) {
            this._popup = this.findBlockOn('popup')
                .on('show', function() {
                    // при ресайзе меняем максимальную высоту без учета текущего отступа от верхнего края
                    this.bindToWin('resize', $.debounce(this._fitPopupSizeToViewport, 100, this));
                }, this)
                .on('before-hide', function(e, data) {
                    var source = data && data.source,
                        innerBlock = this._getInnerBlock();

                    if (innerBlock && source !== 'force') {
                        e.preventDefault();

                        innerBlock.isChanged().done(function(isChanged) {
                            if (isChanged) {
                                this.trigger('close-blocked');
                            } else {
                                this.getPopup().hide({ source: 'force' });
                            }
                        }.bind(this));
                    }
                }, this)
                .on('hide', function() {
                    this.unbindFromWin('resize');
                    this.trigger('close');
                }, this);
        }

        return this._popup;
    },

    /**
     * Задаёт содержимое блока попапа
     * @param {BEMJSON|BEMJSON[]} content BEMJSON блока, который реализует интерфейс i-modal-popup-inner-block-interface
     * @throws {Error} при отсутствии блока требуемого интерфейса
     * @returns {BEM.DOM} блок внутри
     */
    setPopupContent: function(content) {
        var block;

        this.getPopup().setContent(BEMHTML.apply(content));

        block = this._getInnerBlock();

        if (!block) throw new Error('Couldn\'t find block with interface \'i-modal-popup-inner-block-interface\'');

        // даем возможность внутреннему блоку триггерить перерисовку
        // актуально, например, для фидов, которые после ajax-запросов резко
        // увеличиваются в размерах
        block.on('repaint', function() {
            $.debounce(this._fitPopupSizeToViewport, 100, this)();
        }, this);

        this._maxHeight || this._fitPopupSize();

        return block;
    },

    /**
    * Дизейблит/энейблит крестик
    * @param {String} disabledVal - модификатор для disabled
    */
    toggleClose: function(disabledVal) {
        // сам крестик не умеет дизейблится, проще его временно спрятать
        this.getPopup().setMod('has-close', disabledVal ? 'no' : 'yes');
    },

    /**
     * Обработчик ресайза окна
     * Надо пересчитать высоту без учета текущего отступа
     * @private
     */
    _fitPopupSizeToViewport: function() {
        if (this.domElem) {
            // центрирование попапа -  чтобы учесть отступы
            this.repaint()._fitPopupSize();
        }
    },

    /**
     * Подгоняет размеры контента попапа
     * По умолчанию false
     * @private
     */
    _fitPopupSize: function() {
        if (!this.isShown()) return this;

        var body = this.findElem('body'),
            curMaxHeight = this._maxHeight,
            newMaxHeight;

        // если нет контента с body, то не чему подгонять размеры
        if (!body.size()) return this;

        newMaxHeight = this._calculateMaxHeight();
        if (newMaxHeight != curMaxHeight) {
            this._maxHeight = newMaxHeight;
            body.css({ maxHeight: this._maxHeight });
            this.repaint();
        }

        return this;
    },

    /**
     * Рассчет максимальной высоты с учетом отступов
     * @returns {Number} - максимальная высота body попапа
     * @private
     */
    _calculateMaxHeight: function() {
        var margin = this._outsideMargin,
            currentOffsetTop = this.getPopup().domElem.offset().top - BEM.DOM.win.scrollTop(),
            body = this.elem('body');

        return BEM.DOM.getWindowSize().height -
            (this.getPopup().getPlacingSize().height - body.height() +
                margin + Math.max(currentOffsetTop, margin));
    },

    /**
     * Возвращает внутренний блок в попапе, реализующий интерфейс i-modal-popup-inner-block-interface
     * @returns {BEM.DOM}
     * @private
     */
    _getInnerBlock: function() {
        return this.getPopup().findBlockByInterfaceInside('i-modal-popup-inner-block-interface');
    }

}, {

    /**
     * Создаёт блок-декоратор модального попапа
     * @param {Object} [mods] Модификаторы блока декоратора
     * @param {Object} [options] Дополнительные параметры
     * @param {Boolean} [options.hasClose] - Показывать крестик (по умолчанию true)
     * @param {Boolean} [options.bodyScroll] - Возможность прокрутки страницы, когда открыт попап (по умолчанию true)
     * @param {Boolean} [options.childState] - Включает добавление модификатора child-shown, если внутри открыт дочерний popup
     * @returns {BEM.DOM<b-modal-popup-decorator>}
     */
    create: function(mods, options) {
        var $ = window.top.$ || window.$,
            blockName = this.getName(),
            defaults = { hasClose: true, bodyScroll: true, childState: false, offset: 20 },
            opts = u._.extend(defaults, options);

        return $(BEMHTML.apply({
            block: blockName,
            mods: mods,
            bodyScroll: opts.bodyScroll,
            hasClose: opts.hasClose,
            childState: opts.childState,
            offset: opts.offset
        })).bem(blockName);
    }
});
