/**
 * @typedef {Object} TreeStateData
 * @property {Array} state информация о сосотоянии отмеченных галок в дереве
 * @property {Object} changesInfo информация о том, в каких галках регионов были изменения
 * @property {Object} resolvedRegions список регионов, для которых регионы в группах кампании различались
 * и были приведены к единому виду
 * @property {Boolean} wasReset флаг, идентифицирующий факт сброса всех регионов
 */

BEM.DOM.decl('b-regions-tree', {

    onSetMod: {

        js: function() {
            var roots = this.params.roots,
                data = this._data = {},
                checkboxes = this.findBlocksInside('checkbox');

            this._hasIndependentRegions = this.params.hasIndependentRegions;
            this._changesInfo = {};
            this._resolvedRegions = {};

            // чиним двойной BEM-click на чекбоксе (триггерится когда происходит клик по лейблу)
            this._onRegionCheckboxClick = $.debounce(this._onRegionCheckboxClick, 25, this);

            checkboxes.forEach(function(checkbox) {
                var id = this.getMod(checkbox.domElem, 'id'),
                    region = data[id] || (data[id] = {}),
                    rootItem = roots[id];

                region.checkbox = checkbox;

                if (!rootItem) return;

                u._.extend(region, rootItem);

                region.contrastValueGroups && (this._hasRegionsValueDifferences = true);

                region.inners.forEach(function(innerId) {
                    data[innerId] || (data[innerId] = {});
                    data[innerId].parent = id;
                });
            }, this);
        }
    },

    onElemSetMod: {

        region: {

            /**
             * При установки модификатора expanded для region меняется модификатор visible у элемента inner
             * @param {*} elem
             * @param {String} modName
             * @param {String} modVal
             */
            expanded: function(elem, modName, modVal) {
                this.setMod(this._getInnerForRegion(elem), 'visible', modVal);
            }
        }
    },

    /**
     * Выбор/Отмена выбора региона
     * @param {jQuery} regionDom DOM нода региона
     * @param {Boolean} checked флаг выбора региона
     * @param {Boolean} skipIndependent не выбирать независимые регионы
     * @returns {BEM}
     * @private
     */
    setRegionState: function(regionDom, checked, skipIndependent) {
        var regionId = this.getMod(regionDom, 'id'),
            checkbox = this._data[regionId].checkbox,
            isCheckboxDisabled = checkbox.hasMod('disabled', 'yes'),
            isCheckedMode = checked && !isCheckboxDisabled;

        checkbox.setMod('checked', isCheckedMode ? 'yes' : '');

        if (this.getMod(regionDom, 'has-inner') == 'yes' && this.getMod(regionDom, 'visible') !== 'no') {
            //если выбрали/сняли выбор с независимого региона вручную - то нужно прятать предупреждение
            //если сняли выбор с региона, у которого есть дочерние независимые - нужно спрятать предупреждение
            if (skipIndependent && (checkbox.getMod('independent') || checkbox.getMod('independent-parent') || !checked && this._hasIndependentChildren(regionDom))) {
                this.trigger('independent-checkboxes-reset');
            }

            //независимые чекбоксы не чекаются вместе с родителями, но с них снимается checked, когда снимаем чек с родителей
            this._getChildCheckboxes(regionDom, skipIndependent && !!checked).forEach(function(checkbox) {
                var isCheckedMode = checked && !checkbox.hasMod('disabled', 'yes');

                checkbox.setMod('checked', isCheckedMode ? 'yes' : '');
            }, this);
        } else if (skipIndependent && (checkbox.getMod('independent') || checkbox.getMod('independent-parent'))) {
            this.trigger('independent-checkboxes-reset');
        }

        return this;
    },

    /**
     * Изменение набора выбранных регионов
     * @param {Array} regions список id включённых и исключённых регионов
     * @param {Boolean} [skipIndependent] не менять состояние у вложенных независисмых чекбоксов
     * @returns {BEM}
     */
    setTreeState: function(regions, skipIndependent) {
        var first;

        this._resetTreeState();

        // раньше сначала проставляли включённые регионы, а следующим шагом убирали исключённые
        // в результате, если внутри исключённого региона оказывался ранее включённый, мы его исключали (баг DIRECT-34013)

        // сейчас ожидается что в строке регионов, вложенные регионы идут по порядку после родительских
        // например 225,-3,1 <-> Россия (кроме: Центр), Москва и область
        regions.forEach(function(id, index) {
            var state = id > 0,
                regionDom = this.elem('region', 'id', Math.abs(id) + '');

            if (!regionDom.length) return;

            if (index == 0) first = id;

            this
                .setRegionState(regionDom, state, skipIndependent)
                ._setExpanderState(regionDom, state);

        }, this);

        first && this.scrollTo(this.elem('region', 'id', first));

        return this;
    },

    /**
     * Изменение набора выбранных регионов из быстрого выбора
     * включает в себя установку флага wasReset и заполнение _changesInfo
     * @param {Array} regions список id включённых и исключённых регионов
     * @param {Boolean} [skipIndependent] не менять состояние у вложенных независисмых чекбоксов
     * @returns {BEM}
     */
    setFromQuickLinks: function(regions, skipIndependent) {
        this._treeWasReset = true;

        this._changesInfo = {};
        regions.forEach(function(region) {
            this._changesInfo[region] = true;
        }, this);

        this.setTreeState(regions, skipIndependent);
    },

    /**
     * Получить набор выбранных регионов
     * @returns {TreeStateData}
     */
    getTreeState: function() {
        var checked,
            _this = this,
            checkboxes = _this._data,
            state,
            regionIdsInOrder;

        checked = this.findBlocksInside({ block: 'checkbox', modName: 'checked', modVal: 'yes' })
            .reduce(function(res, checkbox) {
                res[_this.getMod(checkbox.domElem, 'id')] = true;

                return res;
            }, {});     

        regionIdsInOrder = this.params.walkOrder.filter(function(id) {
            return checked[id];
        });
        
        state = regionIdsInOrder.reduce(function(res, checkedId) {
            var firstCheckedParentId,
                parentObject,
                excluded = [],
                findInnerExcluded = function(inner) {
                    inner.forEach(function(innerId) {
                        if (checked[innerId]) {
                            checkboxes[innerId].inners && findInnerExcluded(checkboxes[innerId].inners);

                            delete checked[innerId];
                        } else {
                            excluded.push({ id: innerId, name: checkboxes[innerId].name });
                        }
                    });
                },
                itemData;

            // так как элементы удаляются в процессе
            if (!checked[checkedId]) {
                return res;
            }

            firstCheckedParentId = _this._getFirstCheckedParent(checkedId, checked);
            parentObject = checkboxes[firstCheckedParentId];
            delete checked[firstCheckedParentId];

            parentObject.inners && findInnerExcluded(parentObject.inners);

            itemData = {
                id: firstCheckedParentId,
                name: parentObject.name,
                excluded: _.uniq(excluded)
            };

            res.push(itemData);

            return res;
        }, []);

        return {
            state: state,
            changesInfo: this._changesInfo,
            resolvedRegions: this._resolvedRegions,
            wasReset: this._treeWasReset
        }
    },

    /**
     * Сброс всех выбранных и раскрытых регионов
     */
    reset: function() {
        this._treeWasReset = true;

        if (this.params.isCommon && this._hasRegionsValueDifferences) {
            this._showRegionValueDifferencesConfirm(function() {
                this._resetTreeState();

                Object.keys(this._data).forEach(this._resetRegionValueDifferences, this);

                this._changesInfo = {};
            });
        } else {
            this._resetTreeState();
        }
    },

    /**
     * Разворачивание узла дерева
     * @param {jQuery} regionDom - DOM нода региона
     * @param {boolean} [recursive] раскрывать/схлопывать вложенные ноды
     * @returns {BEM}
     */
    expandRegionNode: function(regionDom, recursive) {
        return this._setExpanderState(regionDom, true, recursive);
    },

    /**
     * Сворачивание узла дерева
     * @param {jQuery} regionDom - DOM нода региона
     * @param {boolean} [recursive] раскрывать/схлопывать вложенные ноды
     * @returns {BEM}
     */
    collapseRegionNode: function(regionDom, recursive) {
        return this._setExpanderState(regionDom, false, recursive);
    },

    /**
     * Поиск частичного совпадения названия региона
     * @param {String} query искомая подстрока
     */
    partialSearch: function(query) {
        var last = this._lastSearch;

        query = query.toLowerCase();
        if (last && last.query == query) {
            if (last.result.length) {
                last.showPos = last.showPos == last.result.length - 1 ? 0 : last.showPos + 1;
            }
        } else {
            this.resetAllHighlights();

            last = this._lastSearch = {
                query: query,
                result: [],
                showPos: 0
            };

            $.each(this._data, function(id, region) {
                if (u.text.suggestMatch(region.name, query)) {
                    last.result.push(id);
                    region.parent && this.expandRegionNode(this.elem('region', 'id', region.parent));
                    this.highlightSubstring(this.elem('region', 'id', id), query);
                }
            }.bind(this));
        }

        last.result.length && this.scrollTo(this.elem('region', 'id', last.result[last.showPos]));

        return last.result.length;
    },

    /**
     * Полнотекстовый поиск
     * @param {String} query искомая строка
     * @returns {Boolean} результат поиска
     */
    search: function(query) {
        if (!query) return false;

        query = query.toLowerCase();

        this.resetAllHighlights();

        for (var id in this._data) {
            if (this._data[id].name.toLowerCase() === query) {
                var regionDom = this.elem('region', 'id', id);

                this
                    .expandRegionNode(regionDom, true)
                    .highlightSubstring(regionDom, query)
                    .scrollTo(regionDom);

                return true;
            }
        }

        return false;
    },

    /**
     * Подсветить подстроку в названии региона
     * @param {jQuery} regionDom DOM нода региона
     * @param {String} substr подстрока
     * @returns {BEM}
     */
    highlightSubstring: function(regionDom, substr) {
        var name = this._data[this.getMod(regionDom, 'id')].name,
            pos = name.toLowerCase().indexOf(substr.toLowerCase());

        pos !== -1 && BEM.DOM.update(
            this.findElem(regionDom, 'region-name'),
            name.slice(0, pos) +
            BEMHTML.apply({
                block: 'b-regions-tree',
                elem: 'highlighted',
                tag: 'span',
                content: name.slice(pos, pos + substr.length)
            }) + name.slice(pos + substr.length));

        return this;
    },

    /**
     * Сбросить все подсветки поиска
     * @returns {BEM}
     */
    resetAllHighlights: function() {
        $.each(this.findElem('highlighted'), function(n, dom) {
            var nameDom = $(dom).closest(this.buildSelector('region-name')),
                id = this.getMod(this._getRegionByChildDomNode(nameDom), 'id');

            BEM.DOM.update(nameDom, this._data[id].name);
        }.bind(this));
        this._lastSearch = null;

        return this;
    },

    /**
     * Перемещение полосы прокрутки к указанному региону
     * @param {jQuery} regionDom DOM нода региона
     * @returns {BEM}
     */
    scrollTo: function(regionDom) {
        regionDom.length && this.domElem.scrollTop(regionDom.position().top + this.domElem.scrollTop() - 10);

        return this;
    },

    /**
     * Возвращает массив названий регионов
     * @returns {Array}
     */
    getRegionsNames: function() {
        var res = [];

        $.each(this._data, function(id, region) {
            res.push(region.name);
        });

        return res;
    },

    /**
     * Скрывает регион(ы) по id
     * @param {Array} regionsId идентификаторы регионов
     * @param {Boolean} [show]
     */
    regionsVisibility: function(regionsId, show) {
        var modVal = show ? '' : 'no';

        regionsId.forEach(function(regionId) {
            var region = this.elem('region', 'id', regionId),
                regionInner = this.elem('inner', 'parent-id', regionId);

            this.setMod(region, 'visible', modVal);

            this.findBlocksInside(region, 'checkbox').forEach(function(check) {
                check
                    .setMod('disabled', (show || 'yes'))
                    .delMod('checked');
            });

            if (regionInner){
                this.setMod(regionInner, 'visible', modVal);
                this.findBlocksInside(regionInner, 'checkbox').forEach(function(check) {
                    check
                        .setMod('disabled', (show || 'yes'))
                        .delMod('checked');
                });
            }

        }, this);
    },

    /**
     * Устанавливает родительский попап
     * @param {BEM.DOM<popup>} parentPopup - родительский попап
     */
    setParentPopup: function(parentPopup) {
        this._parentPopup = parentPopup;
    },

    /**
     * Признак наличия различий выбранности в группах кампании для всех регионов
     */
    _hasRegionsValueDifferences: false,

    /**
     * Флаг, идентифицирующий факт сброса регионов
     */
    _treeWasReset: false,

    /**
     * Возвращает DOM контейнера вложенных регионов
     * @param {jQuery} regionDom DOM нода региона
     * @returns {jQuery|undefined}
     * @private
     */
    _getInnerForRegion: function(regionDom) {
        if (this.getMod(regionDom, 'has-inner') == 'no') return;

        var node = regionDom.closest(this.buildSelector('node-wrap')),
            inners = this.findElem(node, 'inner'),
            nodes = this.elem('node-wrap');

        for (var i = 0; i < inners.length; i++) {
            var inner = $(inners[i]);

            if (inner.closest(nodes)[0] == node[0]) {
                return inner;
            }
        }
    },

    /**
     * Возвращает dom региона по dom его внутренних элементов
     * @param {*} dom внутреннего элемента
     * @returns {jQuery}
     * @private
     */
    _getRegionByChildDomNode: function(dom) {
        return dom.closest(this.buildSelector('region'));
    },


    /**
     * Сворачивание/Разворачивание узла дерева
     * @param {jQuery} regionDom - DOM нода региона
     * @param {boolean} expanded - раскрытая – в случае true, иначе скрыта
     * @param {boolean} [recursive] раскрывать/схлопывать вложенные ноды
     */
    _setExpanderState: function(regionDom, expanded, recursive) {
        var id = this.getMod(regionDom, 'id');

        if (expanded) {
            while (id = this._data[id].parent) {
                this._setExpanderState(this.elem('region', 'id', id), true);
            }
        }

        if (this.getMod(regionDom, 'has-inner') != 'yes') return this;

        this.setMod(recursive ? this.findElem(regionDom.closest(this.buildSelector('node-wrap')), 'region', 'has-inner', 'yes') : regionDom,
            'expanded', expanded ? 'yes' : 'no');

        return this;
    },

    /**
     * Проверяет, есть ли у региона независимые потомки
     * @param {JQuery} regionDom
     * @returns {Boolean}
     * @private
     */
    _hasIndependentChildren: function(regionDom) {
        var checkboxes = this.findBlocksInside(this._getInnerForRegion(regionDom), 'checkbox');

        return checkboxes.some(function(cbx) {
            return cbx.hasMod('independent')
        }.bind(this));
    },

    /**
     * Возвращает все вложенные в regionDom чекбоксы/ все кроме независимых и потомков независимых
     * @param {jQuery} regionDom
     * @param {Boolean} skipIndependent не включать в список независимые чекбоксы
     * @returns {Array}
     * @private
     */
    _getChildCheckboxes: function(regionDom, skipIndependent) {
        var independentsArray = [],
            checkboxes = this.findBlocksInside(this._getInnerForRegion(regionDom), 'checkbox'),
            independentCheckboxesFiltered = false;

        if (!this._hasIndependentRegions || !skipIndependent) return checkboxes;

        checkboxes = checkboxes.filter(function(cbx) {
            var id = this.getMod(cbx.domElem, 'id'),
                independent = cbx.getMod('independent');

            if (independent || ~independentsArray.indexOf(this._data[id].parent)) {
                independentsArray.push(id);

                //если чекбокс уже чекнут и мы чекаем родительский - то предупреждение показывать не надо
                !cbx.isChecked() && (independentCheckboxesFiltered = true);

                return false;
            } else {

                return true;
            }

        }.bind(this));

        if (independentCheckboxesFiltered) {
            this._expandIndependentChildNodes(regionDom);
            this.trigger('independent-checkboxes-filtered');
        }

        return checkboxes;
    },

    /**
     * Разворачивает родителей независимых сабрегионов
     * @param {jQuery} regionDom
     * @returns {BEM}
     * @private
     */
    _expandIndependentChildNodes: function(regionDom) {
        var independentsArray = [];

        this.findBlocksInside(this._getInnerForRegion(regionDom), 'checkbox').forEach(function(cbx) {
            var id = this.getMod(cbx.domElem, 'id'),
                independent = cbx.getMod('independent');

            if (independent) {
                independentsArray.push(id);

            }

        }.bind(this));

        independentsArray.forEach(function(id) {
            var parentId = this._data[id].parent,
                regionDom = this.elem('region', 'id', parentId);

            this._setExpanderState(regionDom, true);
        }.bind(this));

        return this;
    },

    /**
     * Возвращает самого "старшего" выбранного родителя
     * @param {String} regionId идентификатор региона
     * @param {Object} checkedHash хэш идентификаторов чекнутых регионов
     * @returns {String}
     * @private
     */
    _getFirstCheckedParent: function(regionId, checkedHash) {
        var result = regionId,
            data = this._data,
            findFirstCheckedList = function(id) {
                checkedHash[id] && (result = id);
                data[id].parent && findFirstCheckedList(data[id].parent);
            };

        findFirstCheckedList(result);

        return result;
    },


    /**
     * Сброс всех выбранных и раскрытых регионов
     * @private
     * @returns {BEM}
     */
    _resetTreeState: function(){
        this.findBlocksInside({ block: 'checkbox', modName: 'checked', modVal: 'yes' }).forEach(function(checkbox) {
            checkbox.setMod('checked', '');
        });

        $.each(this.elem('region', 'expanded', 'yes'), function(n, regionDom) {
            this.collapseRegionNode($(regionDom));
        }.bind(this));

        this.resetAllHighlights();

        //скрываем подсказу про независимые регионы
        this.trigger('independent-checkboxes-reset');

        return this;
    },

    /**
     * Инициализирует и возвращает bem-блок попапа,
     * в котором будет подсказка о различиях настроек регионов в группах кампании
     * @returns {BEM}
     */
    _getRegionValueDifferencesHintPopup: function() {
        if (!this._hintPopup) {
            this._hintPopup = BEM.blocks['b-shared-popup'].getInstance(
                {
                    'fade-out': 'no',
                    adaptive: 'yes',
                    'has-close': 'yes'
                },
                {
                    directions: ['right', 'bottom', 'top']
                });

            this.bindTo(this.domElem, 'scroll', function() {
                this._hintPopup.hide();
            });
        }

        return this._hintPopup;
    },

    /**
     * Открывает подсказку про различия значений в группах камнании для региона
     * @param {jQuery} hintIcon иконка указывающая на различия, около которой будет открываться попап
     */
    _openDifferencesHintPopup: function(hintIcon) {
        var regionDom = this._getRegionByChildDomNode(hintIcon),
            regionData = this._data[this.getMod(regionDom, 'id')];

        this._getRegionValueDifferencesHintPopup()
            .setContent(BEMHTML.apply({
                block: 'b-regions-tree',
                elem: 'contrast-value-info',
                regionData: regionData,
                groups: regionData.contrastValueGroups
            }))
            .show(hintIcon);
    },

    /**
     * Определяет есть ли у региона или его детей различия выборанности в группах кампании
     * @param {String} regionId идентификатор региона
     * @returns {Boolean}
     */
    _hasRegionValueDifferences: function(regionId) {
        var data = this._data,
            check = function(item) {
                if (item.contrastValueGroups) {

                    return true;
                }

                return !!item.inners && item.inners.some(function(innerItemId) {
                    return check(data[innerItemId]);
                });
            };

        return check(this._data[regionId]);
    },

    /**
     * Показывает предупреждения о наличии различий выборанности в группах кампании
     * @param {Function} confirmCallback функция, выполняющаяся при нажатии на кнопку подтверждения
     * @private
     */
    _showRegionValueDifferencesConfirm: function(confirmCallback) {
        BEM.blocks['b-user-dialog']
            .confirm({
                message: this.params.hintForDifferentRegions ||
                    iget2(
                        'b-regions-tree',
                        'region-pokaza-kotoryy-vy',
                        'Регион показа, который вы хотите изменить, настроен по-разному в группах объявлений этой кампании. Например, в одной группе он выбран, а в другой нет. После внесения изменений во всех группах объявлений этот регион показа станет настроен одинаково. Например, станет выбран во всех группах.'
                    ),
                confrimButtonText: iget2('b-regions-tree', 'prodolzhit', 'Продолжить'),
                cancelButtonText: iget2('b-regions-tree', 'otmena', 'Отмена'),
                parentPopup: this._parentPopup,
                onConfirm: confirmCallback,
                callbackCtx: this
            });
    },

    /**
     * Удаляет признак различия выбранности региона в группах кампании
     * @param {String} regionId идентификатор региона
     */
    _resetRegionValueDifferences: function(regionId) {
        if (this._data[regionId].contrastValueGroups) {
            this._data[regionId].contrastValueGroups = undefined;
            this._resolvedRegions[regionId] = true;

            this.findBlockInside(this.elem('region', 'id', regionId), 'icon').destruct();
        }

        this._data[regionId].inners && this._data[regionId].inners.forEach(this._resetRegionValueDifferences, this);
    },

    /**
     * Обработчик клика по чекбоксу
     * @param {BEM} checkbox блок чекбокса по которому произошел клик
     * @private
     */
    _onRegionCheckboxClick: function(checkbox) {
        var checked = checkbox.isChecked(),
            regionDom = this._getRegionByChildDomNode(checkbox.domElem),
            regionId = this.getMod(regionDom, 'id'),
            updateChangesForInners = (function(innerRegionId) {
                if (this._changesInfo[innerRegionId] != undefined) {
                    this._changesInfo[innerRegionId] = checked;
                }

                (this._data[innerRegionId].inners || []).forEach(updateChangesForInners);
            }).bind(this),
            updateRegionState = (function() {
                this.setRegionState(regionDom, checked, true);
                this._changesInfo[regionId] = checked;
                checked && this._setExpanderState(regionDom, checked);

                // если в информации об изменениях дети уже присутсвуют
                // то чекнутость надо унаследовать от родителя
                checked && (this._data[regionId].inners || []).forEach(updateChangesForInners);
            }).bind(this);

        if (this.params.isCommon && this._hasRegionValueDifferences(regionId)) {
            checkbox.toggle();

            this._showRegionValueDifferencesConfirm(function() {
                checkbox.toggle();
                updateRegionState();

                this._resetRegionValueDifferences(regionId);
            });
        } else {
            updateRegionState();
        }
    }
}, {

    live: function() {
        this
            .liveInitOnBlockInsideEvent('click', 'checkbox', function(e, data) {
                this._onRegionCheckboxClick(e.block);
            })
            .liveBindTo('expander', 'pointerclick', function(e) {
                this.toggleMod(this._getRegionByChildDomNode(e.data.domElem), 'expanded', 'yes', 'no');
            })
            .liveBindTo('contrast-value-hint', 'pointerclick', function(e) {
                this._openDifferencesHintPopup(e.data.domElem);
            });
    }

});
