/**
 * "правило"
 * Сразу после создания требуется передать объект контролера, вызовом метода setController
 *
 * @event remove-rule(entityId)         удаление правила
 * @event change-rule(params)           изменение в данных (дни, счетчик), кроме типа правила
 * @event change-rule-type(params)      изменение типа правила
 * @event open-goal-list(type)          событие открытия списка целей. Подписчик должен получить цели и вызвать метод
 *     openGoalList type - тип цели
 *
 *
 * методы:
 * openGoalList(list)                   открыть список целей. list - массив списка целей
 *
 * @namespace b-retargeting-condition-edit-rule
 */
BEM.DOM.decl('b-retargeting-condition-edit-rule', /** @lends b-retargeting-condition-edit-rule */ {

    /**
     * @type {Object} мааппинг значений поля выбора select с модификаторами
     */
    _TYPE_VS_MOD: null,

    /**
     * @type {Object} подпись на кнопке (список), когда не выбрана "цель/сегмент"
     */
    _BUTTON_LABEL_LIST: null,

    /**
     * @type {Object} типы "правил", у которых необходимо указывать дни
     * срок, за который будет подсчитыватся пользователи, выполнившие цель
     */
    _TYPE_HAS_DAY: null,

    /**
     * Блок - список, выбор счетчика цели/сегмента
     * см. elem: __list
     */
    _dropdown: null,

    /**
     * Флаг, указывающий, что установка модификатора типа "правила", происходит первый раз
     * При этом не тригерим события об изменении данных - поскольку они еще не модифицировались
     */
    _firstSetType: true,

    /**
     * Максимальное кол-во элементов в списке, когда нет поля поиска
     * @type {number}
     */
    LIST_POPUP_CONTENT_GOALS_MAX_LENGTH_FOR_VISIBLE_SEARCH: 10,

    /**
     * @this b-retargeting-condition-edit-rule
     */
    onSetMod: {
        /**
         * @this b-retargeting-condition-edit-rule
         */
        js: function() {
            this._subscriptionManager = BEM.create('i-subscription-manager');
            this._fieldType = this.findBlockInside('type', 'select');
            this._fieldType.on('change', this._onChangeType, this);
            this.findBlockOn('remove-rule', 'link').on('click', this._onRemoveRule, this);

            this._TYPE_VS_MOD = u['b-retargeting-condition-edit-rule'].TYPE_VS_MOD;
            this._BUTTON_LABEL_LIST = u['b-retargeting-condition-edit-rule'].BUTTON_LABEL_LIST;
            this._TYPE_HAS_DAY = u['b-retargeting-condition-edit-rule'].TYPE_HAS_DAY;

            this._subscriptionManager.wrap(BEM.blocks['dropdown'])
                .on(this.elem('params'), 'show', this._onListOpen, this);

            this.setMod('type', this._TYPE_VS_MOD[this.getType()]);
        },

        /**
         * Состояние блока
         * @this b-retargeting-condition-edit-rule
         */
        'status-load': {
            /**
             * Данные загружаются
             * Показать индикатор загрузки, пока ходим за данными
             */
            loading: function() {
                this._setContentParams({
                    block: 'b-retargeting-condition-edit-rule',
                    elem: 'loading'
                });
                this._dropdown = null;
            },

            /**
             * Данные загрузились
             * @this b-retargeting-condition-edit-rule
             */
            loaded: function() {
                this._fieldDay = this.findBlockInside('field-day', 'input');

                if (this._fieldDay) {
                    this.dropElemCache();
                    this._fieldDay.on('change', this._changeFieldDay, this);
                    this._changeFieldDay();
                    if (this.params.isReadOnly) {
                        this._fieldDay.setMod('disabled', 'yes');
                    }
                }

                this._dropdown = this.findBlockInside('list', 'dropdown');

                // если тип изменил пользователь - открыть список.
                // для отключения этого поведения - удали эту строку (1 line)
                this._firstSetType || this.showListPopup();
            }
        },

        /**
         * Изменение модификатора типа
         * @lends b-retargeting-condition-edit-rule
         */
        type: function() {
            if (this._firstSetType) {
                this._setParams(this.params.rule);
                this.setMod('status-load', 'loaded');
                this._firstSetType = false;
            } else {
                this.setMod('status-load', 'loading');
                this.trigger(
                    'change-rule-type',
                    {
                        entityId: this.params._entityId,
                        type: this.getType()
                    }
                );
            }
        }
    },

    /**
     * Создание контента (элемент params) - поле выбора цели и дней
     * @param {{ [name], [domain], [goalId], [day], [type], allowToUse, hasGoalList: Boolean, audienceTypeName }} data
     */
    _setParams: function(data) {
        var type = this.getType(),
            modName = this._TYPE_VS_MOD[type];

        // TODO johnson DIRECT-62983
        if (!this._firstSetType && (!this.domElem || !this.hasMod('js', 'inited') || type !== data.type)) {
            return;
        }
        if (data.hasGoalList || data.goalId) {
            this._setContentParams({
                block: 'b-retargeting-condition-edit-rule',
                elem: 'params-' + modName,
                name: data.goalId ? BEMHTML.apply({
                    block: 'b-retargeting-condition-edit-rule',
                    elem: 'goal-name',
                    elemMods: {
                        type: data.type === 'ECOMMERCE_GOAL' ? 'ecommerce' : ''
                    },
                    domain: data.domain,
                    name: data.name,
                    counter: data.counter
                }) : this._BUTTON_LABEL_LIST[type],
                audienceTypeName: data.audienceTypeName,
                allowToUse: data.goalId ? data.allowToUse : true,
                selected: !!(data.goalId),
                day: data.day || u['b-retargeting-condition-edit-rule'].MAX_DAYS
            });
        } else {
            this._setContentParams({
                block: 'b-retargeting-condition-edit-rule',
                elem: 'params-' + modName + '-empty'
            });
        }
    },

    /**
     * Открытие списка
     */
    showListPopup: function() {
        this._dropdown && this._dropdown.getPopup().show(this._dropdown);
    },

    /**
     * Скрытие списка
     */
    hideListPopup: function() {
        this._dropdown && this._dropdown.getPopup().hide();
    },

    /**
     * Открывает список с целями
     * @param {{ goalList, goalSelectedInsideGroupRule, type }} data
     */
    openGoalList: function(data) {
        var popup;

        if (!this._dropdown) {
            return;
        }
        popup = this._dropdown.getPopup();

        if (!popup.isShown() ||  this.getType() !== data.type) {
            return;
        }

        data.goalList.forEach(function(goal) {
            if (data.goalId === goal.id) {
                goal.selected = true;
            } else if (this.indexOf(goal.id) !== -1) {
                goal.disabled = true;
            }
        }, data.goalSelectedInsideGroupRule);

        popup.setContent(
            BEMHTML.apply({
                block: 'b-retargeting-condition-edit-rule',
                elem: 'list-popup-content',
                elemMods: { type: this._TYPE_VS_MOD[data.type] },
                goalList: data.goalList,
                GOALS_MAX_LENGTH: this.LIST_POPUP_CONTENT_GOALS_MAX_LENGTH_FOR_VISIBLE_SEARCH
            }));

        popup.findBlockInside('b-chooser').on('select-item', this._onSelectItem, this);
        popup.findBlockInside('link').bindTo('click', this.hideListPopup.bind(this));
        popup.repaint();
    },

    /**
     * Применяет тип - рисует контролы выбранного типа
     * @param {{ hasGoalList: Boolean, type: String}} data
     */
    applyType: function(data) {
        this._setParams(data);
        this.setMod('status-load', 'loaded');
    },

    /**
     * Деструктор
     * @override;
     */
    destruct: function() {
        this._subscriptionManager.dispose();
        return this.__base.apply(this, arguments);
    },

    /**
     * Удаление
     * также удалить модификатор js_inited
     * @returns {Boolean}
     */
    _onRemoveRule: function() {
        this.trigger('remove-rule', this.params._entityId);
    },

    /**
     * Получить тип "правила"
     * @returns {string}
     */
    getType: function() {
        return this._fieldType.val();
    },

    /**
     * Создать и разместить контент.
     * Это либо поля выбора для выбранного типа цели, либо сообщение, что нужно создать цели / сегменты
     * либо индикатор загрузки данных
     * @param bemjson
     * @private
     */
    _setContentParams: function(bemjson) {
        BEM.DOM.update(this.elem('params'), BEMHTML.apply(bemjson));
    },

    /**
     * Валидация дней
     * @private
     */
    _validateDay: function() {
        return u['b-retargeting-condition-edit-rule'].isValidRule(
            this.getType(),
            this._fieldDay ? +this._fieldDay.val() : undefined
        );
    },

    /**
     * Изменение типа "правила"
     * @private
     */
    _onChangeType: function() {
        setTimeout(function() {
            this.setMod('type', this._TYPE_VS_MOD[this.getType()]);
        }.bind(this), 1);
    },

    /**
     * Изменение значения в поле ввода дней
     * @private
     */
    _changeFieldDay: function() {
        var day = +this._fieldDay.val(),
            valid = this._validateDay(day),
            msg = valid ? '' : iget2(
                'b-retargeting-condition-edit-rule',
                'kolichestvo-dney-dolzhno-byt-new',
                'Количество дней должно быть целым числом от 1 до 540'
            );

        this.elem('notice').text(msg);

        this._fieldDay.setMod('warning', valid ? '' : 'yes');

        if (valid) {
            this.elem('field-day-label').text(u.pluralizeWord([iget2('b-retargeting-condition-edit-rule', 'den', 'день'), iget2('b-retargeting-condition-edit-rule', 'dnya', 'дня'), iget2('b-retargeting-condition-edit-rule', 'dney', 'дней')], day));
        }
        this.trigger(
            'change-rule',
            {
                entityId: this.params._entityId,
                day: day,
                valid: valid
            }
        )
    },

    /**
     * Открывается список счетчиков целей/сегментов
     * Задать min ширину списку - не должен быть меньше owner
     * получить данные, и создать dom
     * @private
     */
    _onListOpen: function() {
        var popup = this._dropdown.getPopup();

        popup.setContent(
            BEMHTML.apply({
                block: 'b-retargeting-condition-edit-rule',
                elem: 'list-popup-content',
                elemMods: { 'loading': true }
            })
        );

        this.trigger('open-goal-list', {
            type: this.getType(),
            entityId: this.params._entityId
        });
    },

    /**
     * Обработчик события выбора элемента списка целей/сегментов
     * @private
     */
    _onSelectItem: function(e, params) {
        this._setListLabel(BEMHTML.apply({
            block: 'b-retargeting-condition-edit-rule',
            elem: 'goal-name-caption',
            name: params.search.name,
            audienceTypeName: params.search.audienceTypeName
        }));
        this.setMod(this.elem('list'), 'allow-to-use', 'yes');
        this.hideListPopup();
        this._dropdown.setMod('selected', 'yes');
        this.trigger('change-rule', {
             entityId: this.params._entityId,
             valid: this._validateDay(),
             goalId: +params.name,
             type: params.type,
             domain: params.goal.domain,
             name: params.goal.name,
             allowToUse: params.goal.allowToUse
        });
    },

    /**
     * Установить подпись на кнопке открытия списка счетчиков целей/сегментов
     * @param {String} name
     * @private
     */
    _setListLabel: function(name) {
        var btn;

        if (this._dropdown) {
            btn = this._dropdown.findBlockInside('button');

            BEM.DOM.update(btn.elem('text'), name);
            // btn.elem('text').text(name || this._BUTTON_LABEL_LIST[this.getType()]);
        }
    }
});
