/**
 * Редактирование существующей или создание новой записи условия подбора аудитории
 *
 * При каждом изменении в данных:
 * change(params)       params - объект с информацией о событии
 *      params - о чем сообщать:
 *      {
 *          mode: 'edit' || 'new'       редактирование или создание новой записи
 *          idChanged:  Boolean         есть изменения, которые нужно сохранить
 *          isValid:    Boolean         данные валидны
 *          saveAsNew:  Boolean         актуально для редактирования: запись можно сохранить как новую (изменено
 *                                      название и состав правил)
 *      }
 *
 * @event warning(reason)                   предупреждения - родитель покажет попап с alert
 *
 * @param {Object} [params.conditionId] id условие ретаргетинга, переданное для редактирования
 * Если нет - создание нового
 *
 *
 * @namespace b-retargeting-condition-edit
 */
BEM.DOM.decl({ name: 'b-retargeting-condition-edit'},  /** @lends b-retargeting-condition-edit */{

    /**
     * @type {RetargetingGoalDTO[]} загруженные цели/сегменты
     * необходимо сразу знать, есть ли выбор - что бы рендерить соответствующий элемент / показывать выбор
     */
    _goalList: null,

    /**
     * Флаги наличия целей определенного типа
     * @type {?{ METRIKA_GOAL, METRIKA_SEGMENT, AUDIENCE_SEGMENT }}
     */
    _hashGoalsByType: null,

    /**
     * @type {i-subscription-manager} управление подписками
     */
    _subscriptionManager: null,

    /**
     * Список доменов, которые есть у целей, используетмых в этом условии
     */
    _domains: null,

    /**
     * Открытый блок
     */
    _innerShownDropdown: null,

    /**
     * запрос за данными (показываем spin)
     * при получении данных изменяем модификатор и содержимое блока
     * вызвать метод блока с названием (blockName) для валидации данных
     */
    onSetMod: {

        /**
         * @this b-retargeting-condition-edit
         */
        js: function() {
            this._subscriptionManager = BEM.create('i-subscription-manager');
            this._dataProvider = u['retargeting-dataprovider'];

            this._getGoals()
                .then(function() {
                    this._initBlock();
                }.bind(this))
                .catch(function(e) {
                    BEM.DOM.update(
                        this.domElem,
                        BEMHTML.apply({
                            block: 'icon-text',
                            mods: { size: 'xs', theme: 'notice' },
                            text: {
                                block: 'b-retargeting-condition-edit',
                                elem: 'messages-reached-max-length-group-rule-text',
                                mix: {
                                    block: 'b-retargeting-condition-edit',
                                    elem: 'messages',
                                    mods: { notice: 'yes' }
                                },
                                content: iget2(
                                    'b-retargeting-condition-edit',
                                    'nevozmozhno-poluchit-spisok-dostupnyh',
                                    'Невозможно получить список доступных целей/сегментов. Повторите попытку через несколько минут.'
                                )
                            }
                        })
                    );
                }.bind(this));
        }
    },

    /**
     * Сообщает, было ли изменение, требующих сохранение
     * @returns {Boolean}
     */
    isChanged: function() {
        return this.model ? this.model.isChanged() : false;
    },

    /**
     * Возвращает текущие значения формы
     * @returns {Object}
     */
    getValue: function() {
        return {
            condition: this.model.toJSON(),
            domains: this._domains
        };
    },

    update: function(data) {
        this.model.update(data);
        this.model.fix();
    },

    /**
     * Деструктор
     * @override;
     */
    destruct: function() {
        this._subscriptionManager.dispose();
        this.model && this.model.destruct();
        return this.__base.apply(this, arguments);
    },

    /**
     * Инициализация блока
     * @private
     */
    _initBlock: function() {
        var elemCollection;

        this.model = this._getModel(this.params);

        this._updateReadOnlyStatus();
        this._renderBlock(this.model);

        this._initNameEvents();
        this._initGroupsEvents();

        this.model.get('groupRuleCollection').forEach(this._addGroupRuleToCollection, this);

        this._initModelEvents();

        // сброс загруженных данных целей/сегментов в период простоя
        this._subscriptionManager.on(BEM.channel('sys'), 'idle', function() {
            this._goalList = null;
        }, this);

        this.bindTo(this.elem('group-rule-collection'), 'scroll', function() {
            this._innerShownDropdown && this._innerShownDropdown.hide();
        });

        this.trigger('ready');
    },

    _initNameEvents: function() {
        this._subscriptionManager.on(
            this.findBlockInside('b-retargeting-condition-edit-name'),
            'change',
            this._onChangeName,
            this
        );
    },

    _initGroupsEvents: function() {
        var elemCollection;

        this.findBlockOn('create-group-rule', 'link').on('click', this._onCreateGroupRule, this);

        elemCollection = this.elem('group-rule-collection');
        this._subscriptionManager.wrap(BEM.blocks['b-retargeting-condition-edit-group-rule'])
            .on(elemCollection, 'create-rule', this._onCreateRule, this)
            .on(elemCollection, 'change-group-rule', this._onChangeGroupRule, this)
            .on(elemCollection, 'remove-group-rule', this._onRemoveGroupRule, this);

        this._subscriptionManager.wrap(BEM.blocks['b-retargeting-condition-edit-rule'])
             .on(elemCollection, 'remove-rule', this._onRemoveRule, this)
             .on(elemCollection, 'change-rule', this._onChangeRule, this)
             .on(elemCollection, 'change-rule-type', this._onChangeRuleType, this)
             .on(elemCollection, 'open-goal-list', this._onOpenGoalList, this);
    },

    _initModelEvents: function() {
        this.model.on('change', this._onChangeDataOfModel, this)
            .on('change-data', this._onChangeData, this)
            .on('change-message', function(e, messageObject) {
                this.updateMessage(messageObject);
            }, this);

        this.model.changeData();
    },

    _updateReadOnlyStatus: function() {
        this.model.set('isReadOnly', !this._goalList.length && this.model.get('mode') !== 'new');
    },

    /**
     * Создает и возврашает модель
     * @param {Object} condition
     * @param {Array} goalList
     * @private
     */
    _getModel: function(data) {

        var model = BEM.MODEL.getOne({
            id: data.modelId,
            name: data.modelName
        });

        if (!model) {
            throw 'Модель не найдена';
        }

        return model;
    },

    /**
     * Отрисовывает содержимое блока
     * @param {BEM.MODEL} model
     * @private
     */
    _renderBlock: function(model) {
        BEM.DOM.update(
            this.domElem,
            BEMHTML.apply(
                {
                    block: 'b-retargeting-condition-edit',
                    elem: 'content-loaded',
                    name: model.get('name'),
                    comment: model.get('comment'),
                    isReadOnly: model.get('isReadOnly')
                })
        );
    },

    /**
     * Получает список целей
     * Рассчитывает флаги наличия целей определенного типа
     * @param {String} [type] тип цели. если не передано - все цели
     * @returns {Promise<RetargetingGoalDTO[]>}
     * @private
     */
    _getGoals: function(type) {
        var goalListPromise;

        if (this._goalList) {
            goalListPromise = Promise.resolve(u._.cloneDeep(this._goalList));
        } else {
            goalListPromise = this._dataProvider
                .getGoalList()
                .then(this._convertGoals.bind(this))
                .then(this._filterAllowedGoals)
                .then(this._calcHashGoalsByType.bind(this))
                .then(function(goalList) {
                    this._goalList = goalList;
                    return u._.cloneDeep(goalList);
                }.bind(this));
        }

        return goalListPromise.then(function(goalList) {
            return type ?
                goalList.filter(function(goal) {
                    return u['retargeting-goal-dto'].matchGoalType(type, goal.type);
                }) :
                goalList;
        });
    },

    /**
     * Преобразует список целей в клиентский формат
     * @param {Array} goalList - список целей
     */
    _convertGoals: function(goalList) {
        return goalList.map(function(goalItem) {
            return u['retargeting-goal-dto'].RetargetingGoalDTO(goalItem);
        }, this)
    },

    /**
     * Фильтрует цели оставляя только доступные
     * @param {Array} goalList - список целей
     */
    _filterAllowedGoals: function(goalList) {
        return goalList.filter(function(goal) {
            return goal.allowToUse;
        });
    },

    _calcHashGoalsByType: function(goalList) {

        var hasEqualType = function(type) {
            return function(goal) {
                return type === goal.type;
            }
        };

        this._hashGoalsByType = {
            METRIKA_GOAL: goalList.some(hasEqualType('METRIKA_GOAL')),
            METRIKA_SEGMENT: goalList.some(hasEqualType('METRIKA_SEGMENT')),
            AUDIENCE_SEGMENT: goalList.some(hasEqualType('AUDIENCE_SEGMENT')),
            ECOMMERCE_GOAL: goalList.some(hasEqualType('ECOMMERCE_GOAL'))
        };

        return goalList;
    },

    /**
     * Обработчик события изменения данных в модели
     * @private
     */
    _onChangeDataOfModel: function() {
        this._getGoals().then(function(goals) {

            this.trigger('change', this.model.getParams());

            this._domains = Object.keys(this.model.get('groupRuleCollection').reduce(function(domains, groupRule) {
                groupRule.ruleCollection.forEach(function(rule) {
                    var domain = getDomain(rule.goalId);

                    return domain && (domains[domain] = true);
                });
                return domains;
            }, {}));

            function getDomain(goalId) {
                for (var i = 0; i < goals.length; i++) {
                    if (goals[i].id == goalId) {

                        return goals[i].domain || null;
                    }
                }

                return null;
            }

        }.bind(this));
    },

    /**
     * Изменяет данные в модели, влияющие на прогноз
     * @param {Event} e - не используем
     * @param {Object} info - данные события
     * @param {Boolean} info.isGroupsValid - валидны ли данные
     * @param {Boolean} info.isNegative - негативно ли условие
     * @param {Boolean} info.canPredict - можно ли сделать прогноз
     * @private
     */
    _onChangeData: function(e, info) {
        this.trigger('change-groups', info);
    },

    /**
     * Создает новую "группу правил" в редактируемом "условии"
     * @public
     */
    _onCreateGroupRule: function() {
        var el = this.elem('create-group-rule-wrap');

        if (this.hasMod(el, 'creating', 'yes')) {
            return;
        }

        this.setMod(el, 'creating', 'yes');

        BEM.DOM.append(
            el,
            BEMHTML.apply({
                block: 'b-retargeting-condition-edit',
                elem: 'content-loading'
            })
        );

        return this.model.createInstanceGroupRule()
            .then(function(groupRule) {
                var blockGroupRule = this._addGroupRuleToCollection(groupRule);
                u.scrollNodeTo(blockGroupRule.domElem, this.elem('group-rule-collection'));

                return blockGroupRule;
            }.bind(this))
            .catch(function(reason) {
                switch (reason) {

                    // достигнуто максимальное кол-во групп правил
                    case 'reached max length':
                        this.trigger('warning', {
                            msg: iget2(
                                'b-retargeting-condition-edit',
                                'dostignuto-maksimalnoe-kolichestvo-naborov',
                                'Достигнуто максимальное количество наборов правил в этом условии.'
                            )
                        });
                        break;

                    // не известная ошибка
                    default:
                        this.trigger('warning', {
                            msg: iget2('b-retargeting-condition-edit', 'ne-udalos-dobavit-novyy', 'Не удалось добавить новый набор правил')
                        });
                        break;
                }
            }.bind(this))
            .then(function() {
                var spin = this.findBlockInside('create-group-rule-wrap', 'spin');

                spin && spin.destruct();
                this.delMod(el, 'creating');

            }.bind(this));
    },

    /**
     * Обработчик события удаления группы правил
     * @param {Event} e
     * @param {String} entityId
     * @private
     */
    _onRemoveGroupRule: function(e, entityId) {

        if (this.model.get('groupRuleCollection').length > 1) {
            this.model.removeGroupRule(entityId);
            e.block.destruct();
        }

    },

    /**
     * Обработчик изменений в "группе правил"
     * @param {Event} e
     * @param {{ type, entityId }} params
     * @private
     */
    _onChangeGroupRule: function(e, params) {
        this.model.changeGroupRule(params);
    },

    /**
     * Добавляет "группу правил" в список
     * Добавляет каждому "правилу" поле hasGoalList, указывающее - есть ли доступные цели/сегменты выбранного типа
     * @param { {type, ruleCollection, _entityId } } groupRule
     * @private
     */
    _addGroupRuleToCollection: function(groupRule) {
        return BEM.DOM.append(
            this.elem('group-rule-collection'),
            BEMHTML.apply(
                {
                    block: 'b-retargeting-condition-edit-group-rule',
                    _entityId: groupRule._entityId,
                    type: groupRule.type,
                    ruleCollection: groupRule.ruleCollection.map(this._supplyRulesWithHasGoalList, this),
                    isReadOnly: this.model.get('isReadOnly')
                })
        )
            .bem('b-retargeting-condition-edit-group-rule');
    },

    /**
     * Обработчик события создания правила
     * Добавляет свойство
     * @param {Event} e
     * @param {String} groupRuleEntityId идентификатор "группы правил" в которой создать "правило"
     * @private
     */
    _onCreateRule: function(e, groupRuleEntityId) {
        var blockGroupRule = e.block;
        return this.model.createInstanceRule(groupRuleEntityId)
            .then(this._supplyRulesWithHasGoalList.bind(this))
            .then(blockGroupRule.addRuleToCollection.bind(blockGroupRule))
            .then(function() {
                this._changeCountRule();
                u.scrollNodeTo(blockGroupRule.elem('create-rule-wrap'), this.elem('group-rule-collection'), {
                    over: 50
                });

            }.bind(this))
            .catch(function(reason) {
                switch (reason) {

                    // достигнуто максимальное кол-во правил
                    case 'reached max length':
                        this.trigger('warning', {
                            msg: iget2(
                                'b-retargeting-condition-edit',
                                'dostignuto-maksimalnoe-kolichestvo-pravil',
                                'Достигнуто максимальное количество правил в этом наборе правил.'
                            )
                        });
                        break;

                    // не известная ошибка
                    default:
                        this.trigger('warning', {
                            msg: iget2('b-retargeting-condition-edit', 'ne-udalos-dobavit-novoe', 'Не удалось добавить новое правило')
                        });
                        break;
                }
            }.bind(this));
    },

    /**
     * Обработчик удаления правила
     * @param {Event} e
     * @param {String} ruleEntityId уникальный идентификатор "правила", которое удаляется
     * @private
     */
    _onRemoveRule: function(e, ruleEntityId) {
        if (this.model.removeRule(ruleEntityId)) {
           setTimeout(function() {
               e.block.destruct();
               this._changeCountRule();
           }.bind(this), 1);
        }
    },

    /**
     * Изменяет модификатор на блоке "группы правил" в зависимости от кол-ва "правил"
     * @private
     */
    _changeCountRule: function() {
        this.findBlocksInside('group-rule-collection', 'b-retargeting-condition-edit-group-rule')
            .forEach(function(block) {
                var ruleLength = block.findBlocksInside('rule-collection', 'b-retargeting-condition-edit-rule').length;

                if (ruleLength > 99) {
                    block.setMod('rule-length', 'hundreds');
                } else if (ruleLength > 9) {
                    block.setMod('rule-length', 'tens');
                } else {
                    block.setMod('rule-length', 'few');
                }
            }, this);
    },

    /**
     * Обработчик события изменений данных в "правиле"
     * @param {Event} e
     * @param {{ entityId, [type], [day], [goalId], [valid] }} params
     * @private
     */
    _onChangeRule: function(e, params) {
        this.model.changeRule(params.entityId, params);
    },

    /**
     * Обработчик события изменений типа "правила"
     * + будет запрос за данными и вызов метода "правила", у которого изменился тип
     * @param {Event} e
     * @param {{ entityId, [type], [valid] }} params
     * @private
     */
    _onChangeRuleType: function(e, params) {
        this.model.changeRule(params.entityId, params);
        this._getGoals(params.type)
            .then(function(list) {
                e.block.applyType(
                    {
                        type: params.type,
                        hasGoalList: this._hashGoalsByType[params.type]
                    });
            }.bind(this));
    },

    /**
     * Получает список целей указанного типа
     * @param {Event} e объект события
     * @param {{ type, entityId }} data
     * @returns {Promise<{ goalList, goalSelectedInsideGroupRule, type, goalId }>}
     * @private
     */
    _onOpenGoalList: function(e, data) {
        return this._getGoals(data.type)
            .then(function(list) {
                e.block.openGoalList({
                    type: data.type,
                    goalList: list,
                    goalId: this.model.getRuleGoalId(data.entityId),
                    goalSelectedInsideGroupRule: this.model.getGoalSelectedInsideGroupRule(data.entityId)
                });
            }.bind(this));
    },

    /**
     * Обработчик изменения в названии, коментарии
     * @param e не обрабатываем
     * @param {{ name, comment }} data
     * @private
     */
    _onChangeName: function(e, data) {
        this.model.set('name', data.name);
        this.model.set('comment', data.comment);
    },

    /**
     * Добавляет флаг наличия "целей/сегментов" такого же типа, как и "правило"
     * @param {RetargetingConditionRuleDTO} rule
     * @returns {{ RetargetingConditionRuleDTO, hasGoalList }} rule
     * @private
     */
    _supplyRulesWithHasGoalList: function(rule) {
        return u._.assign({
            hasGoalList: this._hashGoalsByType[rule.type]
        }, rule);
    },

    /**
     * Изменение сообщений/предупреждений которые видит пользователь
     * @param {Object} message данные
     * @private
     */
    updateMessage: function(message) {
        var errorsPresenter = this.findBlockOn('b-errors-presenter2');

        // к текущим сообщениям добавляем новые
        message = u._.extend(
            this.model.onChangeMessage(),
            message
        );

        BEM.DOM.replace(
            this.findElem('messages'),
            BEMHTML.apply({
                block: 'b-retargeting-condition-edit',
                elem: 'messages',
                message: message
            })
        );

        if (errorsPresenter && message.ERRORS_WHEN_SAVING) {
            errorsPresenter.showErrors(message.ERRORS_WHEN_SAVING.value);
        } else {
            errorsPresenter.clearErrors();
        }
    }

}, {
    live: function() {
        this
            .liveInitOnBlockInsideEvent('show', 'dropdown', function(e) {
                this._innerShownDropdown = e.block;
            })
            .liveInitOnBlockInsideEvent('hide', 'dropdown', function(e) {
                this._innerShownDropdown = null;
            })
    }

});
