/**
 * view model
 *
 * @event change-of-forecast        изменились данные, влияющие на прогноз посетителей
 * @event change-message            изменился список сообщений для пользователя
 * @event change-data(params)       изменились данные
 *
 * @namespace b-retargeting-condition-edit
 */
BEM.MODEL.decl('b-retargeting-condition-edit', {

    /**
     * "Режим работы"
     * edit - редактирование
     * new  - новая запись
     */
    mode: {
        type: 'string'
    },

    /**
     * id записи
     */
    id: {
        type: 'string',
        'default': ''
    },

    /**
     * Условие только для редактирования
     * Устанавливается блоком-владельцем
     */
    isReadOnly: {
        type: 'boolean',
        'default': false,
        validation: {
            validate: function(value) {
                return !value;
            }
        }
    },

    /**
     * Название
     */
    name: {
        type: 'string',
        'default': '',
        validation: {
            validate: function(value) {
                return typeof value === 'string' &&
                    value.length <= this.NAME_MAX_LENGTH && value;
            }
        }
    },

    /**
     * Комментарий
     */
    comment: {
        type: 'string',
        'default': '',
        validation: {
            validate: function(value) {
                return typeof value === 'string' &&
                    value.length <= this.COMMENT_MAX_LENGTH;
            }
        }
    },

    /**
     * Доступность всех "целей/сегментов", указанных условии
     */
    isAccessible: {
        type: 'boolean',
        'default': true,
        validation: {
            validate: function(value) {
                return typeof value === 'boolean';
            }
        }
    },

    /**
     * "Негативное" условие, имеет все groupRule только type=NONE
     */
    isNegative: {
        type: 'boolean',
        'default': false,
        validation: {
            validate: function(value) {
                return typeof value === 'boolean';
            }
        }
    },

    /**
     * Используется ли условие в кампаниях
     */
    isUsed: {
        type: 'boolean',
        'default': false
    },

    /**
     * Поле, указывающее, изменилилсь ли данные в groupRuleCollection
     */
    groupRuleCollectionIsChanged: {
        type: 'boolean',
        'default': false
    },

    /**
     * "группы правил"
     */
    groupRuleCollection: {
        type: 'array',
        'default': [],
        /**
         * Массив "групп правил" добавляем уникальные значения к каждой записи
         * @param value
         * @returns {*}
         * @this b-retargeting-condition-edit
         */
        preprocess: function(value) {
            this.model._originGroupRuleCollection = u._.cloneDeep(value);
            return value.map(this.model._supplyRuleGroupAndRulesWithIds, this.model);
        },
        validation: {
            /**
             * Валидация "групп правил" и "правил"
             * @returns {Boolean}
             * @this b-retargeting-condition-edit
             */
            validate: function() {
                return this._isGroupsValid();
            }
        }
    }
}, /** @lends b-retargeting-condition-edit */{

    /**
     * Максимальная длина названия
     * @type {Number}
     */
    NAME_MAX_LENGTH: 1000,

    /**
     * Максимальная длина комментария
     * @type {Number}
     */
    COMMENT_MAX_LENGTH: 1000,

    /**
     * Максимальное кол-во групп правил в условии
     */
    GROUP_RULE_MAX_LENGTH: 50,

    /**
     * Максимальное кол-во правил в группе правил
     */
    RULE_MAX_LENGTH: 250,

    /**
     * @type {Array} клонированные данные "группы правил" и "правила"
     * для сравнения - были ли изменения.
     * Например изменение порядка в массиве - не является изменением
     */
    _originGroupRuleCollection: null,

    /**
     * Сообщения: предупреждения и ошибки.
     * проверяется при каждом изменении в данных
     * у всех объектов есть свойство value, управляет им метод _onChangeMessage
     */
    _message: {
        /**
         * Условие можно использовать только для корректировок ставок
         * Условие показа:
         * все "группы правил" в "условии" имееют тип "NONE" (не выполнено ни одного)
         */
        ONLY_ADJUSTMENT: {

            /**
             * @type {Boolean} состояние сообщения
             */
            value: false,

            /**
             * @this b-retargeting-condition-edit
             * @returns {Boolean}
             */
            calc: function() {
                return !this.get('isReadOnly') &&
                    (this.get('mode') === 'edit' ? this.get('isNegative') : this.isConditionNegative());
            }
        },

        /**
         * У пользователя нет доступных целей или сегментов
         * Условие показа:
         * поле isReadOnly = true
         */
        READ_ONLY: {
            value: false,
            /**
             * @this b-retargeting-condition-edit
             * @returns {Boolean}
             */
            calc: function() {
                return this.get('isReadOnly');
            }
        },

        /**
         * Превышение кол-ва групп правил
         * Условие показа:
         * превышение кол-ва групп правил > GROUP_RULE_MAX_LENGTH
         */
        EXCEEDING_MAX_LENGTH_GROUP_RULE: {  // TODO выяснить почему
            value: false,            /**
             * @this b-retargeting-condition-edit
             * @returns {Boolean}
             */
            calc: function() {
                return !this.get('isReadOnly') && this.get('groupRuleCollection').length > this.GROUP_RULE_MAX_LENGTH;
            }
        },

        /**
         * Достигнуто максимальное кол-во групп правил
         * Условие показа:
         * кол-во групп правил === GROUP_RULE_MAX_LENGTH
         */
        REACHED_MAX_LENGTH_GROUP_RULE: {
            value: false,
            /**
             * @this b-retargeting-condition-edit
             * @returns {Boolean}
             */
            calc: function() {
                return !this.get('isReadOnly') && this.get('groupRuleCollection').length === this.GROUP_RULE_MAX_LENGTH;
            }
        },

        /**
         * Предупреждение - нельзя именить "негативное" условие на обычное
         */
        CHANGE_ONLY_ADJUSTMENT: {  // TODO выяснить почему
            value: false,
            /**
             * @this b-retargeting-condition-edit
             * @returns {Boolean}
             * @private
             */
            calc: function() {
                if (this.get('mode') === 'new' || this.get('isReadOnly')) {
                    return false;
                }
                return this.get('isNegative') && this.get('isNegative') !== this.isConditionNegative();
            }
        },

        /**
         * Предупреждение - нельзя именить обычное условие на "негативное"
         */
        NOT_CHANGE_SET_NEGATIVE: {   // TODO выяснить почему
            value: false,
            /**
             * @this b-retargeting-condition-edit
             * @returns {Boolean}
             * @private
             */
            calc: function() {
                if (this.get('mode') === 'new' || this.get('isReadOnly')) {
                    return false;
                }
                return !this.get('isNegative') && this.get('isNegative') !== this.isConditionNegative();
            }
        },

        /**
         * Уже используется в объявлениях. Изменение будет распостранено на них
         * кроме негативных условий
         */
        APPLY_ALL_CONDITIONS: {
            value: false,
            /**
             * @this b-retargeting-condition-edit
             */
            calc: function() {
                return this._isDataChanged() && !this.isConditionNegative() && !this.get('isReadOnly') && this.get('isUsed');
            }
        }
    },

    /**
     * Проверка достаточно ли данных для прогноза
     * Условие показа:
     * должен быть установлен минимум один выполненный сегмент или цель
     * @returns {*}
     */
    canPredict: function() {
        var result;

        if (this.get('isReadOnly')) {
            return true;
        }

        if (this.isConditionNegative()) {
            return true;
        }

        if (this.get('mode') === 'edit' && this.get('isNegative')) {
            result = true;
        } else {
            result = !this.get('groupRuleCollection').every(_fnCalcGroup);
        }

        return result;

        function _fnCalcGroup(groupRule) {
            return groupRule.type === 'NONE' || groupRule.ruleCollection.every(_fnCalcRule);
        }

        function _fnCalcRule(groupRule) {
            return !groupRule.goalId;
        }
    },

    /**
     * Возвращает параметры (состояние) блока
     * Сообщает подписчикам, какое состояние кнопок установить, можно ли сохранять
     * @returns {{mode: *, isChanged: *, isValid: *, saveAsNew: (*|boolean)}}
     */
    getParams: function() {
        return {
            mode: this.get('mode'),
            isChanged: this.isChanged(),
            isValid: this.isValid(),
            saveAsNew: this.shouldBeSavedAsNew()
        }
    },

    /**
     * Проверяет, можно ли сохранить редактируемую запись как новую
     * @returns {boolean}
     * @private
     */
    shouldBeSavedAsNew: function() {
       return this.get('mode') === 'edit' && this.get('groupRuleCollectionIsChanged') && this.isValid();
    },

    /**
     * Создает и добавляет новую "группу правил" к "условию"
     * @returns {Promise<Object>}
     * @public
     */
    createInstanceGroupRule: function() {
        if (this.get('groupRuleCollection').length >= this.GROUP_RULE_MAX_LENGTH) {
            return Promise.reject('reached max length');
        }

        return this._createInstanceGroupRule()
            .then(this._supplyRuleGroupAndRulesWithIds.bind(this))
            .then(this._addGroupRuleToCollection.bind(this));
    },

    /**
     * Определяет, является ли "условие" "негативным"
     * условие, у которого все группы правил имеют тип "NONE" - "не выполненно ни одного"
     * @returns {boolean}
     * @public
     */
    isConditionNegative: function() {
        return this.get('groupRuleCollection').every(function(groupRule) {
            return groupRule.type === 'NONE';
        });
    },

    /**
     * Обработчик изменения данных
     * Проверяет, изменились ли данные
     * @public
     */
    changeData: function() {
        if (this.get('mode') === 'new' && this.get('isNegative') != this.isConditionNegative()) {
            this.set('isNegative', this.isConditionNegative());
        }
        this.set('groupRuleCollectionIsChanged', this._isDataChanged());
        this.trigger('change-data', {
            isGroupsValid: this._isGroupsValid(),
            isNegative: this.isConditionNegative(),
            canPredict: this.canPredict(),
            isReadOnly: this.get('isReadOnly')
        });
        this._updateMessage();
    },

    /**
     * Удаляет "группу правил"
     * проверить возможность удаления - последнюю запись удалить нельзя
     * @param entityId
     * @returns {Boolean}
     * @public
     */
    removeGroupRule: function(entityId) {
        var groupRuleCollection = this.get('groupRuleCollection');

        groupRuleCollection.some(function(groupRule, index, arr) {
            if (this._isEntityHaveId(entityId, groupRule)) {
                arr.splice(index, 1);

                return true;
            }
        }, this);

        this.trigger('change');
        this.changeData();

        return true;
    },

    /**
     * Получает список выбранных "целей" внутри "группы правил"
     * @param ruleEntityId
     * @returns {String[]}
     * @public
     */
    getGoalSelectedInsideGroupRule: function(ruleEntityId) {
        return this._findRuleGroupWithRule(ruleEntityId).ruleCollection
            .map(function(rule) {
                return rule.goalId;
            })
            .filter(function(id) {
                return id;
            });
    },

    /**
     * Создает и добавляет новое "правило" в "группу правил"
     * @param {String} groupRuleEntityId  в какой "группе правил" создавать
     * @returns {Promise<{ type, day, valid }>}
     * @public
     */
    createInstanceRule: function(groupRuleEntityId) {
        if (this._findGroupRule(groupRuleEntityId).ruleCollection.length >= this.RULE_MAX_LENGTH) {
            return Promise.reject('reached max length');
        }

        return this._createInstanceRule()
            .then(this._implUnique)
            .then(this._addRule.bind(this, groupRuleEntityId));
    },

    /**
     * Удаляет "правило"
     * Последнюю запись удалить нельзя
     * @param {String} ruleEntityId
     * @returns {Boolean}
     * @public
     */
    removeRule: function(ruleEntityId) {
        var groupRule = this._findRuleGroupWithRule(ruleEntityId);

        if (groupRule.ruleCollection.length > 1) {
            groupRule.ruleCollection = groupRule.ruleCollection.filter(this._isEntityHaveOtherId.bind(this, ruleEntityId));
            this.trigger('change');
            this.changeData();

            return true;
        } else {
            return false;
        }
    },

    /**
     * Изменяет данные "правила"
     * при изменении типа - удалить goalId
     * @param entityId
     * @param {{ [type], [day], [goalId], [valid] }} ruleChange
     * @public
     */
    changeRule: function(entityId, ruleChange) {
        var rule = this._findRule(entityId);

        if (rule) {
            'type' in ruleChange && ruleChange.type !== rule.type && (rule.goalId = null);
            ['type', 'day', 'goalId', 'valid', 'name', 'domain', 'allowToUse']
                .forEach(
                    function(prop) {
                        prop in ruleChange && (this[prop] = ruleChange[prop]);
                    }, rule);
            this.trigger('change');
            this.changeData();
        }
    },

    /**
     * Изменяет данные в "группе правил"
     * @param {{ _entityId, type }} params
     * @public
     */
    changeGroupRule: function(params) {
        var groupRule = this._findGroupRule(params._entityId);
        groupRule && (groupRule.type = params.type);
        this.trigger('change');
        this.changeData();
    },

    /**
     * Возвращает выбранную "цель" у "правила"
     * @param {String} ruleEntityId
     * @returns {?Number}
     * @public
     */
    getRuleGoalId: function(ruleEntityId) {
        var rule = this._findRule(ruleEntityId);

        return rule && rule.goalId || null
    },

    /**
     * Проверка, есть ли заполненные правила
     * @returns {*}
     */
    hasRules: function() {
        var groupRuleCollection = this.get('groupRuleCollection');

        return groupRuleCollection.length &&
            groupRuleCollection.some(function(groupRule) {
                return groupRule.ruleCollection.some(function(rule) {
                    return rule.goalId && u['b-retargeting-condition-edit-rule'].isValidRule(rule.type, rule.day);
                })
            }, this)
    },

    /**
     * Хелпер, создающий объект новой "группы правил"
     * @returns {Promise<Object>}
     * @private
     */
    _createInstanceGroupRule: function() {
        return this._createInstanceRule()
            .then(function(rule) {
                return {
                    type: 'OR',
                    ruleCollection: [rule]
                };
            }.bind(this));
    },

    /**
     * Добавляет "правило" в коллекцию
     * @param groupRule
     * @returns {*}
     * @private
     */
    _addGroupRuleToCollection: function(groupRule) {
        this.get('groupRuleCollection').push(groupRule);
        this.trigger('change');

        return groupRule;
    },

    /**
     * Хелпер, создающий объект новой "правила"
     * Новые "правила" по умолчанию не валидны - ожидают указания в них данных (eg цель/сегмент)
     * @returns {{}}
     * @private
     */
    _createInstanceRule: function() {
        return Promise.resolve({
            type: 'METRIKA_GOAL',
            day: u['b-retargeting-condition-edit-rule'].MAX_DAYS,
            valid: false,
            goalId: 0,
            name: '',
            domain: '',
            allowToUse: true
        });
    },

    /**
     * Добавляет "правила" в коллекцию
     *
     * @param groupRuleEntityId - по нему находим правило, в которое добавляем
     * @param rule
     * @returns {Object|Boolean} успешно добавленный объект | либо false ( если во время создания пользователь успел
     *     удалить правило )
     * @private
     */
    _addRule: function(groupRuleEntityId, rule) {
        var groupRule = this.get('groupRuleCollection').filter(this._isEntityHaveId.bind(this, groupRuleEntityId))[0];

        groupRule && groupRule.ruleCollection.push(rule);
        this.trigger('change');
        this.changeData();

        return groupRule ? rule : false;
    },

    /**
     * Поиск "группы правил", которое содержит "правило" с указанным ruleEntityId
     * @param {String} ruleEntityId
     * @returns {*}
     * @private
     */
    _findRuleGroupWithRule: function(ruleEntityId) {
        return this.get('groupRuleCollection').filter(function(groupRule) {
            return groupRule.ruleCollection.some(this._isEntityHaveId.bind(this, ruleEntityId));
        }, this)[0];
    },

    /**
     * Найти объект "правило" по entityId
     * @param ruleEntityId
     * @raturns {?Object}
     * @private
     */
    _findRule: function(ruleEntityId) {
        var groupRule = this._findRuleGroupWithRule(ruleEntityId);

        return groupRule && groupRule.ruleCollection.filter(this._isEntityHaveId.bind(this, ruleEntityId))[0] || null;
    },

    /**
     * Поиск объекта "группы правил" по entityId
     * @param {String} entityId
     * @returns {{}}
     * @private
     */
    _findGroupRule: function(entityId) {
        return this.get('groupRuleCollection').filter(this._isEntityHaveId.bind(this, entityId))[0];
    },

    /**
     * Проверяет: имеет ли объект указанный entityId
     * @param entityId
     * @param obj
     * @returns {boolean}
     * @private
     */
    _isEntityHaveId: function(entityId, obj) {
        return obj._entityId === entityId;
    },

    /**
     * Проверяет: НЕ имеет ли объект указанный entityId
     * @param entityId
     * @param obj
     * @returns {boolean}
     * @private
     */
    _isEntityHaveOtherId: function(entityId, obj) {
        return !this._isEntityHaveId(entityId, obj);
    },

    /**
     * Добавляет уникальные значение к объектам "группа правил" и "правило"
     * что бы идентифицировать данные которые будут изменяться
     * @param {{ type, ruleCollection }} groupRule объект правила
     * @returns {Object}
     * @private
     */
    _supplyRuleGroupAndRulesWithIds: function(groupRule) {
        this._implUnique(groupRule);
        groupRule.ruleCollection.forEach(this._implUnique);

        return groupRule;
    },

    /**
     * Добавляет свойство с уникальным значением
     * @param {Object} obj
     * @returns {Object} возвратит тот же объект с добавленным свойством "_entityId"
     * @private
     */
    _implUnique: function(obj) {
        obj._entityId = u._.uniqueId();

        return obj;
    },

    /**
     * Проверяет валидны ли "группы правил" и "правила"
     * @returns {Boolean}
     * @private
     */
    _isGroupsValid: function() {
        var groupRuleCollection = this.get('groupRuleCollection');

        if (this.get('isReadOnly')) {
            return false;
        }
        if (this.get('groupRuleCollection').length > this.GROUP_RULE_MAX_LENGTH) {
            return false;
        }
        if (this.get('mode') === 'edit' && this.get('isNegative') !== this.isConditionNegative()) {
            return false;
        }

        return groupRuleCollection.length && groupRuleCollection.every(function(groupRule) {
                return groupRule.ruleCollection.every(
                    function(rule) {
                        return rule.goalId && u['b-retargeting-condition-edit-rule'].isValidRule(rule.type, rule.day);
                    })
            }, this);
    },

    /**
     * Проверяет, изменилилсь ли "группы правил" и "правила"
     * @returns {boolean}
     * @private
     */
     _isDataChanged: function() {
         return this._areConditionsEqual(this.get('groupRuleCollection'), this._originGroupRuleCollection);
     },

    /**
     * Сравнивает данных 2-х условий - есть ли изменения в данных
     * порядок "групп правил" и "правил" не имеет значения,
     * @param {Array} groupRuleCollection0
     * @param {Array} groupRuleCollection1
     * @returns {Boolean} true - данные изменились
     * @private
     */
    _areConditionsEqual: function(groupRuleCollection0, groupRuleCollection1) {
        return !areEqualArrays(groupRuleCollection0, groupRuleCollection1, areGroupRulesEqual);

        /**
         * Проверяет идентичность двух массивов. Порядок не важен
         * Каждому значению в ar0 найти соответствие в ar1.
         * Найденные соответствия в a1 не участвуют в дальнейшем поиске
         * @param {Array} ar0
         * @param {Array} ar1
         * @param {Function} fnEqual функция сравнения
         * @returns {Boolean} true массивы совпадают (порядок не важен)
         */
        function areEqualArrays(ar0, ar1, fnEqual) {
            if (ar0.length !== ar1.length) {
                return false;
            }
            var ar1cp = ar1.slice();

            return ar0.every(function(item0) {
                return ar1cp.some(function(item1, i) {
                    return fnEqual(item0, item1, fnEqual) && !!ar1cp.splice(i, 1);
                });
            }) && !ar1cp.length;
        }

        /**
         * Сравнивает 2-а объекта "групп правил"
         * @param {{type, ruleCollection}} gr0
         * @param {{type, ruleCollection}} gr1
         */
        function areGroupRulesEqual(gr0, gr1) {
            return gr0.type === gr1.type && areEqualArrays(gr0.ruleCollection, gr1.ruleCollection, areRuleEqual);
        }

        /**
         * Сравнивает 2-а объекта "правил"
         * @param {{type day, goalId}} r0
         * @param {{type, day, goalId}} r1
         */
        function areRuleEqual(r0, r1) {
            return r0.type === r1.type && r0.day === r1.day && r0.goalId === r1.goalId;
        }
    },

    /**
     * Обработчик, подготавливающий сообщения для пользователя
     * @private
     */
    onChangeMessage: function() {
        var msg, message = {}, key;

        for(key in this._message) {
            if (!this._message.hasOwnProperty(key)) {
                continue;
            }
            msg = this._message[key];
            if (msg.value != msg.calc.call(this)) {
                msg.value = !msg.value;
            }
            message[key] = msg.value && {
                value: msg.value
            };
        }
        return message;
    },

    /**
     * хелпер для  Обработчик, подготавливающий сообщения для пользователя
     * @private
     */
    _updateMessage: function() {
        this.trigger('change-message');
    }
});
