/**
 * @event b-retargeting-conditions-list#editCondition
 * @type {Object}
 * @property {String} conditionId идентификатор условия для редакторования
 *
 * @event b-retargeting-conditions-list#deleteCondition
 * @type {Object}
 * @property {String} conditionId идентификатор условия для удаления
 *
 * @fires b-retargeting-conditions-list#change событие изменения количества выбранных условий
 * @fires b-retargeting-conditions-list#updated-exist-condition изменение существующего условия
 * @fires b-retargeting-conditions-list#edit-сondition событие нажатия на кнопку редактирования одного из условий
 *
 * @fires b-retargeting-conditions-list#deleteCondition событие нажатия на кнопку удаления одного из условий
 * @fires b-retargeting-conditions-list#add-condition событие клика по кнопке добавить условие
 * @fires b-retargeting-conditions-list#added-new-condition добавлено новое условие
 */
BEM.DOM.decl('b-retargeting-conditions-list', {
    onSetMod: {
        js: function() {
            this._items = this.findBlocksInside('b-retargeting-conditions-list-item') || [];
            this._allChooser = this.findBlockOn('all-chooser', 'checkbox');
            this._checkboxesGroup = this.findBlockOn('b-checkboxes-group');

            this._initEvents();
        }
    },

    /**
     * Возвращает данные всех выбранных условий
     * @returns {Array}
     */
    getSelectedConditions: function() {
        return this.findBlocksInside({ block: 'b-retargeting-conditions-list-item', modName: 'selected', modVal: 'yes' })
            .map(function(item) {
                return item.params.condition;
            });
    },

    /**
     * Устанавливает/сбрасывает прелоадинг удаления условия ретаргетинга
     * @param {Boolean} isLoading флаг, определяющий остановить или запустить прелоадинг
     * @param {String} conditionId идентификатор условия
     */
    setDeleteConditionLoading: function(isLoading, conditionId) {
        this._getConditionBlockById(conditionId).setMod('delete-progress', isLoading ? 'yes' : '');
    },

    /**
     * Удаляет условие ретаргетинга из списка
     * @param {String} conditionId идентификатор условия
     */
    deleteCondition: function(conditionId) {
        var conditionBlock = this._getConditionBlockById(conditionId),
            conditionIsSelected = conditionBlock.getMod('selected') === 'yes';

        conditionBlock.destruct();

        conditionIsSelected && this.trigger('change');

        this._onItemsListChange();

        this._repaintConditionNumbers();
    },

    /**
     * Обновляет условие ретаргетинга в списке в соответствиии с новыми данными
     * @param {Object} conditionData данные условия
     */
    updateCondition: function(conditionData) {
        var conditionBlock = this._getConditionBlockById(conditionData.ret_cond_id),
            isSelected = conditionBlock.getMod('selected');

        if (this.params.singleChoice) {
            this.repaintItems(conditionData);
        } else {
            BEM.DOM.replace(
                conditionBlock.domElem,
                BEMHTML.apply({
                    block: 'b-retargeting-conditions-list-item',
                    mods: {
                        // Если в условии есть НЕ доступные цели подсвечиваем условие
                        goals: conditionData.is_accessible ? undefined : 'unavailable',
                        // Дизейблим условия, доступные только для корректировок ставок
                        'for-rates-adjustment-only': conditionData.is_negative ? 'yes' : '',
                        id: conditionData.ret_cond_id,
                        selected: isSelected
                    },
                    condition: conditionData,
                    index: conditionBlock.getIndex()
                })
            );
        }

        this._onItemsListChange();
        this.trigger('updated-exist-condition', conditionData);
    },

    /**
     * Добавляет новое условие ретаргетинга к списку
     * @param {Object} conditionData данные условия
     */
    addCondition: function(conditionData) {
        var isNegative = conditionData.is_negative,
            domElem;

        if (this.params.singleChoice) {
            this.repaintItems(conditionData, isNegative || conditionData.ret_cond_id);
            domElem = this._getConditionBlockById(conditionData.ret_cond_id).domElem;   
        } else {
            domElem = BEM.DOM.append(
                this.elem('body'),
                BEMHTML.apply({
                    block: 'b-retargeting-conditions-list-item',
                    mods: {
                        // Если в условии есть НЕ доступные цели подсвечиваем условие
                        goals: conditionData.is_accessible ? undefined : 'unavailable',
                        // Дизейблим условия, доступные только для корректировок ставок
                        'for-rates-adjustment-only': isNegative ? 'yes' : '',
                        id: conditionData.ret_cond_id,
                        selected: isNegative ? '' :'yes'
                    },
                    condition: conditionData,
                    index: this._items.length + 1
                }));
        }
        u.scrollNodeTo(domElem, domElem.parent());
        this._onItemsListChange();
        this._onChangeChooserState();
        this.trigger('added-new-condition', conditionData);
    },

    /**
     * Перерисовать элементы
     * для радио-баттонов. при добавлении нового или перерисовке одного - он вылетает из группы и чекается отдельно
     * лечим - перерисовываем блок с радиобаттонами полностью
     *
     * @param conditionNew {Object} новое условие
     * @param [idSelected] {Number} выделенная строка
     */
    repaintItems: function(conditionNew, idSelected) {
        var isUpdate,
            blockRadio,
            conditions = this.findBlocksInside({ block: 'b-retargeting-conditions-list-item' })
                .map(function(item) {
                    var condition;
                    if (item.params.condition.ret_cond_id == conditionNew.ret_cond_id) {
                        isUpdate = true;
                        condition = conditionNew;
                    } else {
                        condition = item.params.condition;
                    }
                    condition.isSelected = item.hasMod('selected', 'yes');

                    return condition;
                });
        if (!isUpdate) {
            conditions.push(conditionNew);
        }
        BEM.DOM.update(
            this.elem('body'),
            BEMHTML.apply({
                block: 'b-retargeting-conditions-list',
                elem: 'items-template',
                conditions: conditions,
                singleChoice: this.params.singleChoice
            })
        );
         idSelected && (blockRadio = this.findBlockInside('radiobox')) && blockRadio.val(idSelected);
    },

    /**
     * Изменить видимость элемента - подсказка по гео-таргетингу
     * @param {Boolean} visible показать / скрыть элемент
     */
    setVisibleGeoHint: function(visible) {
        this.setMod(this.elem('geo-hint'), 'hidden', visible ? '' : 'yes');
    },

    /**
     * Удаляет блок и подписки на события
     */
    destruct: function() {
        BEM.blocks['b-retargeting-conditions-list-item'].un(this.domElem, 'change editCondition deleteCondition');
        this._iSearchable && this._iSearchable.destruct();
        this.findBlockOn('search', 'input').un('change');
        return this.__base.apply(this, arguments);
    },

    /**
     * Подписывается на события контроллов
     * @private
     */
    _initEvents: function() {
        if (!this.elem('body').length) {
            return;
        }

        // событие change при выборе всех условий сразу может генериться большое количество раз
        // поэтому используем debounce
        BEM.blocks['b-retargeting-conditions-list-item']
            .on(this.domElem, 'change', $.debounce(this._onChangeChooserState, 200, this));

        BEM.blocks['b-retargeting-conditions-list-item'].on(this.domElem, 'editCondition', this._onEditButtonClicked, this);
        // BEM.blocks['b-retargeting-conditions-list-item'].on(this.domElem, 'editCondition', this._onEdit, this);
        BEM.blocks['b-retargeting-conditions-list-item'].on(this.domElem, 'deleteCondition', this._onDeleteButtonClicked, this);

        this._initSearchField();

        // скрываем открытые попапы со списком кампаний, закрываем открытые подсказки влож при скроле
        this.bindTo('body', 'scroll', $.debounce(function() {
            this._items.forEach(function(item) {
                item.hideCampaignDropdown();
                item.hideAdjustmentTooltip();
            });
        }, 300, true, this));

        this.bindTo('add', 'click', function() {
            this.trigger('add-condition');
        });
    },

    /**
     * Инициализирует поле для поиска
     * @private
     */
    _initSearchField: function() {
        var searchableItems = this._getSearchableItems();

        this._iSearchable = BEM.create({ block: 'i-searchable' }, { items: searchableItems })
            .on('search.found', this._onSearchFound, this)
            .on('search.missed', this._onSearchMissed, this)
            .on('search.finish', this._onSearchFinish, this);

        this._iSearchable.matchRow = function(row, iterator) {
            var textToSearch = $.trim(iterator.query.text).toLowerCase(),
                textToSearchIn = row.name.toLowerCase();

            return textToSearchIn.indexOf(textToSearch) !== -1;
        };

        this.findBlockOn('search', 'input').on('change', $.debounce(function(event) {
            this._iSearchable.search({ text: event.block.val() });
        }, 300, this), this);
    },

    /**
     * Возвращает массив вариантов поиска для блока i-searchable
     * @private
     * @returns {Array}
     */
    _getSearchableItems: function() {

        return this._items.map(function(item) {

            return {
                block: item,
                name: item.params.condition.condition_name
            }
        }, this);
    },

    /**
     * Обработчик события успешного соотвествия одного из вариантов для поиска условию поиска
     * @private
     */
    _onSearchFound: function(event, data) {
        data.item.block.delMod('hidden');
    },

     /**
      * Обработчик события неуспешного соотвествия одного из вариантов для поиска условию поиска
      * @private
      */
    _onSearchMissed: function(event, data) {
         data.item.block.setMod('hidden', 'yes');
    },

    /**
     * Обработчик события завершения поиска среди возможных вариантов
     * @private
     */
    _onSearchFinish: function() {
        var hiddenItemsNumber = (this.findBlocksInside({ block: 'b-retargeting-conditions-list-item', modName: 'hidden', modVal: 'yes' }) || []).length;

        this.setMod(this.elem('search-empty'), 'hidden', hiddenItemsNumber === this._items.length ? '' : 'yes');

        this._allChooser && this._allChooser.setMod('disabled', hiddenItemsNumber ? 'yes' : '');
    },

    /**
     * Обработчик события изменения списка условий ретаргетинга
     * @private
     */
    _onItemsListChange: function() {
        this._items = this.findBlocksInside('b-retargeting-conditions-list-item') || [];

        this._checkboxesGroup && this._checkboxesGroup.rearrange();

        this._iSearchable.updateItems(this._getSearchableItems());

        this.setMod(this.elem('list-controls'), 'hidden', this._items.length ? '' : 'yes');
        this.setMod(this.elem('empty'), 'hidden', this._items.length ? 'yes' : '');
    },

    /**
     * Обработчик события изменения выбранности одного из условий ретаргетинга
     * @private
     */
    _onChangeChooserState: function() {
        this._checkboxesGroup && this._checkboxesGroup.rearrange();
        this.trigger('change');
    },

    /**
     * Обработчик события нажатия на кнопку редактрования условия
     * @param {Event} event
     * @param {Object} eventData
     * @param {Object} eventData.conditionId идентификатор условия ретаргетинга
     * @private
     */
    _onEditButtonClicked: function(event, eventData) {
        this.trigger('edit-condition', { conditionId: eventData.conditionId });
    },

    /**
     * Обработчик события нажатия на кнопку удаления условия
     * @param {Event} event
     * @param {Object} eventData
     * @param {Object} eventData.conditionId идентификатор условия ретаргетинга
     * @private
     */
    _onDeleteButtonClicked: function(event, eventData) {
        this.trigger('deleteCondition', { conditionId: eventData.conditionId });
    },

    /**
     * Возвращает блок условия по его id
     * @param {String} conditionId идентификатор условия
     * @private
     * @returns {BEM} блок соответсвующий условию
     */
    _getConditionBlockById: function(conditionId) {
        return this.findBlockInside({ block: 'b-retargeting-conditions-list-item', modName: 'id', modVal: '' + conditionId });
    },

    /**
     * Пересчитывает номера условий в списке
     * @private
     */
    _repaintConditionNumbers: function() {
        this._items.forEach(function(item, n) {
            item.updateIndex(n + 1);
        });
    }
});
