/**
 * Блок выбора стратегий
 * Параметры:
 * {Array} params.strategies – массив идентификаторов доступных для выбора стратегий и задизейбленных платформ
 * {String} params.strategyModel - данные стратегии в обработанном для блока виде
 * События:
 * @fires cancel – событие отмены изменения стратегии
 * @fires save – событие сохранения изменений стратегии
 */
BEM.DOM.decl({ name: 'b-strategy2-choose', baseBlock: 'i-glue' }, {
    //Список стратегий для которых требуется перерисовка DOM
    SETTINGS_TO_REDRAW: ['opt-cpa'],

    onSetMod: {

        js: function() {
            // сохраняем данные, пришедшие от кампании - они сохранены
            var strategyModelData = this.params.strategyModel;

            this._subscriptionManager = BEM.create('i-subscription-manager');
            this._applyData(strategyModelData, this.params.dayBudgetSettings);

            this._initControlButtons();
            this._tabs = this.findBlockInside('tabs');

            this.__base.apply(this, arguments);

            this._drawChooser(this._savedData.platform, this._savedData.name);

            this._tabs && this._subscriptionManager
                .on(this._tabs, 'change', function(e, data) {
                    var platform;

                    switch (data.currentIndex) {
                        case 0: platform = u.strategy.ALL; break;
                        case 1: platform = u.strategy.SEARCH; break;
                        case 2: platform = u.strategy.NET; break;
                    }

                    this.model.set('platform', platform);
                }, this);

            // слушаем смену состояния модели
            this._subscriptionManager.wrap(this.model)
                .on('strategyName', 'change', this._onNameChange, this)
                .on('platform', 'change', this._onPlatformChange, this);

            // инициируем модель
            this.model.update({ strategyName: this._savedData.name, platform: this._savedData.platform }).fix();

        },

        loading: function(modName, modVal) {
            this.onLoading(modVal == 'yes');
        }
    },

    onElemSetMod: {
        'strategy-settings': {
            show: function(elem, modName, modVal) {
                var shownElem = this.getMod(elem, 'type'),
                    strategy = this.findElem('strategy-settings', 'type', shownElem),
                    settings = this.findBlockInside(strategy, 'b-strategy2-settings');
                settings.setMod('show', modVal);
            }
        }
    },

    onLoading: function(isLoading) {
        var disabled = isLoading ? 'yes' : '';

        this.trigger('loading', { isLoading: isLoading });

        this._saveButton
            .setMod('disabled', disabled)
            .setText(iget2('b-strategy2-choose', 'sohranit', 'Сохранить') + (isLoading ? '…' : ''));

        this._cancelButton.setMod('disabled', disabled);
    },

    /**
     * Реакция при клике на кнопку Отмена
     */
    onCancelBtnClick: function() {
        this.trigger('cancel');
    },

    /**
     * Реакция при клике на кнопку Сохранить
     */
    onSaveBtnClick: function() {
        var validateResult = this.validate();

        if (validateResult && validateResult.isValid) {
            var strategyValue = this.getValue();

            this.trigger('save', strategyValue);
        }
    },

    /**
     * Возврашает текущее значение блока
     * @returns {{name: *, platform: *, options: (*|Object|{name, options}|{name: string, options: {}}), dayBudgetData:
     *     (*|Boolean|string|{active, set, sum, show_mode})}}
     */
    getValue: function() {
        var settingsBlock = this.findBlockInside(this.findElem('strategy-settings', 'show', 'yes'), 'b-strategy2-settings'),
            isDayBudgetEnabled = settingsBlock.isDayBudgetEnabled(),
            isCpaPayForConversionEnabled = settingsBlock.isCpaPayForConversionEnabled(),
            isContextLimitEnabled = settingsBlock.isContextLimitEnabled(),
            platform = this.model.get('platform'),
            campaignType = this.params.campaignType,
            strategyOptions = settingsBlock.getOptions(platform),
            coverageWeekMode = strategyOptions.coverageWeekMode;

        return {
            coverageWeekMode: coverageWeekMode,
            name: settingsBlock.getName(),
            platform: platform,
            options: strategyOptions,
            dayBudgetData: isDayBudgetEnabled && settingsBlock._dayBudget.getData(),
            contextLimit: isContextLimitEnabled ? settingsBlock.getContextLimit() : 0,
            enable_cpc_hold: isContextLimitEnabled ? settingsBlock.getEnableCpcHold() : 1,
            isCpaPayForConversionEnabled: isCpaPayForConversionEnabled,
            isChanged: settingsBlock.model.isChanged(),
            attributionModel: u['attribution-model'].isNewPolicyEnabled(campaignType) ? settingsBlock.model.get('attributionModel') : null
        }
    },

    /**
     * кеширует и пробрасывает события дневного бюджета,
     * которые пробрасываются из директа/коммандера
     */
    setDayBudgetEvent: function(event) {
        this._onMaximumDailyChangeCountExceed = event;

        this._setDayBudgetEvent(BEM.blocks['b-strategy2-settings'].getModNameByStrategyName(this.model.get('strategyName'), this.params.campaignType, this._savedData.options))
    },

    /**
     * Определяет доступность настроек дневного бюджета для текущей стратегии
     * @returns {Boolean}
     */
    isDayBudgetEnabled: function() {
        var strategy = this.findBlockInside(this.findElem('strategy-settings', 'show', 'yes'), 'b-strategy2-settings');
        return strategy && strategy._dayBudget &&
            this.getMod(this.elem('settings'), 'with-day-budget') === 'yes';
    },

    /**
     * Проверяет были ли изменения в модели стратегии
     * @returns {Boolean}
     */
    isChanged: function() {
        return this.model.isChanged() ||
            this.findBlockInside(this.findElem('strategy-settings', 'show', 'yes'), 'b-strategy2-settings')
                .model.isChanged();
    },

    /**
     * Проверяет настройки стратегии (возвращает true в случае успеха)
     * и сохраняет дневной бюджет
     * @returns {Object}
     */
    save: function() {
        //@heliarian оставлено для совместимости со старым кодом
        return this.validate();
    },

    /**
     * Проверяет валидность стратегии и дневного бюджета
     * @returns {{isValid: (*|boolean), isDayBudgetEnabled: (*|Boolean)}}
     * @private
     */
    validate: function() {
        var settingsBlock = this.findBlockInside(this.findElem('strategy-settings', 'show', 'yes'), 'b-strategy2-settings'),
            isValid = settingsBlock.validate(),
            isDayBudgetEnabled = settingsBlock.isDayBudgetEnabled(),
            isDayBudgetValid = !isDayBudgetEnabled || settingsBlock._dayBudget.validate();

        if (isValid && isDayBudgetValid) {
            return { isValid: isValid && isDayBudgetValid, isDayBudgetEnabled: isDayBudgetEnabled };
        }
    },

    /**
     * Деструктор
     * @override
     */
    destruct: function() {
        this._subscriptionManager.dispose();
        this.model.destruct();

        this.__base.apply(this, arguments);
    },

    /**
     * Показать настройки стратегии
     * @param {String} name – название (идентификатор стратегии)
     */
    _showStrategySettings: function(name) {
        var modName = BEM.blocks['b-strategy2-settings'].getModNameByStrategyName(name, this.params.campaignType, this._savedData.options),
            elem = this.findElem('strategy-settings', 'type', modName),
            data = this._getSettingData(name),
            templateSetting = function() {
                return this._getSettingHtml(modName, name, data);
            }.bind(this),
            settings;

        this.delMod(this.findElem('strategy-settings', 'show', 'yes'), 'show');

        if (elem.length) {
            this.setMod(elem, 'show', 'yes');

            if (u._.includes(this.SETTINGS_TO_REDRAW, modName)) {
                BEM.DOM.replace(this.findElem('strategy-settings', 'type', modName), templateSetting());
            } else {
                this.findBlockOn(elem, 'b-strategy2-settings').clearErrorMessages();
            }
        } else {
            BEM.DOM.append(this.elem('strategies-settings-panel'), templateSetting());
        }

        settings = this.findBlockOn(this.findElem('strategy-settings', 'type', modName), 'b-strategy2-settings');
        settings.setMod('platform', this.model.get('platform'));
        this._setDayBudgetEvent(modName);

        this.setMod(this.elem('settings'), 'with-day-budget',
            settings.isDayBudgetEnabled() ? 'yes' : '');
    },

    /**
     * Пробрасывает событие про дневной бюджет стратегии, если он у нее есть
     * @param {String} modName - модификатор стратегии
     * @private
     */
    _setDayBudgetEvent: function(modName) {
        var settings = this.findBlockOn(this.findElem('strategy-settings', 'type', modName), 'b-strategy2-settings');

        if (settings.isDayBudgetEnabled() && settings.setDayBudgetEvent) {
            settings.setDayBudgetEvent(this._onMaximumDailyChangeCountExceed)
        }
    },

    /**
     * Инициализируем кнопки с контроллами
     * @returns {BEM}
     * @private
     */
    _initControlButtons: function() {
        this._saveButton = this.findBlockOn('save', 'button');
        this._cancelButton = this.findBlockOn('cancel', 'button');

        this._subscriptionManager
            .on(this._saveButton, 'click', this.onSaveBtnClick, this)
            .on(this._cancelButton, 'click', this.onCancelBtnClick, this);

        return this;
    },

    /**
     * сохранение данных от сервера во внутренние поля
     * @param {Object} strategyModelData - данные о стратегии
     * @param {Object} dayBudgetModelData - настройки дневного бюджета
     * @private
     */
    _applyData: function(strategyModelData, dayBudgetModelData) {
        if (!!strategyModelData) {
            this._savedData = u._.assign({}, strategyModelData);
            this._savedData.dayBudget = dayBudgetModelData;
        }
    },

    /**
     * Обработчик события изменения поля name модели
     * @private
     */
    _onNameChange: function() {
        var currentStrategyName = this.model.get('strategyName');

        // не отрисовываем стратегии дважды при инициализации моделей
        if (this.model.get('platform') != '') {
            this._showStrategySettings(currentStrategyName);
            this._updateChooser(currentStrategyName, this.model.get('platform'));
        }
    },

    /**
     * Обновление шапки - по кликам на табы
     * @param {Event} e - событие изменения таба
     * @param {Object} data - данные (номер выбранного таба)
     * @private
     */
    _onPlatformChange: function(e, data) {
        var platform = data.value,
            name = this.model.get('strategyName'),
            mediaType = this.params.campaignType,
            options = this._savedData.options,
            goal_id = options.goal_id || (options.search || {}).goal_id || (options.net || {}).goal_id,
            platformId;

        switch (platform) {
            case u.strategy.ALL: platformId = 0; break;
            case u.strategy.SEARCH: platformId = 1; break;
            case u.strategy.NET: platformId = 2; break;

        }

        if (name === u.strategy.AUTOBUDGET_WEEK_BUNDLE) {
            name = u.strategy.AUTOBUDGET_AVG_CLICK;
        } else if (name === u.strategy.AUTOBUDGET) {
            name = goal_id ?
                mediaType === 'mobile_content' ? u.strategy.AUTOBUDGET_AVG_CPI : u.strategy.AUTOBUDGET_AVG_CPA :
                u.strategy.AUTOBUDGET_AVG_CLICK;
        }

        this._tabs && this._tabs.activate(platformId);

        this._drawChooser(platform, name);
        this._updateChooser(name, platform);
        this._showStrategySettings(this.model.get('strategyName'));
    },

    /**
     * Устанавливаем галочку напротив выбранной стратегии
     * @param {String} name - имя текущей стратегии
     * @param {String} platform - текущая платформа
     * @private
     */
    _updateChooser: function(name, platform) {
        var chooser = this.findBlockInside('b-chooser');

        chooser.check(u.beminize(name));
    },

    /**
     * отрисовка блока чузера со стратегиями
     * @param {String} platform - в сетях, на поиске, везде
     * @param {String} currentName - имя текущей стратегии
     * @private
     */
    _drawChooser: function(platform, currentName) {
        var strategies = {},
            paramsStrategies = this.params.strategies,
            campaignType = this.params.campaignType,
            // в рамках DIRECT-72782: Бюджет на произвольный период
            // добавлены стратегии которые не размещаются на отдельном табе, а лишь являются частным случаем
            parentStrategyName = u.strategy.getParent(currentName, this.params.strategyModel.options) || currentName,
            strategyMap = function(value) {
                var isDisabled = u._.includes(paramsStrategies.disabledStrategies, value),
                    title = u.strategy.getTitleByName(
                        value,
                        undefined,
                        campaignType,
                        {},
                        this.params.isChangeManualStrategyName
                    );

                if (isDisabled) {
                    var info = u.strategy.getDisabledInfo(value);

                    title = [
                        title,
                        {
                            block: 'b-icon-hint',
                            mix: {
                                block: 'b-strategy2-choose',
                                elem: 'disable-info-icon'
                            },
                            icon: { 'size-13': 'question', theme: 'light' },
                            content: {
                                block: 'b-strategy2-choose',
                                elem: 'disable-info-tooltip',
                                content: info
                            }
                        }
                    ];
                }

                return {
                    name: value,
                    title: title,
                    selected: value === parentStrategyName,
                    disabled: isDisabled
                };
            };

        strategies.handStrategies = (u._.compact(paramsStrategies.hand) || [])
            .map(strategyMap, this);

        strategies.autoStrategies = (u._.compact(paramsStrategies.auto) || [])
            .map(strategyMap, this);

        strategies.autoFirst = paramsStrategies.autoFirst;

        BEM.DOM.update(this.elem('chooser-pane'), BEMHTML.apply({
            block: 'b-strategy2-choose',
            mods: u._.omit(this.getMods(), ['js']),
            elem: 'strategies',
            strategies: strategies
        }));
        this._bindChooserEvents(this.findBlockInside('b-chooser'));
    },

    /**
     * Привязка событий чузера
     * @private
     */
    _bindChooserEvents: function(chooser) {
        this._subscriptionManager
            .on(chooser, 'change', function(e, data) {
                if (data.selected){
                    //b-chooser превращает все даши (_) в именах в дефисы (-)
                    //поэтому сохраняем исходное название в extraParams
                    var name = data.extraParams.origName;

                    this.model.set('strategyName', name);
                }
            }, this);
    },

    /**
     * Проверяет, что текущая сохраненная стратегия отображается как max_clicks
     * Это верно, если выбрана одна из "ручных" стратегий, кроме максимального охвата
     * @param {Object} options - сохраненные опции
     * @returns {boolean}
     * @private
     */
    _isMaxClicks: function(options) {
        var currentName  = this._savedData.dataName,
            searchName = options.search && options.search.name,
            handStrategies = [u.strategy.DEFAULT];

        return handStrategies.indexOf(currentName) >= 0 || (currentName === u.strategy.DIFFERENT_PLACES && (handStrategies.indexOf(searchName) >= 0));

    },

    /**
     * Возвращает данные для отображения стратегии
     * @param {String} name – название (идентификатор стратегии)
     * @returns {Object}
     */
    _getSettingData: function(name) {
        var savedData = this._savedData,
            jsonCampaign = savedData.campaign,
            isDifferentPlaces = savedData.isDifferentPlaces,
            campaignType = this.params.campaignType,
            innerName = isDifferentPlaces ? savedData.options.search.name : savedData.dataName,
            // является ли стратегия ручной "максимум кликов", для которой
            // нужны отдельные данные
            isMaxClicksStrategy = name == u.strategy.MAXIMUM_CLICKS || this._isMaxClicks(savedData.options),
            commonOptions = {
                name: name,
                title: u.strategy.getTitleByName(name, undefined, campaignType, savedData.options, this.params.isChangeManualStrategyName),
                currency: jsonCampaign.currency || 'YND_FIXED',
                metrika: jsonCampaign.metrika || {},
                rmpCounters: jsonCampaign.rmpCounters || {},
                mediaType: jsonCampaign.mediaType,
                where: savedData.where,
                isNewCamp: jsonCampaign['new_camp'],
                isSaved: name === savedData.dataName,
                minGoalsOnCampaign: savedData.minGoalsOnCampaign,
                dayBudgetSettings: savedData.dayBudget,
                // только для РМП
                isTrackUrlNeeded: jsonCampaign.isNewRmpEnabled,
                mobileAppHasTrackUrl: u._.get(jsonCampaign, 'selected_mobile_app.trackers', []).length > 0,
                meaningfulGoals: jsonCampaign.meaningful_goals,
                lastStrategyChange: savedData.lastStrategyChange
            };

        // обработка ситуации, когда у нас была сохранена только сетевая стратегия
        // и мы переключаемся на другой радиобаттон
        if (innerName == 'stop') {
            innerName = savedData.options.net.name;
            isDifferentPlaces = false;
            // если была стратегия максимум кликов, то надо все галочки сбросить
            if (innerName == u.strategy.MAXIMUM_COVERAGE) {
                innerName = 'default';
            }
        }

        if (isMaxClicksStrategy) {
            return u._.assign(
                commonOptions, {
                    isDifferentPlaces: isDifferentPlaces,
                    platform: this.model.get('platform'),
                    innerName: innerName,
                    options: savedData.options,
                    savedName: savedData.name
                });
        }

        return commonOptions;
    },

    /**
     * Формирует HTML для отображения стратегии
     * @param {String} modName – название модификатора настройки
     * @param {String} strategyName – название (идентификатор стратегии)
     * @param {Object} modelData – данные для отображения стратегии
     * @returns {Boolean}
     */
    _getSettingHtml: function(modName, strategyName, modelData) {
        var savedData = this._savedData,
            mediaType = this.params.campaignType,
            platform = this.model.get('platform'),
            isNetStrategy = platform === u.strategy.NET && mediaType !== 'performance',
            isMcbanner = mediaType === 'mcbanner',
            cpmAuto = (modName === 'cpm-max-shows') || (modName === 'cpm-max-coverage') || (modName === 'avg-cpv'),
            data;

        // для ГО на поиске: ручная стратегия "как на сетях"
        // автоматические -  как смарты
        if (isMcbanner) {
            data = modelData.name === u.strategy.SEARCH_MAXIMUM_COVERAGE ? modelData : u._.assign(savedData.options, modelData);
        } else if (isNetStrategy) {
            // для сетевой стратегии, кроме Смартов
            data = modelData.name === u.strategy.MAXIMUM_CLICKS ?
                modelData :
                u._.assign({}, (savedData.options.net ? savedData.options.net : savedData.options), modelData)
        } else {
            // для поисковых и Смартов
            data = modelData.name === u.strategy.MAXIMUM_CLICKS ?
                modelData :
                u._.assign({}, savedData.options, modelData);
        }

        // Если выбрана цель goal_id = 13 (Все ключевые цели)
        // и происходит переключение на другую стратегию
        // и в новой стратегии не допустимо значение goal_id = 13 (есть только цель "Вовлеченные сессии"), то сбрасываем значение goal_id
        if (data.goal_id === u.consts('allKeyGoals') && savedData.options.name !== strategyName) {
            var hasNonDefaultGoals = data.meaningfulGoals && data.meaningfulGoals.some(function (goal) {
                return goal.goal_id !== u.consts('keyGoalsEngagedSessions');
            });

            if (data.name === "autobudget_roi" && !hasNonDefaultGoals) {
                data.goal_id = undefined;
            }

            if ((data.name === "autobudget_avg_cpa" || data.name === "autobudget") && !hasNonDefaultGoals) {
                data.goal_id = undefined;
            }
        }

        // Сохраняем поле для установки охвата стратегии в b-strategy2-settings
        data.strategyCmpAutoCoverage = this.params.strategyModel.campaign.strategy_cpm_auto_coverage;
        data.isCpaPayForConversionsStrategyAllowed = this.params.isCpaPayForConversionsStrategyAllowed;
        data.isCpaPayForConversionEnabled = this.params.isCpaPayForConversionEnabled;

        data.frontpageType = this.params.frontpageType;

        data.impressionStandardTime = this.params.impressionStandardTime;
        data.eshowsBannerRate = this.params.eshowsBannerRate;
        data.eshowsVideoRate = this.params.eshowsVideoRate;
        data.eshowsVideoType = this.params.eshowsVideoType;

        return BEMHTML.apply({
            block: 'b-strategy2-settings',
            mods: {
                name: modName,
                type: cpmAuto ? 'cpm-auto': undefined
            },
            mix: [{
                block: 'b-strategy2-choose',
                elem: 'strategy-settings',
                elemMods: {
                    type: modName,
                    show: 'yes'
                }
            }],
            modelData: data,
            platform: platform,
            needHints: savedData.needHints,
            campaignPeriod: this.params.campaignPeriod,
            campaignType: mediaType,
            walletInfo: this.params.walletInfo,
            cid: this.params.strategyModel.campaign.cid,
            metrikaCounter: this.params.strategyModel.campaign.metrika_counters,
            isNewCamp: this.params.strategyModel.campaign.new_camp,
            strategyName: this.params.strategyModel.name,
            minAutobudgetAvgCpm: this.params.minAutobudgetAvgCpm,
            customBudgetMinValue: this.params.strategyModel.campaign.strategy_with_custom_period_min_available_budget,
            // Параметры для запроса в прогноз стратегии
            rf: this.params.strategyModel.campaign.rf, // oграничение показов
            rfReset: this.params.strategyModel.campaign.rfReset, // ограничение показов
            multipliersPct: this.params.strategyModel.campaign.multipliersPct, // корректировки
            frontpageType: this.params.frontpageType, // версия морды
            isChangeManualStrategyName: this.params.isChangeManualStrategyName, // фича, изменяется ли название ручной стратегии,
            contextLimit: this.params.contextLimit,
            enable_cpc_hold: this.params.enable_cpc_hold,
            attributionModel: this.params.attributionModel, // модель атрибуции
            currentStrategyName: savedData.name,
            isSavedPeriodStrategyEditable: this.params.isSavedPeriodStrategyEditable
        });
    }
});
