/**
 * Родительская модель стратегий:
 * «Максимум показов по минимальной цене» b-strategy2-settings_name_cpm-max-shows
 * «Снижение цены повторных показов» b-strategy2-settings_name_cpm-max-coverage
 */
BEM.MODEL.decl({ model: 'b-strategy2-settings_type_cpm-auto', baseModel: 'b-strategy2-settings' }, {

    // флаг, для идентификации, что это автоматическая cpm-стратегия
    cpmAuto: {
        type: 'boolean',
        default: true
    },

    // флаг, для идентификации, что это avg_cpv стратегия
    isAvgCpv: {
        type: 'boolean',
        default: false
    },

    // радиобокс поля «Тратить»
    // определяет какой из табов(«за период» или «за неделю») активный
    spendMode: {
        type: 'enum',
        enum: ['period', 'week'],
        default: 'period'
    },

    // идентификатор кампании
    cid: 'string',

    // ограничение частоты показов
    rf: 'string',

    // период ограничения частоты показов
    rfReset: 'string',

    // значения корректировок по каждому типу
    multipliersPct: {
        type: 'object',
        default: {}
    },

    // радиобокс поля «Рекомендованая цена» за период
    // определяет какой из табов(«широкий охват» или «узкий интерес») активный
    coveragePeriodMode: {
        type: 'enum',
        enum: ['wide', 'narrow'],
        default: 'wide'
    },

    // радиобокс поля «Рекомендованая цена» за неделю
    // определяет какой из табов(«широкий охват» или «узкий интерес») активный
    coverageWeekMode: {
        type: 'enum',
        enum: ['wide', 'narrow'],
        default: 'wide'
    },

    // чекбокс «Продлевать автоматически»
    autoProlongation: {
        type: 'boolean',
        default: true
    },

    // рекомендованная цена за период
    recommendationPricePeriod: {
        type: 'number',
        default: 0,
        preprocess: function(value) {
            return BEM.blocks['i-utils'].numberFormatter.round(value, {
                precision: 2
            });
        }
    },

    // рекомендованная цена за неделю
    recommendationPriceWeek: {
        type: 'number',
        default: 0,
        preprocess: function(value) {
            return BEM.blocks['i-utils'].numberFormatter.round(value, {
                precision: 2
            });
        }
    },

    // флаг, указывающий на сохраненную стратегию
    isSavedPeriodStrategy: {
        type: 'boolean'
    },

    // флаг, указывающий на режим редактирования сохраненной стратегии по периоду
    // (можно изменить параметры создав стратегию заново, либо отредактировав часть разрешенных полей)
    periodStrategyEditMode: {
        type: 'enum',
        enum: ['change', 'recreate', 'none'],
        default: 'none'
    },

    customBudgetMinValue: 'number',

    // средняя цена клика за тысячу показов для таба «за неделю»
    clickBid: {
        type: 'number',
        preprocess: function(value) {
            var currency = this.model.get('currency'),
                isAvgCpv = this.model.get('isAvgCpv'),
                maxCpmPrice = isAvgCpv ?
                    u.currencies.getConst(currency, 'MAX_AVG_CPV') :
                    u.currencies.getConst(currency, 'MAX_CPM_PRICE'),
                minPrice = isAvgCpv ?
                    u.currencies.getConst(currency, 'MIN_AVG_CPV') :
                    u.currencies.getConst(currency, 'MIN_AUTOBUDGET_AVG_CPM');

            if (this.model.get('minAutobudgetAvgCpm')) {
                minPrice = this.model.get('minAutobudgetAvgCpm');
            }

            return this.model.preprocessSliderValue(
                value,
                minPrice,
                maxCpmPrice
            );
        },
        format: function(value) {
            return BEM.blocks['i-utils'].numberFormatter.roundFormatted(value, {
                precision: 2,
                exponentDelimiter: ' '
            });
        }
    },

    // ставка бюджета для таба «за неделю»
    weekBid: {
        type: 'blank-number',
        precision: 2,
        validation: {
            needToValidate: function() {
                return this.get('spendMode') === 'week';
            },
            rules: {
                lte: {
                    value: function() {
                        return u.numberFormatter.round(u.currencies.getConst(this.get('currency'), 'MIN_DAILY_BUDGET_FOR_PERIOD') * 7);
                    },
                    text: function() {
                        var currencyConst = u.currencies.get(this.get('currency'));

                        return iget2(
                            'b-strategy2-settings',
                            'nedelnyy-byudzhet-ne-mozhet',
                            'Недельный бюджет не может быть меньше {foo}',
                            { foo: u.currencies.formatSum(
                                    this.get('currency'),
                                    7 * currencyConst.MIN_DAILY_BUDGET_FOR_PERIOD
                                )
                            }
                        );
                    }
                },
                gte: {
                    value: function() {
                        return u.currencies.getConst(this.get('currency'), 'MAX_AUTOBUDGET')
                    },
                    text: function() {
                        return iget2(
                            'b-strategy2-settings',
                            'ukazan-slishkom-bolshoy-nedelnyy',
                            'Указан слишком большой недельный бюджет'
                        );
                    }
                }
            }
        }
    },

    // средняя цена клика за тысячу показов для таба «за период»
    clickBidPeriod: {
        type: 'number',
        preprocess: function(value) {
            var currency = this.model.get('currency'),
                isAvgCpv = this.model.get('isAvgCpv'),
                maxCpmPrice = isAvgCpv ?
                    u.currencies.getConst(currency, 'MAX_AVG_CPV') :
                    u.currencies.getConst(currency, 'MAX_CPM_PRICE'),
                minPrice = isAvgCpv ?
                    u.currencies.getConst(currency, 'MIN_AVG_CPV') :
                    u.currencies.getConst(currency, 'MIN_AUTOBUDGET_AVG_CPM');

            if (this.model.get('minAutobudgetAvgCpm')) {
                minPrice = this.model.get('minAutobudgetAvgCpm');
            }

            return this.model.preprocessSliderValue(
                value,
                minPrice,
                maxCpmPrice
            );
        },
        format: function(value) {
            return BEM.blocks['i-utils'].numberFormatter.roundFormatted(value, {
                precision: 2,
                exponentDelimiter: ' '
            });
        }
    },

    // бюджет для таба «за период»
    budget: {
        type: 'blank-number',
        precision: 2,
        exponentDelimiter: ' ',
        validation: {
            needToValidate: function() {
                return this.get('spendMode') === 'period';
            },
            rules: {
                required: {
                    text: iget2('b-day-budget2-settings', 'ne-ukazan-budget-za-period', 'Не указан бюджет за период'),
                    validate: function(val) {
                        return val !== undefined;
                    }
                },
                lte: {
                    needToValidate: function() {
                        return this.get('periodStrategyEditMode') !== 'change';
                    },
                    value: function () {
                        var currencyConst = u.currencies.get(this.get('currency'));

                        return u.numberFormatter.round(this.get('period_range') * currencyConst.MIN_DAILY_BUDGET_FOR_PERIOD);
                    },
                    text: function () {
                        var currencyConst = u.currencies.get(this.get('currency'));

                        return iget2(
                            'b-strategy2-settings',
                            'period-byudzhet-ne-mozhet',
                            'Бюджет за период не может быть меньше {foo}',
                            {
                                foo: u.currencies.formatSum(
                                    this.get('currency'),
                                    this.get('period_range') * currencyConst.MIN_DAILY_BUDGET_FOR_PERIOD
                                ),
                                context: 'foo - числовое значение и валюта'
                            }
                        );
                    }
                }
            }
        }
    },

    // url ссылки на описание стратегии
    helpLinkUrl: {
        calculate: function() {
            return u['b-strategy2-settings'].getHelpLinkUrl('cpm-max-coverage');
        }
    },

    // настройки календаря для пересоздания стратегии «за париод»
    defaultCalendarSettings: {
        type: 'object',
        default: {}
    },

    // период кампании
    campaignPeriod: {
        type: 'object',
        default: {},
        validation: {
            needToValidate: function() {
                return this.get('spendMode') === 'period';
            },
            rules: {
                finishNotBefore: {
                    validate: function(campaignPeriod) {
                        if (!campaignPeriod.finish) {
                            return true;
                        }

                        var current = u.moment().format('YYYY-MM-DD');

                        return u['b-date-range-picker'].getDiff(current, campaignPeriod.finish, 'days') > 0;
                    },
                    text: function() {
                        return iget2(
                            'b-strategy2-settings',
                            'campaign-finish-not-before-tomorrow',
                            'Чтобы задать новый период в стратегии, нужно поменять дату окончания кампании. Дата окончания кампании должна быть не ранее завтрашнего дня'
                        );
                    }
                }
            }
        }
    },

    // период стратегии, содержит поле start и finish в формате 'YYYY-MM-DD'
    period: {
        type: 'object',
        default: {}
    },

    // количество дней между началом в концом периода стратегии
    period_range: {
        type: 'number',
        dependsFrom: ['period'],
        calculate: function(period) {
            var start = period.start,
                finish = period.finish;

            return (start && finish) ?
                u.moment(finish).diff(u.moment(start), 'day') + 1 : // учитываем сегодняшний день + 1
                u['b-strategy2-settings'].DEFAULT_STRATEGY_PERIOD;
        },
        validation: {
            needToValidate: function() {
                return this.get('spendMode') === 'period';
            },
            rules: {
                smallRange: {
                    validate: function(range) {
                        var start = this.get('period').start,
                            current = u.moment().format('YYYY-MM-DD');

                        // Пример: сегодня 1 ноября.
                        // Выбрать даты флайта 1-2 ноября нельзя.
                        // 1-3 ноября можно, 2-3 ноября тоже можно.
                        return !(range === 1 && u.moment(current).diff(start, 'day') === 0);
                    },
                    text: function() {
                        return iget2(
                            'b-strategy2-settings',
                            'selected-period-is-too-smal',
                            'Выбранный период слишком мал для выполнения условий стратегии'
                        );
                    }
                },
                bigRange: {
                    validate: function(range) {
                        return range <= u['b-strategy2-settings'].MAX_STRATEGY_PERIOD;
                    },
                    text: function() {
                        return iget2(
                            'b-strategy2-settings',
                            'selected-period-is-too-big',
                            'Период не может быть больше {maxStrategyPeriod} дня',
                            {
                                some: 'Период не может быть больше {maxStrategyPeriod} дней',
                                many: 'Период не может быть больше {maxStrategyPeriod} дней',
                                maxStrategyPeriod: u['b-strategy2-settings'].MAX_STRATEGY_PERIOD,
                                count: u['b-strategy2-settings'].MAX_STRATEGY_PERIOD
                            }
                        );
                    }
                }
            }
        }
    },

    // настройки стратегии
    options: {
        type: 'object',
        calculate: function() {
            var period = this.get('period');

            switch (this.get('spendMode')) {
                case 'period':
                    var res = {
                        budget: this.get('budget'),
                        auto_prolongation: this.get('autoProlongation') ? 1 : 0,
                        start: period.start,
                        finish: period.finish
                    };

                    if (this.get('isAvgCpv')) {
                        res['avg_cpv'] = this.get('clickBidPeriod');
                    } else {
                        res['avg_cpm'] = this.get('clickBidPeriod');
                    }

                    return res;
                case 'week':
                    var optionWeek = {
                        sum: this.get('weekBid')
                    };

                    if (this.get('isAvgCpv')) {
                        optionWeek['avg_cpv'] = this.get('clickBid');
                    } else {
                        optionWeek['avg_cpm'] = this.get('clickBid');
                    }

                    // сохраняем настройки выбранного охвата для недельного периода для новой кампании
                    if (!this.get('cid')) {
                        optionWeek.coverageWeekMode = this.get('coverageWeekMode');
                    }

                    return optionWeek;
            }
        }
    },

    /**
     * Динамически вычисляемый аналог константы MIN_AUTOBUDGET_AVG_CPM
     */
    minAutobudgetAvgCpm: {
        type: 'number',
        default: 0
    },

    /**
     * Версия морды где будет размещаться реклама
     * Используется в cpm_yndx_frontpage типе кампании
     */
    frontpageType: {
        type: 'array',
        default: []
    },

    /**
     * Стандарт определения видимости для медийной рекламы
     */
    impressionStandardTime: 'string',

    /**
     * Оптимизация показов для медийных баннеров
     */
    eshowsBannerRate: 'string',

    /**
     * Оптимизация показов для медийного видео
     */
    eshowsVideoRate: 'string',

    /**
     * Тип оптимизация показов медийного видео
     */
    eshowsVideoType: 'string'

}, {

    /**
     * Приводит ставку к виду «1 000.00», так же удерживает ее в границах min, max
     * @param {String} value
     * @param {Number} min
     * @param {Number} max
     * @returns {String}
     */
    preprocessSliderValue: function(value, min, max) {
        value = Number(u.numberFormatter.clear(value, {
            precision: 2,
            fail: min
        }));

        value > max && (value = max);
        value < min && (value = min);

        return value;
    }

});
