/**
 * Генерируется при изменении значений внутренних контролов.
 *
 * @event b-time-targeting-holidays#change
 * @param {Object} data
 * @param {'intoAccountWeekend'|'intoAccountHolidays'|'dontShowOnHolidays'|'holidaysRange'| 'holidaysLevel'} data.type Тип контрола.
 * @param {Boolean|String|Array} data.value Новое значение контрола. У holidaysRange это массив вида [from, to].*/

BEM.DOM.decl('b-time-targeting-holidays', {

    onSetMod: {
        js: function() {
            this._selectsCache = {};
            this._checkboxCache = {};
        }
    },

    /**
     * Обрабатывает изменение чекбокса учета рабочих выходных.
     *
     * @param {Event} event Объект события.
     * @private
     */
    _onIntoAccountWeekendChange: function(event) {
        this.trigger('change', { type: 'intoAccountWeekend', value: event.target.isChecked() });
    },

    /**
     * Обрабатывает изменение чекбокса учета праздничных дней.
     *
     * @param {Object} event Объект события.
     * @private
     */
    _onIntoAccountHolidayChange: function(event) {
        var controls = this.elem('holidays-controls'),
            isChecked = event.target.isChecked();

        this
            .setMod(controls, 'hidden', isChecked ? '' : 'yes')
            .trigger('change', { type: 'intoAccountHolidays', value: isChecked });
    },

    /**
     * Обрабатывает изменение показывать/не показывать в праздничные дни.
     *
     * @param {Object} event Объект события.
     * @private
     */
    _onHolidaysRadioboxChange: function(event) {
        /*jshint -W018*/
        var value = !!+event.target.val();

        this._setAvailability(value);

        this.trigger('change', { type: 'dontShowOnHolidays', value: !value })
    },

    /**
     * Обрабатывает изменение времени показа в праздничные дни.
     *
     * @param {Event} event Объект события.
     * @private
     */
    _onHolidaysTimeSelectChange: function(event, data) {
        event.target.hasMod('name', 'from') && this._redrawHolidaysTo(data.index);

        this.trigger('change', {
            type: 'holidaysRange',
            value: [+this._getSelect('from').val(), +this._getSelect('to').val()]
        });
    },

    /**
     * Обрабатывает изменение уровня цены клика в праздничные дни.
     * @private
     */
    _onHolidaysSelectLevelChange: function() {
        this.trigger('change', { type: 'holidaysLevel', value: this._getSelect('level').val() });
    },

    /**
     * Переключает доступность выбора времени показа.
     *
     * @param {Boolean} isEnable Флаг доступности.
     * @returns {BEM}
     * @private
     */
    _setAvailability: function(isEnable) {
        this._getSelects().forEach(function(select) {
            select.setMod('disabled', isEnable ? '' : 'yes');
        }, this);

        return this;
    },

    /**
     * Перерисовывает селект выбора окончания времени показа.
     *
     * @param {Number} from Время с которого будут идти показы.
     * @returns {BEM}
     * @private
     */
    _redrawHolidaysTo: function(from) {
        this._getSelect('to')
            .setOptions(this._getNewOptionsForHolidaysTo(from + 1,  +this._getSelect('to').val()));

        return this;
    },

    /**
     * Генерирует новые опции для селекта выбора окончания времени показа.
     *
     * @param {Number} from Время с которого будут идти показы.
     * @param {Number} current Текущее значение.
     * @returns {Array}
     * @private
     */
    _getNewOptionsForHolidaysTo: function(from, current) {
        var options = [],
            undef;

        if (current < from) current = from;

        for (var i = from; i <= 24; i++) {
            options.push(
                {
                    item: 'option',
                    value: i,
                    selected: current === i ? 'selected' : undef ,
                    content: u['b-time-targeting'].getFormattedHours(i)
                }
            );
        }

        return options;
    },

    /**
     * Получает и кэширует селект определенного типа.
     *
     * @param {String} type Тип селекта(модификатор: type).
     * @returns {BEM}
     * @private
     */
    _getSelect: function(type) {
        return this._selectsCache[type] ||
            (this._selectsCache[type] = this.findBlockOn(this.elem('select', 'type', type), 'select'));
    },

    /**
     * Получает и кэширует все внутренние селекты.
     *
     * @returns {Array}
     * @private
     */
    _getSelects: function() {
        return this._selects || (this._selects = this.findBlocksOn('select', 'select'));
    },

    /**
     * Получает и кэширует чекбокс определенного типа.
     *
     * @param {String} type Тип чекбокса(модификатор: type).
     * @returns {BEM}
     * @private
     */
    _getCheckbox: function(type) {
        return this._checkboxCache[type] ||
            (this._checkboxCache[type] =  this.findBlockOn(this.elem('checkbox-control', 'type', type), 'checkbox'));
    },

    /**
     * Получает и кэширует радиобокс выбора показывать/ не показывать в праздничные дни.
     *
     * @returns {BEM}
     * @private
     */
    _getHolidaysRadiobox: function() {
        return this._holidaysRadiobox ||
            (this._holidaysRadiobox = this.findBlockOn(this.elem('holidays-controls'), 'radiobox'));
    },

    /**
     * Обработчик изменения селектов
     * @param {jQuery.Event} e
     * @param {Object} data
     * @private
     */
    _onSelectChange: function(e, data) {
        var block = e.block,
            domElem = block.domElem,
            selectType = this.elem('select').is(domElem) && this.getMod(domElem, 'type');

        if (selectType) {
            this._selectsCache[selectType] || (this._selectsCache[selectType] = block);

            if (selectType === 'from') {
                this._onHolidaysTimeSelectChange(e, data);
            } else if (selectType === 'to') {
                this._onHolidaysTimeSelectChange(e, data);
            } else if (selectType === 'level') {
                this._onHolidaysSelectLevelChange(e, data);
            }
        }
    },

    /**
     * Обработчик изменения радио группы
     * @param {jQuery.Event} e
     * @param {Object} data
     * @private
     */
    _onRadioboxChange: function(e, data) {
        var block = e.block,
            domElem = block.domElem;

        if (this.elem('radiobox').is(domElem)) {
            this._holidaysRadiobox || (this._holidaysRadiobox = block);

            this._onHolidaysRadioboxChange(e, data);
        }
    },

    /**
     * Обработчик изменения чекбоксов
     * @param {jQuery.Event} e
     * @param {Object} data
     * @private
     */
    _onCheckboChange: function(e, data) {
        var block = e.block,
            domElem = block.domElem,
            checkboxType = this.elem('checkbox-control').is(domElem) && this.getMod(domElem, 'type');

        if (checkboxType) {
            this._checkboxCache[checkboxType] || (this._checkboxCache[checkboxType] = block);

            if (checkboxType === 'intoAccountHolidays') {
                this._onIntoAccountHolidayChange(e, data);
            } else if (checkboxType === 'intoAccountWeekend') {
                this._onIntoAccountWeekendChange(e, data);
            }
        }
    }

}, {

    live: function() {
        this
            .liveInitOnBlockInsideEvent('change', 'select', function(e, data) {
                this._onSelectChange(e, data);
            })
            .liveInitOnBlockInsideEvent('change', 'radiobox', function(e, data) {
                this._onRadioboxChange(e, data);
            })
            .liveInitOnBlockInsideEvent('change', 'checkbox', function(e, data) {
                this._onCheckboChange(e, data);
            });
    }

});
