BEM.DOM.decl('b-time-targeting-scale-board', {
    /**
     * Генерируется при начале настройки часов'.
     *
     * @event b-time-targeting-scale-board#brush-down
     */

    /**
     * Генерируется при окончании настройки часов'.
     *
     * @event b-time-targeting-scale-board#brush-up
     */

    /**
     * Генерируется при выборе ячейки настройки часов. Данное событие запрашивает новый уровень клика для ячейки'.
     *
     * @event b-time-targeting-scale-board#set
     * @param {Object} data данные выбранной ячейки
     * @param {String|Number} data.day день ячейки
     * @param {String} data.hourCode код часа ячейки
     * @param {$.Promise} data.promise промис для получения нового уровня для данной ячейки
     * @param {Boolean} data.isGroupSet было ли выбрано более одной ячейки.
     */

    /**
     * Генерируется при выборе чекбокса часа или дня'.
     *
     * @event b-time-targeting-scale-board#checked-line
     * @param {Object} data данные выбора
     * @param {'day'|'hour'} data.type тип чекбокса
     * @param {String|Number} data.value значение выбора
     */

    /**
     * Генерируется при снятии чекбокса часа или дня'.
     *
     * @event b-time-targeting-scale-board#unchecked-line
     * @param {Object} data данные выбора
     * @param {'day'|'hour'} data.type тип чекбокса
     * @param {String|Number} data.value значение выбора
     */

    onSetMod: {
        js: function() {
            this._isBrushActive = false;
            this._changeDayLineCache = {};
            this._changeHourLineCache = {};

            this._bindEvents();
        }
    },

    /**
     * Переключает чекбокс заданного типа.
     *
     * @param {'day'|'hour'} type Тип чекбокса.
     * @param {String} modVal Значение модификатора чекбокса.
     * @param {Boolean} state Флаг на установку состояния check/uncheck.
     * @returns {BEM}
     */
    setCheckbox: function(type, modVal, state) {
        this._getCheckbox(type, modVal).setMod('checked', state ? 'yes' : '');

        return this;
    },

    /**
     * Устанавливает отображение уровня цены клика часа.
     *
     * @param {jQuery} elem Элемент часа.
     * @param {Number} level Уровень цены клика.
     * @returns {BEM}
     */
    setHourLevel: function(elem, level) {
        this.setMod(elem, 'level', level);
        elem.text(level).attr('title', level + '%');

        return this;
    },

    /**
     * Подписываемся/отписываемся от событий
     * @param {Object} options
     * @param {Boolean} options.isBind
     * @private
     * @returns {*}
     */
    _bindEvents: function(options) {
        this.bindTo('hour', 'mouseenter mouseleave mousedown mouseup', function(event) {
            ({
                mouseenter: this._onHourEnter,
                mouseleave: this._onHourLeave,
                mousedown: this._onHourDown,
                mouseup: this._onHourUp
            })[event.type].call(this, event);
        });

        return this;
    },

    /**
     * Обрабатывает нажатие на элемент часа.
     *
     * @param {Event} event Объект события.
     * @private
     */
    _onHourDown: function(event) {
        this._onSetHourLevel(event);
        this._isBrushActive = true;
        this.trigger('brush-down');
        // лечим залипание
        this.bindTo(this.domElem, 'mouseleave', this._onHourUp);

        event.preventDefault();

    },

    /**
     * Обрабатывает отжатие кнопки мыши.
     *
     * @private
     */
    _onHourUp: function() {
        this._isBrushActive = false;
        this.unbindFrom(this.domElem, 'mouseleave');
        this.trigger('brush-up');
    },

    /**
     * Обрабатывает наведение курсора на элемент часа.
     *
     * @param {Event} event Объект события.
     * @private
     */
    _onHourEnter: function(event) {
        this._isBrushActive ?
            this._onSetHourLevel(event) :
            this.setMod($(event.currentTarget), 'hovered', 'yes');
    },

    /**
     * Обрабатывает уход курсора с элемента часа.
     *
     * @param {Event} event Объект события.
     * @private
     */
    _onHourLeave: function(event) {
        this.delMod($(event.currentTarget), 'hovered');
    },

    /**
     * Обрабатывает установку уровня цены клика на часе.
     *
     * @param {Event} event Объект события.
     * @private
     */
    _onSetHourLevel: function(event) {
        var dfd = $.Deferred(),
            targetElem = this.findElem($(event.currentTarget), 'hour'),
            _this = this,
            day = this.getMod(targetElem, 'day'),
            hourCode = this.getMod(targetElem, 'code');

        this.trigger('set', { day: day, hourCode: hourCode, promise: dfd, isGroupSet: this._isBrushActive });

        dfd.done(function(level) {
            _this.setHourLevel(targetElem, level);
        });
    },

    /**
     * Обрабатывает клик на чекбоксе.
     *
     * @param {Event} event Объект события.
     * @private
     */
    _onCheckboxClick: function(event) {
        var modName = event.target.params.hasOwnProperty('day') ? 'day' : 'code';

        this.trigger(
            event.target.isChecked() ? 'checked-line' : 'unchecked-line',
            { type: modName === 'day' ? 'day' : 'hour', value: event.target.params[modName] });
    },

    /**
     * Проверяет все ли часы отмечены в заданной линии.
     *
     * @param {String} modName Имя модификатора линии.
     * @param {String|Number} modVal Значение модификатора.
     * @returns {Boolean}
     * @private
     */
    _isLineFull: function(modName, modVal) {
        return !this.elem('hour', modName, modVal).toArray().some(function(elem) {
            return this.hasMod($(elem), 'level', '0');
        }, this);
    },

    /**
     * Получает чекбокс заданного типа.
     *
     * @param {'day'|'hour'} type Тип чекбокса.
     * @param {String} modVal Значение модификатора чекбокса.
     * @returns {BEM}
     * @private
     */
    _getCheckbox: function(type, modVal) {
        var params = {
            day: { elem: 'day', mod: 'index' },
            hour: { elem: 'hour-checkbox', mod: 'code' }
        }[type];

        return this.findBlockInside(this.elem(params.elem, params.mod, modVal), 'checkbox');
    }

}, {
    live: function() {
        this.liveInitOnBlockInsideEvent('click', 'checkbox', function(event) {
            this._onCheckboxClick(event);
        });

        return false;
    }
});
