(function() {
/*jshint -W015*/
var Brush = function(level, canvas) {
    this._level = level;
    this._newCanvas = canvas;
};

/**
 * Применяет изменения таблицы таргетинга и выполняет колбек.
 *
 * @param {String|Number} x Координата x указателя(день).
 * @param {String} y Координата y указателя(час).
 * @param {Function} callbackFn Колбек после установки значения в модель.
 * @param {Object} [ctx] Контекст колбека.
 */
Brush.prototype.draw = function(x, y, callbackFn, ctx) {
    this._newCanvas.setCell(x, y, this._level);
    callbackFn.call(ctx || this, this._level);
};

BEM.DOM.decl({ block: 'b-time-targeting', modName: 'preset', modVal: 'other' }, {

    onSetMod: {
        js: function() {
            this.__base.apply(this, arguments);

            var hasNoExtend,
                mode;

            this._hourLineCache = {};
            this._hourElemCache = {};

            this._storeExtendedCells = {};
            this._fixBrush = null;


            // TODO порефакторить включение расширенного режима
            mode = this.model.get('mode');
            hasNoExtend = this.model.get('noExtend');
            this.setMod('mode', hasNoExtend ? 'no-extend' : mode);
            mode === 'extend' && !hasNoExtend && this.afterCurrentEvent(function() {
                this._getModeSwitcher().setMod('checked', 'yes');
            }, this);
        },

        mode: {
            simple: function() {
                this._setSimpleMode();
            },

            extend: function() {
                $.isEmptyObject(this._storeExtendedCells) || this._setExtendedCellsFormCache();

                this._fixBrush !== null && this.model.set('brushLevel', this._fixBrush);
                this._getTimeTargetingScaleBoard().setMod('type', 'expanded');

                this._getTimeTargetingScale()
                    .on('change', function(e, data) { this.model.set('brushLevel', data) }, this)
                    .setMod('visible', 'yes');

                this._storeExtendedCells = {};
            },

            'no-extend': function() {
                this._setSimpleMode();
            }
        }
    },

    /**
     * Генерирует строковый код временного таргетинга.
     * @returns {string}
     * @private
     */
    _getCode: function() {
        var code = '',
            table = this.model.get('timeTargetTable');

        for (var dayIndex = 0; dayIndex < 7; dayIndex++) {
            var day = '' + (dayIndex + 1);

            /*jshint -W083*/
            u['b-time-targeting'].FULL_HOUR_CODES.forEach(function(hourCode) {
                var value = table.getCell(dayIndex, hourCode);

                value && (day += hourCode + (value != 100 ? u['b-time-targeting'].LEVEL_CODES[value] : ''));
            });
            code += day;
        }

        return code;
    },

    /**
     * Устанавливает доступность расширенного режима настроек
     * @param {Boolean} isExtendAvailable
     */
    setExtendModeAvailable: function(isExtendAvailable) {
        if (isExtendAvailable) {
            this.model.set('noExtend', false)
        } else {
            this.model.update({ noExtend: true, brushLevel: 100 });
        }

        this.setMod('mode', isExtendAvailable ? 'mode' : 'no-extend');

        this.model.fix();
    },

    /**
     * Устанавливает простой режим управления таймтаргетингом
     * @private
     */
    _setSimpleMode: function() {
        var brushLevel = this.model.get('brushLevel');

        brushLevel && (this._fixBrush = brushLevel) && this.model.set('brushLevel', 100);
        this._cacheExtendedCells();
        this._getTimeTargetingScaleBoard().setMod('type', 'simple');

        this._getTimeTargetingScale()
            .un('change')
            .setMod('visible', 'no');
    },

    /**
     * Запоминает состояние таблицы таргетинга при смене режима на простой.
     *
     * @private
     */
    _cacheExtendedCells: function() {
        var board = this._getTimeTargetingScaleBoard(),
            table = this.model.get('timeTargetTable');

        table.fullScan('raw', function(cell, value) {
            if (value > 0 && value <= 200) {
                this._storeExtendedCells[cell[0] + ':' + cell[1]] = value;
                board.setHourLevel(this._getHourElem(cell[0], cell[1]), 100);
                table.setCell(cell[0], cell[1], 100, true);
            }
        }, this);
    },

    /**
     * Восстанавливает состояние таблицы таргетинга при возврате режима с простого на расширенный.
     *
     * @private
     */
    _setExtendedCellsFormCache: function() {
        var board = this._getTimeTargetingScaleBoard(),
            table = this.model.get('timeTargetTable'),
            re = /(\w+):(\w+)/,
            _this = this,
            cell;

        $.each(this._storeExtendedCells, function(key, value) {
            cell = key.match(re).slice(1);

            board.setHourLevel(_this._getHourElem(cell[0], cell[1]), value);
            table.setCell(cell[0], cell[1], value, true);
        });
    },

    /**
     * Обрабатывает показ попапа таргетинга.
     *
     * @private
     */
    _initEvents: function() {
        this.__base.apply(this, arguments);

        this.model
            .on('timeTargetTable', 'change', this._onTimeTargetTableChange, this)
            .on('mode', 'change', this._onModeChange, this);
    },

    /**
     * Получает инстанс блока уровня цены клика.
     *
     * @private
     * @returns {BEM}
     */
    _getTimeTargetingScale: function() {
        return this._timeTargetingScale || (this._timeTargetingScale = this.findBlockInside('b-time-targeting-scale'));
    },

    /**
     * Получает инстанс блока таблицы таргетинга.
     *
     * @private
     * @returns {BEM}
     */
    _getTimeTargetingScaleBoard: function() {
        return this._timeTargetingScaleBoard ||
            (this._timeTargetingScaleBoard = this.findBlockInside('b-time-targeting-scale-board'));
    },

    /**
     * Получает инстанс блока тумблера режима настройки.
     *
     * @private
     * @returns {BEM}
     */
    _getModeSwitcher: function() {
        return this._modeSwitcher || (this._modeSwitcher = this.findBlockOn('mode-switcher', 'tumbler'));
    },

    /**
     * Получает инстанс блока с пресетами и счетчиком кол-ва рабочих часов.
     *
     * @private
     * @returns {BEM}
     */
    _getToolbar: function() {
        return this._toolbar || (this._toolbar = this.findBlockInside('b-time-targeting-toolbar'));
    },

    /**
     * Обработчик изменения поля суммы часов
     * @param {Event} e событие
     * @param {Object} data данные
     * @param {Number} data.value значение
     * @private
     */
    _onTotalHoursChange: function(e, data) {
        this.__base.apply(this, arguments);

        this._getToolbar().setTotalHours(data.value, e.target.isValid());
    },

    /**
     *  Обрабатывает изменение режима настройки.
     *
     * @private
     * @param {event} e Событие.
     * @param {Object} data Новый режим.
     */
    _tumblerModChange: function(e, data) {
        var mode = data.checked ? 'extend' : 'simple';

        this
            .setMod('mode', mode)
            .model.set('mode', mode);
    },

    /**
     *  Обрабатывает изменение пресетов.
     *
     * @private
     * @param {event} e Событие.
     * @param {Object} data Выбранный пресет.
     */
    _onToolbarEvent: function(e, data) {
        data === 'all' ? this._setAllHour() : this._setWorkHour();
    },

    /**
     *  Обрабатывает начало изменения ячейки/группы ячеек таблицы таргетинга.
     *
     * @private
     * @param {event} e Событие.
     * @param {Object} data Данные измененной ячейки.
     * @param {Object} data.day День.
     * @param {Object} data.hourCode Код часа.
     */
    _onBoardSet: function(e, data) {
        var dayIndex = +data.day - 1,
            hourCode = data.hourCode,
            timeTargetTable,
            brushLevel;

        if (!data.isGroupSet) {
            timeTargetTable = this.model.get('timeTargetTable');
            brushLevel = this.model.get('brushLevel');

            this._brushDown(dayIndex, hourCode, brushLevel, timeTargetTable);

            this._getTimeTargetingScaleBoard().onFirst('brush-up', function() {
                this._brushUp(brushLevel);
            }, this);
        }

        this._brushApply(dayIndex, hourCode, data.promise);
    },

    /**
     *  Обрабатывает групповое изменение ячейки таблицы таргетинга
     *
     * @private
     * @param {event} event Событие.
     * @param {Object} data Данные линии которую надо поменять.
     * @param {Object} data.type Тип линии: день или час.
     * @param {Object} data.value Индекс линии.
     */
    _onLineChange: function(event, data) {
        this._setLine(
            data.type,
            data.value,
            event.type === 'checked-line' ? this.model.get('brushLevel') : 0);
    },

    /**
     *  Обрабатывает изменение Таблицы временного таргетинга.
     *
     * @private
     * @param {event} e Событие.
     * @param {Object} data Измененные данные.
     * @param {'set-column'|'set-row'} [data.type] Тип изменения - изменяем целиком строку/колонку.
     * @param {Number} data.value Значение, которое установили.
     * @param {Object} data.cell Измененная ячейка.
     * @param {Object} data.cells Измененные ячейки.
     */
    _onTimeTargetTableChange: function(e, data) {
        var board = this._getTimeTargetingScaleBoard(),
            dayIndex = data.type ? data.cells[0][0] : data.cell[0],
            hourCode = data.type ? data.cells[0][1] : data.cell[1],
            checkedFn = {
                'set-row': this._checkForRow.bind(this, board, data.cells, dayIndex),
                'set-column': this._checkForColumn.bind(this, board, data.cells, hourCode)
            },
            uncheckedFn = {
                'set-row': this._uncheckForRow.bind(this, board, data.cells, dayIndex),
                'set-column': this._uncheckForColumn.bind(this, board, data.cells, hourCode)
            };

        if (data.value !== 0) {
            if (data.type) {
                checkedFn[data.type]()
            } else {
                this._toggleForCell(board, dayIndex, hourCode);
            }
        } else {
            if (data.type) {
                uncheckedFn[data.type]();
            } else {
                board.setCheckbox('hour', hourCode, false).setCheckbox('day', dayIndex, false);
            }
        }

        this.model.set('timeTargetCode', this._getCode());

        this._storeExtendedCells = {};
    },

    /**
     *  Обрабатывает изменение режима таблицы временного таргетинга.
     * @param {jQuery.Event} e
     * @param {Object} data
     * @private
     */
    _onModeChange: function(e, data) {
        this.model.set('timeTargetCode', this._getCode());
    },

    /**
     * Обрабатывает начло окрашивания таблицы временного таргетинга.
     *
     * @private
     * @param {Number} x Координата по оси X - день.
     * @param {String} y Координата по оси Y - часовой код.
     * @param {Number} level Устанавливаемый уровень цены клика.
     * @param {MODEL.FIELD.types.data-table} canvas Таблица таргетинга.
     */
    _brushDown: function(x, y, level, canvas) {
        var currentPoint = canvas.getCell(x, y);

        this._brush = new Brush(currentPoint && currentPoint === level ? 0 : level, canvas);
    },

    /**
     * Обрабатывает окончание окрашивания таблицы временного таргетинга.
     *
     * @private
     * @param {Number} newLevel Устанавливаемый уровень цены клика.
     */
    _brushUp: function(newLevel) {
        this.model.set('brushLevel', newLevel);
    },

    /**
     * Окрашивает таблицу временного таргетинга.
     *
     * @private
     * @param {Number} x Координата по оси X - день.
     * @param {String} y Координата по оси Y - часовой код.
     * @param {promise} promise Промис данный таблицей временного таргетинга,
     * ожидает уровень цены клика, что-бы окрасить в нужное значение.
     */
    _brushApply: function(x, y, promise) {
        this._brush && this._brush.draw(x, y, function(level) {
            promise.resolve(level);
            this.afterCurrentEvent(function() {
                this._checkEmpty();
            });
        }.bind(this));
    },

    /**
     * Устанавливает линю таблицы временного таргетинга в нужное значение.
     *
     * @private
     * @param {String} type Тип линии: день или час.
     * @param {String|Number} index индекс линии.
     * @param {Object} level Устанавливаемый уровень цены клика.
     */
    _setLine: function(type, index, level) {
        var board = this._getTimeTargetingScaleBoard(),
            table = this.model.get('timeTargetTable');

        if (type === 'day') {
            this._setNewDayLine(index - 1, level);
            table.setRow(index - 1, level);
        } else {
            this._setNewHourLine(index, level);
            table.setColumn(index, level);
        }

        this._getTableLine(type === 'day' ? 'day' : 'code', index).toArray().forEach(function(elem) {
            board.setHourLevel($(elem), level);
        });

        this._checkEmpty();
    },

    /**
     * Устанавливает все часы таблицы временного таргетинга в нужное значение.
     *
     * @private
     * @param {Object} [level] Устанавливаемый уровень цены клика.
     */
    _setAllHour: function(level) {
        level = level === undefined ? this.model.get('brushLevel') : level;

        for (var i = 0; i < 7; i++) {
            this._setLine('day', i + 1, level || 100);
        }
    },

    /**
     * Устанавливает рабочие часы таблицы временного таргетинга в нужное значение.
     *
     * @private
     */
    _setWorkHour: function() {
        var board = this._getTimeTargetingScaleBoard(),
            table = this.model.get('timeTargetTable'),
            brushLevel = this.model.get('brushLevel'),
            level,
            timeCode;

        // Пробегаем по всем дням и строкам
        for (var day = 0; day < 7; day++)
            for (var i = 0; i != 24; ++i) {
                timeCode = String.fromCharCode(i + 65); // просто получаем буквы A-X
                // если клетка в рабочее время выбираем установленый уровень кисти, а если нет, то ставим ноль
                level = u['b-time-targeting'].WORK_HOUR_CODES.indexOf(timeCode) > -1 && day < 5 ?
                    (brushLevel || 100) :
                    0;
                table.setCell(day, timeCode, level);
                board.setHourLevel($(this._getHourElem(day, timeCode)), level);
            }
    },

    /**
     * Устанавливает значение для всего дня в таблицы модели временного таргетинга.
     *
     * @private
     * @param {Number} dayIndex Индекс дня.
     * @param {Number} level Устанавливаемый уровень цены клика.
     */
    _setNewDayLine: function(dayIndex, level) {
        this.model.get('timeTargetTable').setRow(dayIndex, u['b-time-targeting'].FULL_HOUR_CODES, level);
    },

    /**
     * Устанавливает значение для всех дней определенного часа в таблицы модели временного таргетинга.
     *
     * @private
     * @param {Number} hourCode Код часа.
     * @param {Number} level Устанавливаемый уровень цены клика.
     */
    _setNewHourLine: function(hourCode, level) {
        this.model.get('timeTargetTable').setColumn(hourCode, [0, 1, 2, 3, 4, 5, 6], level);
    },

    /**
     * Отмечает чекбоксы при установке значений для дня.
     *
     * @param {BEM} blockBoard Инстанс блока таблицы таргетинга.
     * @param {Array} cells Измененные ячейки.
     * @param {String|Number} dayIndex Индекс дня.
     * @private
     */
    _checkForRow: function(blockBoard, cells, dayIndex) {
        this._isLineFull('day', dayIndex) &&
            blockBoard.setCheckbox('day', dayIndex, true);

        cells.forEach(function(cell) {
            this._isLineFull('hour', cell[1]) &&
                blockBoard.setCheckbox('hour', cell[1], true);
        }, this);
    },

    /**
     * Снимает чекбоксы при установке значений для дня в ноль.
     *
     * @param {BEM} blockBoard Инстанс блока таблицы таргетинга.
     * @param {Array} cells Измененные ячейки.
     * @param {String|Number} dayIndex Индекс дня.
     * @private
     */
    _uncheckForRow: function(blockBoard, cells, dayIndex) {
        blockBoard.setCheckbox('day', dayIndex, false);
        cells.forEach(function(cell) {
            blockBoard.setCheckbox('hour', cell[1], false);
        });
    },

    /**
     * Отмечает чекбоксы при установке значений для часа.
     *
     * @param {BEM} blockBoard Инстанс блока таблицы таргетинга.
     * @param {Array} cells Измененные ячейки.
     * @param {String} hourCode Код часа.
     * @private
     */
    _checkForColumn: function(blockBoard, cells, hourCode) {
        this._isLineFull('hour', hourCode) && blockBoard.setCheckbox('hour', hourCode, true);

        cells.forEach(function(cell) {
              this._isLineFull('day', cell[0]) && blockBoard.setCheckbox('day', cell[0], true);
        }, this);
    },

    /**
     * Снимает чекбоксы при установке значений для часа в ноль.
     *
     * @param {BEM} blockBoard Инстанс блока таблицы таргетинга.
     * @param {Array} cells Измененные ячейки.
     * @param {String} hourCode Код часа.
     * @private
     */
    _uncheckForColumn: function(blockBoard, cells, hourCode) {
        blockBoard.setCheckbox('hour', hourCode, false);
        cells.forEach(function(cell) {
            blockBoard.setCheckbox('day', cell[0], false);
        });
    },

    /**
     * Переключает чекбоксы для определенного часа.
     *
     * @param {BEM} blockBoard Инстанс блока таблицы таргетинга.
     * @param {String|Number} dayIndex Индекс дня.
     * @param {String} hourCode Код часа.
     * @private
     */
    _toggleForCell: function(blockBoard, dayIndex, hourCode) {
        blockBoard
            .setCheckbox('hour', hourCode, this._isLineFull('hour', hourCode))
            .setCheckbox('day', dayIndex, this._isLineFull('day', dayIndex));
    },

    /**
     * Получает элементы часа дня или определенного часа таблицы таргетинга.
     *
     * @param {'day'|'code'} modName Имя модификтора.
     * @param {String|Number} modVal Значение модификатора.
     * @returns {jQuery}
     * @private
     */
    _getTableLine: function(modName, modVal) {
        var key = modName + ':' + modVal;

        return this._hourLineCache[key] ||
            (this._hourLineCache[key] = this._getTimeTargetingScaleBoard().elem('hour', modName, modVal));
    },

    /**
     * Получает элемент определенного часа.
     *
     * @param {String|Number} dayIndex Индекс дня.
     * @param {String} hourCode Код часа.
     * @returns {jQuery}
     * @private
     */
    _getHourElem: function(dayIndex, hourCode) {
        var key = dayIndex + ':' + hourCode;

        return this._hourElemCache[key] ||
            (this._hourElemCache[key] =
                this._getTimeTargetingScaleBoard()
                    .findElem(this._getTableLine('day', +dayIndex + 1), 'hour', 'code', hourCode));
    },

    /**
     * Проверяет все ли часы отмечены в заданной линии.
     *
     * @param {'day'|'hour'} type Тип линии.
     * @param {String|Number} key Индекс дня или код часа.
     * @returns {Boolean}
     * @private
     */

    _isLineFull: function(type, key) {
        var table = this.model.get('timeTargetTable'),
            total = type === 'day' ? 24 : 7,
            count = 0;

        table[type == 'day' ? 'rowScan' : 'columnScan'](key, function(cell, value) {
            if (!value) return true;

            count++;
        });

        return count === total;
    },

    /**
     * Возвращает в исходное состояние таблицу временного таргетинга.
     *
     * @returns {BEM}
     * @private
     */
    _resetScaleBoard: function() {
        var board = this._getTimeTargetingScaleBoard(),
            setHour = board.setHourLevel.bind(board),
            table = this.model.get('timeTargetTable'),
            hourCodes = u['b-time-targeting'].FULL_HOUR_CODES,
            noExtend = this.hasMod('mode', 'no-extend');

        for (var row = 0; row < 7 ; row++) {
            /*jshint -W083*/
            hourCodes.forEach(function(column) {
                var value = table.getCell(row, column) || 0;

                noExtend && value && (value = 100);
                setHour(this._getHourElem(row, column), value);
            }, this);

            board.setCheckbox('day', row, this._isLineFull('day', row));
        }

        hourCodes.forEach(function(column) {
            board.setCheckbox('hour', column, this._isLineFull('hour', column));
        }, this);

        return this;
    },

    /**
     * Проверяет не пустая ли таблица и если пустая, то выставляет всё
     * @private
     */
    _checkEmpty: function() {
        // Если пользователь попытался снять все ячейки, то выставляем все обратно
        if (this.model.get('timeTargetTable').isEmpty())
            this._setAllHour();
    },

    /**
     * Сбрасывает изменения в контроле
     * @private
     */
    _reset: function() {
        var model = this.model;

        this.__base.apply(this, arguments);

        this._resetScaleBoard();

        this._getTimeTargetingScale().setLevel(model.get('brushLevel'), true);

        this._getModeSwitcher().setMod('checked', model.get('mode') == 'simple' ? '' : 'yes');
    },

    destruct: function() {
        this.model
            .un('timeTargetTable', 'change', this._onTimeTargetTableChange, this)
            .un('mode', 'change', this._onModeChange, this);

        this.__base.apply(this, arguments)
    }

}, {

    live: function() {
        this
            .liveInitOnBlockInsideEvent('change', 'tumbler', function(e, data) {
                if (this.elem('mode-switcher').is(e.block.domElem)) {
                    this._tumblerModChange(e, data);
                }
            })
            .liveInitOnBlockInsideEvent('selected', 'b-time-targeting-toolbar', function(e, data) {
                this._onToolbarEvent(e, data);
            })
            .liveInitOnBlockInsideEvent('set', 'b-time-targeting-scale-board', function(e, data) {
                this._onBoardSet(e, data);
            })
            .liveInitOnBlockInsideEvent('checked-line unchecked-line', 'b-time-targeting-scale-board',
                function(e, data) {
                    this._onLineChange(e, data);
                });

        return this.__base.apply(this, arguments);
    }

});

})();
