/**
 * @event b-time-targeting#change
 * @type {Object}
 * @property {Boolean} isChanged Есть ли изменения в блоке
 *
 * @fires b-time-targeting#change Событие изменения значения блока
 */
BEM.DOM.decl({ block: 'b-time-targeting' }, {

    onSetMod: {

        js: function() {
            var modelParams = this.params.modelParams;

            this.model = BEM.MODEL.create({ name: modelParams.name, id: modelParams.id }, modelParams.data);

            this._initEvents();
        }

    },

    /**
     * Возвращает выбранное значени
     * @returns {Object}
     */
    getValue: function() {
        var data;

        if (!this.model.isValid()) {
            return { error: true };
        }

        data = this.model.toJSON();

        return {
            timeZone: {
                id: data.timezoneId,
                text: data.timezoneText
            },
            timeTargetCode: data.timeTargetCode,
            isHolidaySettingsEnabled: data.intoAccountHolidays,
            isWorkingWeekendEnabled: data.intoAccountWeekend,
            holidayShowSettings: {
                isShowing: !data.dontShowOnHolidays,
                showingFrom: data.holidaysFrom,
                showingTo: data.holidaysTo,
                coefficient: data.holidaysTimeTargetLevel
            },
            isExtendModeOn: !data.noExtend && data.mode == 'extend',
            preset: data.timeTargetPreset
        };
    },

    /**
     * Валидно ли значение
     * @returns {Boolean}
     */
    isValid: function() {
        return this.model.isValid();
    },

    /**
     * Фиксирует изменение текущих настроек
     */
    fixChanges: function() {
        this.model.fix();
    },

    /**
     * Возвращает флаг наличия изменений
     * @returns {Boolean}
     */
    isChanged: function() {
        return this.model.isChanged();
    },

    /**
     * Отменяет текущие настройки
     */
    cancel: function() {
        setTimeout(this._reset.bind(this), 150);
    },

    /**
     * Обрабатывает показ попапа таргетинга.
     *
     * @private
     */
    _initEvents: function() {
        this.model
            .on('change', this._onModelChange, this)
            .on('error', this._onError, this)
            .on('totalHours', 'change', this._onTotalHoursChange, this);
    },

    /**
     * Обработчик изменения модели
     * @param {jQuery.Event} e
     * @private
     */
    _onModelChange: function(e) {
        this.trigger('change', { isChanged: this.isChanged() });
    },

    /**
     * Очищает сообщения об ошибках в попапе
     * @private
     */
    _clearErrors: function() {
        var errors = this.elem('errors');

        errors.html('');
        this.setMod(errors, 'empty', 'yes');
    },

    /**
     * Обработчик ошибок модели
     * @param {Event} e событие
     * @param {Object} data данные
     * @param {Object} data.erorrs ошибки
     * @private
     */
    _onError: function(e, data) {
        var errors = this.elem('errors');

        errors.html(data.errors.map(function(e) {
            return e.text;
        }).join('<br/>'));
        this.setMod(errors, 'empty', 'no');
    },

    /**
     * Обработчик изменения поля суммы часов
     * @param {Event} e событие
     * @param {Object} data данные
     * @param {Number} data.value значение
     * @private
     */
    _onTotalHoursChange: function(e, data) {
        this._clearErrors();
    },

    /**
     * Получает инстанс блока выбора таймзоны.
     *
     * @private
     * @returns {BEM}
     */
    _getTimezone: function() {
        return this._timezone || (this._timezone = this.findBlockInside('b-time-targeting-timezone'));
    },

    /**
     * Получает инстанс блока настройки показов на выходных и праздниках.
     *
     * @private
     * @returns {BEM}
     */
    _getHolidays: function() {
        return this._holidays || (this._holidays = this.findBlockInside('b-time-targeting-holidays'));
    },

    /**
     *  Обрабатывает изменение таймзоны.
     *
     * @private
     * @param {event} e Событие.
     * @param {Object} data Данные таймзоны
     * @param {Object} data.id id новой таймзоны.
     * @param {Object} data.text Название новой таймзоны.
     * @param {Object} data.region Название региона таймзоны
     */
    _onTimezonesChange: function(e, data) {
        this.model
            .set('timezoneRegionId', data.region)
            .set('timezoneId', data.id)
            .set('timezoneText', data.text);

        this._clearErrors();
    },

    /**
     * Обрабатывает изменение контролов показа на выходных днях.
     *
     * @private
     * @param {Event} event Событие.
     * @param {Object} data Измененные данные.
     * @param {Object} data.type Тип контрола.
     * @param {Object} data.value Новое значение.
     */
    _onHolidaysControlChange: function(event, data) {
        var model = this.model,
            handlers = {
                intoAccountHolidays: model.set.bind(model, 'intoAccountHolidays', data.value),
                intoAccountWeekend: model.set.bind(model, 'intoAccountWeekend', data.value),
                dontShowOnHolidays: model.set.bind(model, 'dontShowOnHolidays', data.value),
                holidaysLevel: model.set.bind(model, 'holidaysTimeTargetLevel', data.value),
                holidaysRange: function() {
                    model
                        .set('holidaysFrom', data.value[0])
                        .set('holidaysTo', data.value[1]);
                }
            };

        handlers[data.type]();
    },

    /**
     * Перерисовывает блок настройки показов на выходных и праздниках.
     *
     * @private
     */
    _redrawHolidays: function() {
        var modelData = u._.extend(this.model.toJSON(), { showWorkingWeekendCheckbox : this.params._data.showWorkingWeekendCheckbox });

        this.__self.replace(this._getHolidays().domElem, BEMHTML.apply({
            block: 'b-time-targeting-holidays',
            data: modelData
        }));

        // Сбрасываем закешированный замененный BEM-блок
        this._holidays = null;
    },

    /**
     * Получает общее кол-во отмеченных часов в рабочее время.
     *
     * @returns {Number}
     * @private
     */
    _getTotalWorkHour: function() {
        var count = 0;

        this.model.get('timeTargetTable').fullScan(function(cell, value) {
            if (cell[0] > 4) return true;

            value && count++;
        });

        return count;
    },

    /**
     * Сбрасывает изменения в контроле
     * @private
     */
    _reset: function() {
        var model = this.model;

        model.rollback();

        this._redrawHolidays();

        this._getTimezone().setTimezone(model.get('timezoneRegionId'), model.get('timezoneId'));
    },

    destruct: function() {
        this.model.destruct();

        this.__base.apply(this, arguments);
    }


}, {

    live: function() {
        this
            .liveInitOnBlockInsideEvent('change', 'b-time-targeting-timezone', function(e, data) {
                this._onTimezonesChange(e, data);
            })
            .liveInitOnBlockInsideEvent('change', 'b-time-targeting-holidays', function(e, data) {
                this._onHolidaysControlChange(e, data);
            });

        return false;
    }

});
