BEM.DOM.decl('banner-image-crop', {

    /**
     * Объект с координатами и размерами выбранной области
     *
     * @typedef {Object} jQueryCropSelection
     * @property {Number} x координата x1
     * @property {Number} y координата y1
     * @property {Number} x2 координата x2
     * @property {Number} y2 координата y2
     * @property {Number} w ширина
     * @property {Number} h высота
     */

    /**
     * @type {String}
     * Значение соотношения сторон по умолчанию
     */
    _defaultRatio: '0',

    /**
     * @type {Array}
     * максимальный размер выбранной при обрезке области (по умолчанию не ограничен)
     */
    _cropMaxSize: [5000, 5000],

    /**
     * @type {Array}
     * минимальный размер выбранной при обрезке области
     */
    _cropMinSize: [112, 84],

    onSetMod: {

        js: function() {
            var maxSize = u.preview.getCropMaxSize(),
                availableRatios = this.params.availableRatios;

            if (this.params.isMobileContent) {
                this._cropMaxSize = [maxSize.width, maxSize.height];
            }

            if (!u._.includes(availableRatios, this._defaultRatio)) {
                this._defaultRatio = availableRatios[0];
            }

            this._ratio = this._defaultRatio;
        },

        disabled: function(modName, modVal) {
            if (this._imageCrop) {
                this._imageCrop[modVal ? 'disable' : 'enable']();
            }
        }

    },

    /**
     * Инициализирует редактор
     * @param {Object} options Опции для image-crop
     * @returns {Deferred}
     */
    init: function(options) {
        var imageElem = this.findElem('image'),
            side = Math.min(options.editorWidth, options.editorHeight),
            x = parseInt((options.editorWidth - side) / 2),
            y = parseInt((options.editorHeight - side) / 2),
            cropOptions = {
                boxWidth: options.editorWidth,
                boxHeight: options.editorHeight,
                setSelect: [x, y, x + side, y + side],
                maxSize: this._cropMaxSize,
                minSize: this._cropMinSize,
                allowEmptySelection: false
            };

        this.reset();
        this._previewUrl = options.previewUrl;
        imageElem
            .attr('src', options.previewUrl)
            .width('')
            .height('');

        if (options.origImageWidth && options.origImageHeight) {
            cropOptions.trueSize = [options.origImageWidth, options.origImageHeight];
        }

        return BEM.blocks['image-crop']
            .create(imageElem, cropOptions)
            .then(function(imageCrop) {
                // Если позвали init с другой картинкой, то этот редактор не активируем
                if (options.previewUrl !== this._previewUrl) {
                    return;
                }

                this._imageCrop = imageCrop;
                this._ratio = this._getInitialRatio();
                imageCrop.setPreservedRatio(this._ratio);
                imageCrop.on('select', function(e, selection) {
                    var validationResult = this._validateSelectionSize(selection);

                    // отправляем простой и нормализованный selection
                    // простой может понадобиться например в превью для более плавной отрисовки
                    if (validationResult.error) {
                        this.trigger('error', {
                            selection: selection,
                            normalizedSelection: this._normalizeSelection(selection),
                            type: validationResult.type,
                            text: this._getErrorText(validationResult.type)
                        });
                    } else {
                        this.trigger('select', {
                            selection: selection,
                            normalizedSelection: this._normalizeSelection(selection)
                        });
                    }
                }, this);
                this._adjustSelectionToAspectRatio();

                if (this._getRatioSelector()) {
                    this._getRatioSelector().val(this._ratio);
                }

                this.setMod('active', 'yes');

                if (this.hasMod('disabled')) {
                    imageCrop.disable();
                }
            }.bind(this));
    },

    /**
     * Вырезает изображение и вставляет его в канву в соответствии с пропорциями вырезки и возвращает dataURL картинки
     * @param {Object} size объект с размерами, в которые нужно вписать вырезанное изображение
     * @param {Number} size.width ширина
     * @param {Number} size.height высота
     * @protected
     */
    getCroppedPreviewDataURL: function(size) {
        var canvas = document.createElement('canvas'),
            context = canvas.getContext ? canvas.getContext('2d') : null,
            image = this.findElem('image').get(0),
            scaled;

        if (context) {
            scaled = this._imageCrop.getScaledSelection();

            u._.extend(canvas, size);

            // адаптация под соотношение сторон
            if (scaled.w > scaled.h) {
                canvas.width = canvas.height * this._imageCrop.getSelectionRatioValue();
            } else {
                canvas.height = canvas.width / this._imageCrop.getSelectionRatioValue();
            }

            context.drawImage(image, scaled.x, scaled.y, scaled.w, scaled.h, 0, 0, canvas.width, canvas.height);

            return canvas.toDataURL();
        }

        return null;
    },

    /**
     * Текущее желаемое соотношение сторон
     * @returns {*|String}
     */
    getPreservedRatio: function() {
        return this._ratio;
    },

    /**
     * Удаление редактора на картинке
     */
    reset: function() {
        if (this._imageCrop) {
            // этот хак решает проблему, при которой картинка таинственно появляется в IE независимо от стилей
            this._imageCrop.destruct();
            this._imageCrop = null;
        }

        this._previewUrl = null;
        this.delMod('active');
        // jCrop жестко ставит display:block - модификаторами не справиться.
        this.findElem('image').hide();
    },

    /**
     * Обработчик нажатия на чекбоксы
     * @param {String} ratio Соотношение сторон через двоеточие
     * @private
     */
    _onChangeRatio: function(ratio) {
        this._ratio = ratio;
        this._imageCrop.setPreservedRatio(ratio);
        this._adjustSelectionToAspectRatio();
    },

    /**
     * Возвращает блок выбора соотношения сторон изображения при его обрезке
     * @returns {BEM.DOM|null}
     * @private
     */
    _getRatioSelector: function() {
        return this._aspectRatioSelector || (this._aspectRatioSelector = this.findBlockOn('aspect-ratio', 'radio-button'));
    },

    /**
     * Проверка на то, что изображение соответствует требованиям для формата 16:9
     * @param {Number[]} bounds массив с шириной и высотой изображения
     * @returns {Boolean}
     * @private
     */
    _isCroppingWideImage: function(bounds) {
        var boundX = bounds[0],
            boundY = bounds[1];

        // изображение соответствует требованиям для формата 16:9
        return Math.abs(16 / 9 - boundX / boundY) < 0.01 && boundX >= 1080 && boundY >= 607;
    },

    /**
     * Получение начального соотношения сторон для выделяемой области
     * @returns {String} Соотношение сторон через двоеточие
     * @private
     */
    _getInitialRatio: function() {
        // проверяем соответствует изображение требованиям для формата 16:9 и есть ли он среди доступных
        // также этот формат по умолчанию для РМП
        if (
            this.params.isMobileContent ||
            (this._isCroppingWideImage(this._imageCrop.getBounds()) && u._.includes(this.params.availableRatios, '16:9'))
        ) {
            return '16:9';
        } else {
            return this._defaultRatio;
        }
    },

    /**
     * Подгоняет соотношение сторон выбранной области после смены желаемого соотношения сторон
     * @private
     */
    _adjustSelectionToAspectRatio: function() {
        var ratio43 = 4 / 3,
            ratio34 = 3 / 4,
            imageCrop = this._imageCrop,
            bounds = imageCrop.getBounds(),
            boundX = bounds[0],
            boundY = bounds[1],

            // соотношение сторон изображения
            imgAspectRatio = boundX / boundY,

            // числовое значение заданного желаемого соотношения сторон
            preservedRatioValue = imageCrop.getPreservedRatioValue(),

            // соотношение сторон для выделенной области
            areaAspectRatio = preservedRatioValue ||
                (boundX > boundY && imgAspectRatio > ratio43  ?
                    ratio43 :
                    (boundX < boundY && imgAspectRatio < ratio34 ? ratio34 : 0)),

            x = 0,
            y = 0;

        if (areaAspectRatio) {
            if (areaAspectRatio > imgAspectRatio) {
                // нужно двигать вниз
                y = Math.round((boundY - boundX / areaAspectRatio) / 2);
            } else {
                // нужно двигать вправо
                x = Math.round((boundX - boundY * areaAspectRatio) / 2);
            }
        }

        this.setMod('aspect-ratio', preservedRatioValue ? 'fixed' : 'free');

        imageCrop.setSelect([x, y, boundX - x, boundY - y]);
        imageCrop.focus();
    },

    /**
     * Сохраняет текущие координаты выделенной области для обрезки изображения
     * @param {jQueryCropSelection} selection
     * @returns {Object}
     * @private
     */
    _normalizeSelection: function(selection) {
        var isWide = this._ratio == '16:9',
            x = Math.max(parseInt(selection.x), 0),
            y = Math.max(parseInt(selection.y), 0),
            x2,
            y2,
            width = parseInt(selection.w),
            height = parseInt(selection.h),
            selectionRatio = width / height,
            ratio = isWide ? 16 / 9 : Math.min(Math.max(selectionRatio, 3 / 4), 4 / 3),
            bounds = this._imageCrop.getBounds();

        if (isWide) {
            if (selectionRatio < ratio) {
                height = Math.round(width / ratio);
            } else {
                width = Math.round(height * ratio);
            }

            if (width < 1080 || height < 607) {
                width = 1080;
                height = 607;
            }
        } else {
            if (ratio < 1) {
                height = parseInt(width / ratio);
            } else {
                width = parseInt(height * ratio);
            }
        }

        x2 = Math.min(x + width, bounds[0]);
        y2 = Math.min(y + height, bounds[1]);

        return {
            x: x,
            y: y,
            x2: x2,
            y2: y2,
            w: x2 - x,
            h: y2 - y
        };
    },

    /**
     * Валидация размеров выделенного поля изображения.
     * @returns {Object}
     * @private
     */
    _validateSelectionSize: function(selection) {
        var isWide = this._ratio === '16:9',
            w = selection.x2 - selection.x,
            h = selection.y2 - selection.y,
            selectedAspectRatio = w / h;

        // если соотношение 16:9, то проверяем минимальный размер 1080x607
        if (isWide && (w < 1080 || h < 607)) {
            return { error: true, type: 'size-16-9' };
        }

        // иначе минимальный размер 450x450 и соотношение от 3:4 до 4:3
        if (!isWide) {
            if (w < 450 || h < 450) {
                return { error: true, type: 'size' };
            }

            // todo@dima117a сравнение вещественных чисел с точностью 0.01
            if ((selectedAspectRatio - 4 / 3 > 0.01) || (3 / 4 - selectedAspectRatio > 0.01 )) {
                return {
                    error: true,
                    type: selection.w > selection.h ? 'wide-scale' : 'narrow-scale'
                };
            }
        }

        return { error: false };
    },

    /**
     * Возвращает текст ошибки валидации
     * @param {String} errorType Ключ ошибки валидации
     * @returns {String|Undefined}
     * @private
     */
    _getErrorText: function(errorType) {
        return ({
            'size': iget2('banner-image-crop', 'razmer-izobrazheniya-dolzhen-byt', 'Размер изображения должен быть от 450px по короткой стороне'),
            'size-16-9': iget2('banner-image-crop', 'razmer-izobrazheniya-dolzhen-byt-100', 'Размер изображения должен быть не менее 1080х607px'),
            'wide-scale': iget2('banner-image-crop', 'vybrannaya-oblast-slishkom-shirokaya', 'Выбранная область слишком широкая'),
            'narrow-scale': iget2('banner-image-crop', 'vybrannaya-oblast-slishkom-uzkaya', 'Выбранная область слишком узкая')
        })[errorType];
    }

}, {

    live: function() {
        this
            .liveInitOnBlockInsideEvent('change', 'radio-button', function(e) {
                this._onChangeRatio(e.block.val())
            });

        return true;
    }

});
