(function() {
    /**
     * "Уведомлящий" эмиттер, сообщает о факте подписки вызовом переданного onSubscribe
     * @constructor
     * @param {$.observable} emitter исходный эмиттер
     * @param {Function} onSubscribe колбек на подписку
     */
    function NotifyingEmitter(emitter, onSubscribe) {
        this._emitter = emitter;
        this._onSubscribe = onSubscribe;
    }

    NotifyingEmitter.prototype = {
        /**
         * Совершает подписку на событие
         * @see $.observable.on
         * @returns {this}
         */
        on: function() {
            this._emitter.on.apply(this._emitter, arguments);
            this._onSubscribe(this._emitter, arguments);

            return this;
        },
        /**
         * Совершает одноразовую подписку на событие
         * @see $.observable.onFirst
         * @returns {this}
         */
        onFirst: function() {
            this._emitter.onFirst.apply(this._emitter, arguments);
            this._onSubscribe(this._emitter, arguments);

            return this;
        }
    };

    BEM.decl('i-subscription-manager', {
        onSetMod: {
            js: function() {
                this._subscriptions = [];
            }
        },

        /**
         * Совершает подписку на событие и запоминает параемтры подписки,
         * чтобы в дальнейшем можно было совершить отписку
         * @see $.observable.on
         * @return {this}
         */
        on: function(/*emitter, ...*/) {
            var args = Array.prototype.slice.call(arguments),
                emitter = args.shift(),
                notifyingEmitter = this.wrap(emitter);

            notifyingEmitter.on.apply(notifyingEmitter, args);

            return this;
        },

        /**
         * Совершает подписку на событие и запоминает параемтры подписки,
         * чтобы в дальнейшем можно было совершить отписку
         * @see $.observable.on
         * @return {this}
         */
        onFirst: function(/*emitter, ...*/) {
            var args = Array.prototype.slice.call(arguments),
                emitter = args.shift(),
                notifyingEmitter = this.wrap(emitter);

            notifyingEmitter.onFirst.apply(notifyingEmitter, args);

            return this;
        },

        /**
         * Возвращает эмиттер, "обернутый" в "уведомляющий" эмиттер
         * "Уведомляющий" эмиттер имеет тот же интерфейс что и $.observable, но при этом позволяет запоминать подписки
         * Т.е. wrap нужно использовать если хочется сделать несколько подписок на одном эмиттере, которые
         * затем планируется разом удалить
         * @param {$.observable} emitter
         * @returns {NotifyingEmitter}
         */
        wrap: function(emitter) {
            return new NotifyingEmitter(emitter, this._onSubscribe.bind(this));
        },

        /**
         * Колбек для NotifyingEmitter запоминающий подписки
         * @param {$.observable} emitter исходный эмиттер
         * @param {*[]} args аргументы подписки
         */
        _onSubscribe: function(emitter, args) {
            this._subscriptions.push({
                emitter: emitter,
                args: args
            });
        },

        /**
         * Отменяет все запомненные подписки, в обратном порядке
         */
        dispose: function() {
            this._subscriptions.reverse().forEach(function(s) {
                s.emitter.un.apply(s.emitter, s.args);
            });
            this._subscriptions = [];
        }
    });
}());
