(function() {
    //соотношение между названиями псведо-валют и нормальных валют
    var PSEUDO_CURRENCIES = {
            ynd_fixed: 'YND_FIXED',
            rub: 'RUB',
            grivna: 'UAH',
            tenge: 'KZT',
            tr_lira: 'TRY'
        },
        EPSILON = 1e-7, //значения, большие или равные этой константе принимаются за 0.01
        NBSP = String.fromCharCode(160); //неразрывный пробел

    /**
     * В ЛИ код валюты имеет вид отличный от профи - так что его надо преобразовать к стандартному
     * @param {String} code код валюты
     * @returns {String}
     */
    function formatCurrencyCode(code) {
        if (PSEUDO_CURRENCIES[code]) return PSEUDO_CURRENCIES[code];

        return code;
    }

    /**
     * Определяем, нужно ли использовать псевдо-валюту вместо нормальной
     * @param {String} currency код валюты
     * @param {Object} pseudoCurrency псевдовалюта
     * @returns {Boolean}
     */
    function usePseudoCurrency(currency, pseudoCurrency) {
        return !!(currency == 'YND_FIXED' && pseudoCurrency && pseudoCurrency.name);
    }

    /**
     * currencies нельзя использовать в preprocess т.к. на момент его выполнения u.consts еще не работает
     */
    u.register({
        currencies: {
            /**
             * инициализация блока - кеширование констант
             * @param {Object} consts - константы валют
             */
            init: function(consts) {
                this.currencies = consts.currencies;
                this.pseudoCurrency = consts.pseudoCurrency;
                this.currencyTextsDescription = consts.currencyTextsDescription;
                this.fixedCurrencyRates = consts.fixedCurrencyRates;
                this.locationNameI18N = consts.locationNameI18N;
            },
            /**
             * Функция возвращает валюту по её коду
             * @param {String} currency код валюты
             * @returns {Object} объект валюты
             * Не используется в псевдо валюте
             */
            get: function(currency) {
                currency = currency || 'YND_FIXED';
                currency = formatCurrencyCode(currency);

                return this.currencies[currency];
            },

            /**
             * Функция вывода строки с именем валюты, по дефолту выводится сокращенное название - например 'руб.'
             * @param {String} currency код валюты
             * @param {Object} [opt]
             * @param {Object} [opt.isPseudo] использовать псевдо-валюту
             * @param {Number} [opt.full] показвать полное название валюты
             * @returns {String} имя валюты
             */
            getName: function(currency, opt) {
                currency = formatCurrencyCode(currency);

                var field = opt && opt.full ? 'full_name' : 'name',
                    name;

                if (opt && opt.isPseudo && usePseudoCurrency(currency, this.pseudoCurrency)) {
                    name = this.pseudoCurrency.name;
                } else if (u.currencies.get(currency)) {
                    name = u.currencies.get(currency)[field];
                } else {
                    name = currency;
                }

                //имя константы попадает в танкер из i-const.js, а здесь iget служит для перевода по ключу
                return this.locationNameI18N('common-i18n', name);
            },

            /**
             * Возвращает текст с указанным идентификатором для переданной валюты.
             * @param {String} currency код валюты
             * @param {String} textId идентификатор текста
             * @returns {String}
             */
            getText: function(currency, textId) {
                currency = formatCurrencyCode(currency);

                var descriptions = this.currencyTextsDescription;

                return descriptions[textId] ?
                    //имя константы попадает в танкер из i-const.js.<lang>, а здесь iget служит для перевода по ключу
                    this.locationNameI18N('common-i18n', descriptions[textId][currency]) :
                    '';
            },

            /**
             * Округляет ставку в большую/меньшую сторону до шага торгов в указанной валюте
             * Не используется в псевдо-валюте
             * @param {Number} price ставка для округления
             * @param {String} currency код валюты
             * @param {'up'|'down'} roundDirection направление округления
             * Не используется в псевдо валюте
             * @returns {Number}
             */
            roundPriceToStep: function(price, currency, roundDirection) {
                currency = formatCurrencyCode(currency);

                var step = u.currencies.getConst(currency, 'AUCTION_STEP'),
                    priceRoundedDown = parseInt(u.round2s((price * 1e6) / (step * 1e6))) * step;

                return priceRoundedDown + (roundDirection == 'up' && price > priceRoundedDown ? step : 0);
            },


            /**
             * Возвращает значение валютной константы в данной валюте
             * @param {String} currency код валюты
             * @param {String} constName код константы
             * @param {Object} [opt]
             * @returns {String|Number|Object}
             */
            getConst: function(currency, constName, opt) {
                currency = formatCurrencyCode(currency);

                return u.currencies.get(currency)[constName] * (opt && opt.isPseudo && usePseudoCurrency(currency, this.pseudoCurrency) ? u.currencies.getConvUnitRatePseudo() : 1);
            },

            /**
             * Функция для вывода определённой суммы денег, определяемых валютной константой, в определённой валюте
             * @param {String} currency код валюты
             * @param {String} constName код константы
             * @param {Object} [opts]
             * @param {Boolean} [opts.isPseudo] использовать псевдо-валюту
             * @returns {String}
             */
            formatConst: function(currency, constName, opts) {
                currency = formatCurrencyCode(currency);

                return u.currencies.formatSum(currency, u.currencies.getConst(currency, constName, opts), opts);
            },

            /**
            * Возвращает ставку конвертации для псевдо-валюты
            * @returns {Number}
            */
            getConvUnitRatePseudo: function() {
                return this.pseudoCurrency.rate;
            },

            /**
             * Возвращает ставку конвертации по отношению к валюте currency
             * @param {String} currency код валюты
             * @returns {Number}
             */
            getConvUnitRate: function(currency) {
                currency = formatCurrencyCode(currency);

                var currentRate = this.fixedCurrencyRates.YND_FIXED[currency],
                    currentDate = u.moment().format('YYYYMMDD'),
                    lastDate = '0';

                //даты приходят в формате YYYYMMDD, так что сравнение их как чисел корректно
                Object.keys(currentRate).forEach(function(date) {
                    if (+lastDate < +date && +date <= +currentDate) lastDate = date;
                });

                return currentRate[lastDate].with_nds;
            },

            /**
             * Функция для вывода определённой суммы денег, в определённой валюте
             * @param {String} currency код валюты
             * @param {Number|String} sum сумма денег
             * @param {Object} [opts] дополнительные параметры форматирования {delim: '&nbsp', nds: 1}
             * @param {String} [opts.delim] разделитель между числом и названием валюты
             * @param {Number|undefined} [opts.discount] скидка
             *  (1 - включая НДС, 0 - без учёта НДС, undefined - ничего не пишем про НДС)
             * @param {1|0|undefined} [opts.nds] указывать ли информацию о НДС

             * @returns {String} сумма денег в валюте
             */
            formatSum: function(currency, sum, opts) {
                if (sum < 0.01 && sum >= EPSILON) {
                    sum = 0.01;
                }

                currency = formatCurrencyCode(currency);
                opts = opts || {};
                opts.exponentDelimiter = NBSP;

                var delim = opts.delim || ' ',
                    res = u.numberFormatter.format(sum, opts) + delim + u.currencies.getName(currency, opts);

                if (opts.nds !== undefined && currency !== 'YND_FIXED') {
                    if (opts.discount !== undefined) {
                        if (opts.nds) {
                            res = iget2('i-utils', 's-vklyuchaya-nds-i', '{foo} включая НДС и без учёта скидки', {
                                foo: res
                            });
                        } else {
                            res = iget2('i-utils', 's-bez-uchyota-nds', '{foo} без учёта НДС и скидки', {
                                foo: res
                            });
                        }
                    } else {
                        if (opts.nds) {
                            res = iget2('i-utils', 's-vklyuchaya-nds', '{foo} включая НДС', {
                                foo: res
                            });
                        } else {
                            res = iget2('i-utils', 's-bez-uchyota-nds-187', '{foo} без учёта НДС', {
                                foo: res
                            });
                        }
                    }
                }

                return res;
            },

            /**
             * Возвращает строчку "1 у.е. = ..."
             * @param {String} currency
             * Не используется в псевдо валюте
             * @returns {String}
             */
            convUnitExplanation: function(currency) {
                currency = formatCurrencyCode(currency);

                return '1 ' + iget2('i-utils', 'u-e', 'у.е.') + NBSP + '=' + NBSP + u.currencies.formatSum(currency, u.currencies.getConvUnitRate(currency));
            }
        }
    });

})();
