(function() {

    var EPSILON = 1e-7,
        numberGroupsReg = /(\d)(?=(\d\d\d)+(?!\d))/g;

    /**
     * Округляет дробную часть переданного значения оставляя 2 знака после запятой
     *
     * @param {Number} val
     * @returns {Number}
     */
    function round2s(val) {
        return !val ? 0 : Math.round(parseInt(val * 1e6) / 1e6 * 100) / 100;
    }

    /**
     * Округляет дробную часть переданного значения оставляя n знаков после запятой
     *
     * @param {Number} val
     * @param {Number} n - кол-во знаков после запятой
     * @returns {Number}
     */
    function roundNs(val, n) {
        var pow = Math.pow(10, n);
        return !val ? 0 : Math.round(parseInt(val * 1e6) / 1e6 * pow) / pow;
    }

    /**
     * Добавляет разделитель между каждыми тремя цифрами числа
     *
     * @param {Number} number число
     * @param {String} [separator] разделитель, по умолчанию пробел
     * @returns {String}
     * @private
     */
    function _addNumberGroupSeparators(number, separator) {
        return (number + '').replace(numberGroupsReg, '$1' + (separator || String.fromCharCode(160)));
    }

    /**
     * Форматирует переданное значение цены.
     *
     * @param {Number} price цена
     * @param {String} [separator] разделитель групп
     * @returns {Number|String}
     */
    function formatPrice(price, separator) {

        if (price === undefined)  return price;

        //Если значение больше 1e-7 но меньше 0.01 округляем до 0.01
        price = price >= EPSILON && price < 0.01 ? 0.01 : round2s(price).toFixed(2);

        if (separator !== '') {
            price = _addNumberGroupSeparators(price, separator);
        }

        return price;
    }

    /**
     * Форматирует показы или средства
     *
     * @param {Number|String} value показы или средства кампании
     * @param {String} [name] тип переданного значения (shows, money)
     * @returns {String}
     */
    function formatShowsOrMoney(value, name) {
        if (!value || value === '-' || (name && ['shows', 'money'].indexOf(name) < 0)) {
            return name != 'shows' && value === 0 ? value.toFixed(2) + '' : value;
        }

        return name === 'shows' ?
            _formatUnitsShows(value) :
            formatPrice(value);
    }

    /**
     * Форматирует количество показов
     *
     * @param {Number} shows
     * @returns {String}
     * @private
     */
    function _formatUnitsShows(shows) {

        // форматируем нулями
        var res = ((shows || 0) / 1000).toFixed(3);

        if (res === '0.000') {

            return '0';
        }

        return _addNumberGroupSeparators(res);
    }

    /**
     * Форматирует число с разделением цифр на группы
     * Преобразует число к целому
     *
     * Пример:
     * u.formatInt(1234123456789.123456) ->  1 234 123 456 789
     * u.formatInt(1234123456789.123456, '-X-') -> 1-X-234-X-123-X-456-X-789
     *
     * @param {Number} val значение для форматирования
     * @param {String} [separator] разделитель, неразрывный пробел по умолчанию
     * @returns {String|undefined}
     */
    function formatInt(val, separator) {

        if (val === undefined) return val;

        return _addNumberGroupSeparators(parseInt(val), separator);
    }

    /**
     * Подставляет в строку переданные аргументы в нужные места
     * (плейсхолдер обозначается как %s)
     * @param {String} string - строка форматирования
     * @returns {String}
     */
    function formatString(string) {
        var args = [].slice.call(arguments, 1),
            i = 0;

        return string.replace(/%s/g, function() {
            return args[i++];
        });
    }

    /**
     * Превращает число в удобный для воспринятия человеком формат,
     * пр. f(213245213, 2)  -> 213.24 млн
     *
     * @param {Number} numb - число
     * @param {Number} precision - точность после запятой
     * @returns {*|String|BEMJSON}
     */
    function humanizeNumberValue(numb, precision) {
        var preparedValue = String(roundNs(numb, 0)),
            countOfParts = Math.floor((preparedValue.length - 1) / 3),
            value = numb / Math.pow(1000, countOfParts),
            numericAbbreviations = ['',
                iget2('i-utils', 'ns-th', 'тыс'),
                iget2('i-utils', 'ns-m', 'млн'),
                iget2('i-utils', 'ns-b', 'млрд'),
                iget2('i-utils', 'ns-tr', 'трлн'),
                iget2('i-utils', 'ns-q', 'квадрлн', { context: 'квадриллион' }),
                iget2('i-utils', 'ns-qi', 'квнтлн', { context: 'квинтиллион' }),
                iget2('i-utils', 'ns-sx', 'скстлн', { context: 'секстиллион' }),
                iget2('i-utils', 'ns-sp', 'сптлн', { context: 'септиллион' }),
                iget2('i-utils', 'ns-o', 'октлн', { context: 'октиллион' }),
                iget2('i-utils', 'ns-n', 'нонлн', { context: 'нониллион' }),
                iget2('i-utils', 'ns-dc', 'дцлн', { context: 'дециллион' })
            ];

        return iget2('i-utils', 'humanize-number', '{number}&nbsp;{abbr}', {
            number: roundNs(value, precision),
            abbr: numericAbbreviations[countOfParts],
            context: '100 млн'
        });
    }

    u.register({
        formatString: formatString,
        formatInt: formatInt,
        round2s: round2s,
        formatPrice: formatPrice,
        formatShowsOrMoney: formatShowsOrMoney,
        humanizeNumberValue: humanizeNumberValue
    });
})();
