(function() {
    /**
     * Набор выполняемых проверок по имени стратегии
     */
    var validationByName = {
        'default': [
            'no-metrika',
            'metrika-goal-deleted',
            'metrika-domains-count',
            'metrika-cpa-deviation',
            'metrika-apc-deviation'
        ],
        'autobudget': [
            'no-metrika-optional-visits',
            'metrika-goal-deleted',
            'metrika-domains-count',
            'metrika-cpa-deviation',
            'metrika-apc-deviation'
        ],
        'autobudget_avg_cpa': [
            'no-metrika-optional-visits',
            'metrika-goal-deleted',
            'metrika-domains-count',
            'metrika-cpa-deviation',
            'metrika-apc-deviation'
        ],
        'autobudget_optimization_cpa': [
            'no-metrika',
            'metrika-goal-deleted',
            'metrika-cpa-deviation',
            'metrika-apc-deviation'
        ],
        'autobudget_roi': [
            'no-metrika-optional-visits',
            'metrika-goal-deleted',
            'metrika-domains-count',
            'metrika-cpa-deviation',
            'metrika-apc-deviation'
        ],
    },

    /**
     * Список функций-проверок
     */
    checks = {
        'no-metrika': checkMetrika,
        'no-metrika-optional-visits': checkMetrika,
        'metrika-goal-deleted': function(data) {
            return data.goalId !== '' && (data.goal.goal_status === 'Deleted' || data.goal.counter_status == 'Deleted')
        },
        'metrika-domains-count': function(data) {
            return data.metrika.compaign_domains_count > 1;
        },
        'metrika-cpa-deviation':  function(data) {
            return data.metrika.cpa_deviation;
        },
        'metrika-apc-deviation': function(data) {
            return data.metrika.apc_deviation
        }
    };

    function checkMetrika(data) {
        if (u.consts('failedToFetchMetrika')) {
            // если метрика не работает - то не нужно дизейблить стратегии
            return false;
        }
        return (!data.goals || !data.goals.length || data.allCountersDeleted || data.allGoalsDeleted);
    }

    /**
     * Вспомогательный метод выполнения списка проверок
     * @param {String} name – название стратегии или 'default'
     * @param {Object} metrika – данные по Яндекс Метрике (dm-metrika json)
     * @param {String} goalId – номер цели
     * @param {String} where – площадка
     * @param {Number} minGoalsOnCampaign - минимальное количество статистики по цели, чтобы она считалась активной
     * @param {String} metrikaCounter - номер счетчика метрики (пусто, если счетчик не подключен)
     * @returns {String|undefined}
     */
    function _runChecks(name, metrika, goalId, where, minGoalsOnCampaign, metrikaCounter) {
        var data = {
            metrika: metrika || {},
            goals: metrika.campaign_goals || [],
            goal: {},
            allCountersDeleted: true,
            allGoalsDeleted: true,
            goalId: goalId,
            where: where === 'context' ? where + '_' : '',
            metrikaCounter: metrikaCounter
        };

        data.goals.forEach(function(goalData) {
            var isCurrent = goalData.goal_id === goalId;

            isCurrent && (data.goal = goalData);

            if (goalData.counter_status === 'Active' || isCurrent) {
                data.allCountersDeleted = false;
            }
            if (goalData.goal_status === 'Active' || isCurrent) {
                data.allGoalsDeleted = false;
            }
        });

        var result = (validationByName[name] || validationByName.default).reduce(function(err, ruleName) {
            if (!err && checks[ruleName](data, minGoalsOnCampaign)) {
                err = ruleName;
        }

            return err;

        }, false);
        return result;
    }

    u.register({
        /**
         * Возвращает идентификатор предупреждения Яндекс Метрики
         * @param {Object} metrika – данные по Яндекс Метрике (dm-metrika json)
         * @param {String} goalId – номер цели
         * @param {String} where – площадка
         * @param {String} strategyName – название стратегии
         * @param {Number} minGoalsOnCampaign - минимальное количество статистики по цели, чтобы она считалась активной
         * @param {String} metrikaCounter - номер счетчика метрики (пусто, если счетчик не подключен)
         * @returns {String|undefined}
         */
        getMetrikaWarning: function(metrika, goalId, where, strategyName, minGoalsOnCampaign, metrikaCounter) {
            return _runChecks(strategyName, metrika, goalId, where, minGoalsOnCampaign, metrikaCounter);
        }
});

})();
