describe('i-utils__number-formatter', function() {
   var formatter = BEM.blocks['i-utils'].numberFormatter;

   describe('format', function() {
       var format = formatter.format;

       it('must round numbers to closest', function() {
           expect(format(10.123)).to.be.equal('10.12')
           expect(format(10.129)).to.be.equal('10.13')
       });

       it('must floor numbers correctly', function() {
           expect(format(10.123, { roundType: 'floor' })).to.be.equal('10.12');
           expect(format(10.129, { roundType: 'floor' })).to.be.equal('10.12');
       });

       it('must format big numbers correctly', function() {
           expect(format(5e+25)).to.be.equal('50 000 000 000 000 000 000 000 000.00');
       });

       it('must format small numbers correctly', function() {
           expect(format(5e-8)).to.be.equal('0.00');
       });

       it('must fill in trailing zeros, if needed', function() {
           expect(format(10.3)).to.be.equal('10.30')
       });

       it('must format null to zero', function() {
           expect(format(null)).to.be.equal('0.00');
       });

       it('must delimit thousands', function() {
           expect(format(12345)).to.be.equal('12 345.00')
       });

       it('must return NaN if input is not a number', function() {
           expect(format('abc')).to.be.eql(NaN);
           expect(format('12a')).to.be.eql(NaN);
       });

       it('must allow specify desired precision', function() {
           expect(format(12.3456, { precision: 3 })).to.be.equal('12.346')
           expect(format(12.3456, { precision: 0 })).to.be.equal('12')
       });

       it('must allow specify desired exponent delimeter', function() {
           expect(format(12345, { exponentDelimiter: ',' })).to.be.equal('12,345.00')
       });

       it('must allow specify desired float delimeter', function() {
           expect(format(12.345, { floatDelimiter: ',' })).to.be.equal('12,35')
       });

       it('must allow specify desired fail value', function() {
           expect(format('penguin', { fail: 'not a number' })).to.be.equal('not a number');
       });
   });

   describe('round', function() {
       var round = formatter.round;

       it('must round number to closest', function() {
           expect(round(10.123)).to.be.equal(10.12);
           expect(round(10.129)).to.be.equal(10.13);
       });

       it('must round null to zero', function() {
           expect(round(null)).to.be.equal(0);
       });

       it('must return NaN if input is not a number', function() {
           expect(round('abc')).to.be.eql(NaN);
           expect(round('12a')).to.be.eql(NaN);
       });

       it('must ceil if specified roundType is ceil', function() {
           expect(round(10.123, { roundType: 'ceil' })).to.be.equal(10.13);
       });

       it('must floor if specified roundType is floor', function() {
           expect(round(10.119, { roundType: 'floor' })).to.be.equal(10.11);
       });

       it('must allow specify desired fail value', function() {
           expect(round('penguin', { fail: 'not a number' })).to.be.equal('not a number');
       });
   });

   describe('clear', function() {
       var clear = formatter.clear;

       it('must replace possible mistyped dot to dot', function() {
           expect(clear('12,34')).to.be.equal('12.34');
           expect(clear('12/34')).to.be.equal('12.34');
           expect(clear('12ю34')).to.be.equal('12.34');
           expect(clear('12Ю34')).to.be.equal('12.34');
           expect(clear('12б34')).to.be.equal('12.34');
           expect(clear('12Б34')).to.be.equal('12.34');
           expect(clear('12<34')).to.be.equal('12.34');
           expect(clear('12>34')).to.be.equal('12.34');
       });

       it('must fill in integer zero part if needed', function() {
           expect(clear('.2')).to.be.equal('0.2');
       });

       it('must trim leading zeros', function() {
           expect(clear('012')).to.be.equal('12');
           expect(clear('0')).to.be.equal('0');
       });

       it('must return value unmodified if input is not a number', function() {
           expect(clear('penguin')).to.be.equal('penguin');
       });

       it('must remove all non-digits characters', function() {
           expect(clear('blah 123 blah!')).to.be.equal('123')
       });
   });
});
