(function() {
    /**
     * @class NumberFormatter
     * @private
     * @type {Object}
     * @property {String} sign знак числа
     * @property {Array} coefficients массив цифр коэффициэнта числа
     * @property {String} exponent значение экспоненты числа
     */
    function NumberFormatter(number) {
          var match = String(+number).toLowerCase().match(/^0*?(-?)(\d+\.?\d*)(e([+-]?\d+))?$/),
              sign,
              coefficients,
              exponent,
              dot;

          if (!match) {
              this.notNumber = true;
              return;
          }

          sign = match[1];
          coefficients = match[2];
          exponent = parseFloat(match[4] || '0');

          dot = coefficients.indexOf('.');
          exponent += (dot !== -1) ? (dot - 1) : (coefficients.length - 1);

          this.sign = sign;
          this.coefficients = coefficients
              // удаляем точку (должна быть удалена перед удалением пробелов вначале строки)
              .replace('.', '')
              .replace(/^0*/, function(zeros) {
                // удаляем пробелы в начале строки и учитываем это в экспоненте
                exponent -= zeros.length;
                return '';
              })
              // удаляем нули в конце строки
              .replace(/0*$/, '')
              .split('')
              .map(function(d) {
                return parseInt(d);
              });

          if (this.coefficients.length === 0) {
              this.coefficients.push(0);
              exponent++;
          }

        this.exponent = exponent;
        this.isZero = this.coefficients[0] == 0 && this.exponent == 0;
    }

    /**
     * Округляет числовой объект до нужного знака после запятой
     * @param {Number} precision количество знаков после запятой
     * @param {('round'|'floor'|'ceil')} options.roundType тип округления
     * @returns {NumberFormatter}
     */
    NumberFormatter.prototype.round = function(precision, roundType) {
        var result = u._.assign({}, this),
            coefficients = result.coefficients,
            fraction,
            i,
            removed;

        fraction = result.exponent + 1 + precision;

        // добавляем нули в начало при необходимости
        while (fraction <= 0) {
            coefficients.unshift(0);
            result.exponent++;
            fraction++;
        }

        // округляем число до заданной точности
        if (coefficients.length > fraction) {
            removed = coefficients.splice(fraction, coefficients.length - fraction);

            if ((roundType === 'round' && removed[0] >= 5) || roundType === 'ceil') {
                i = fraction - 1;

                coefficients[i]++;

                while (coefficients[i] === 10) {
                    coefficients.pop();

                    if (i === 0) {
                        coefficients.unshift(0);
                        result.exponent++;
                        i++;
                    }

                    i--;
                    coefficients[i]++;
                }
            }
        }

        return result;
    };

    /**
     * Возвращает отформатированное строчное представление числового объекта
     * @param {Object} options
     * @param {Number} options.precision округление
     * @param {('round'|'floor'|'ceil')} options.roundType тип округления
     * @param {String} options.exponentDelimiter разделитель разрядов
     * @param {String} options.floatDelimiter десятичный разделитель
     * @param {Boolean} [options.showPositiveSign] добавляет «+»
     * @returns {String}
     */
    NumberFormatter.prototype.getFormattedString = function(options) {
        var roundedNumberFormatter,
            coefficients,
            fraction,
            exponent,
            sign;

        // округляем до заданной точности
        roundedNumberFormatter = this.round(options.precision, options.roundType);
        sign = roundedNumberFormatter.sign;

        coefficients = roundedNumberFormatter.coefficients;
        exponent = roundedNumberFormatter.exponent + 1;
        fraction = exponent + options.precision;

        // добавляем нули в конец, если нужно
        if (coefficients.length < fraction) {
            coefficients = coefficients.concat(getZerosArray(fraction - coefficients.length));
        }

        // добавляем нули в начало, если нужно
        if (exponent < 0) {
            coefficients = getZerosArray(-exponent + 1).concat(coefficients);
            exponent = 1;
        }

        // вставляем разделитель для float чисел
        if (options.precision) {
            coefficients.splice(exponent, 0, (exponent === 0) ? ('0' + options.floatDelimiter) : options.floatDelimiter);
        }

        if (options.showPositiveSign && !roundedNumberFormatter.isZero && sign !== '-') {
            sign = '+';
        }

        return sign
            + formatIntegerPart(coefficients.slice(0, exponent).join(''), options.exponentDelimiter)
            + coefficients.slice(exponent).join('');
    };

    /**
     * Создает массив из нулей
     * @param {Number} arrayLength количество элементов в массиве
     * @returns {Array}
     */
    function getZerosArray(arrayLength) {
        var arr = [],
            i;

        for (i = 0; i < arrayLength; i++) {
            arr.push(0);
        }

        return arr;
    }

    /**
     * Форматирует целую часть числа
     * @param {Number|String} number форматируемое число
     * @param {String} delimeter разделитель тысячных разрядов
     * @returns {String}
     */
    function formatIntegerPart(number, delimeter) {
        var decRegexp = /(\d)(\d\d\d)($|\D)/,
            replaced;

        number = number + '';

        while ((replaced = number.replace(decRegexp, '$1' + delimeter + '$2$3')) != number) {
            number = replaced;
        }

        return number;
    }

    /**
     *  возвращает округленное число
     */
    function roundNumber(value, options) {
        options = u._.assign({
            precision: 2,
            roundType: 'round',
            fail: NaN
        }, options);

        if (isNaN(value)) return options.fail;

        var pow = Math.pow(10, options.precision);

        value = Math[options.roundType || 'round'](multFloats(value, pow)) / pow;

        return value;
    }

    function multFloats(a, b) {
        var atens = Math.pow(10, String(a).length - String(a).indexOf('.') - 1),
            btens = Math.pow(10, String(b).length - String(b).indexOf('.') - 1);

        return (a * atens) * (b * btens) / (atens * btens);
    }

    var formatter = {
            /**
             * Форматирование числа с округлением.
             * @param {Number} number
             * @param {Object} options
             * @param {Number} options.precision округление
             * @param {('round'|'floor'|'ceil')} options.roundType тип округления
             * @param {String} options.exponentDelimiter разделитель разрядов
             * @param {String} options.floatDelimiter десятичный разделитель
             * @param {Boolean} [options.showPositiveSign] добавляет «+»
             * @param {*} options.fail результат, возвращаемый при неудаче
             * @returns {String}
             */
            format: function(number, options) {
                var numberObject;

                options = u._.assign({
                    precision: 2,
                    roundType: 'round',
                    exponentDelimiter: ' ',
                    floatDelimiter: '.',
                    fail: NaN
                }, options);

                numberObject = new NumberFormatter(number);

                return numberObject.notNumber ? options.fail : numberObject.getFormattedString(options);
            },

            /*
             * Простое округление - возвратит всегда число
             */
            round: function(number, options) {
                options = u._.assign({
                    precision: 2,
                    roundType: 'round',
                    fail: NaN
                }, options);

                return roundNumber(number, options);
            },

            /*
             * Округление с форматированием, возвратит строку с заданными параметрами окгруления
             * Например округляя 2 с точностью 4, вернёт строку 2.0000
             * @deprecated используйте format
             */
            roundFormatted: function(number, options) {
                options = u._.assign({
                    precision: 2,
                    roundType: 'round',
                    fail: NaN,
                    exponentDelimiter: '',
                    floatDelimiter: '.'
                }, options);

                return formatter.format(number, options);
            },

            /**
             * Пытается превратить переданную строку(пользовательский ввод) в строку содержащую число
             * @param {Number} number исходная строка, которую нужно превратить в число
             * @param {Object} options
             * @param {*} options.fail результат, возвращаемый при неудаче
             * @param {Number} options.precision округление
             * @param {String} options.mode режим, если strict, то
             * @returns {String|*}
             */
            clear: function(number, options) {
                options = u._.assign({
                    fail: number,
                    precision: 2
                }, options);

                //заменяем потенциальные варианты точки на точку
                number = (number + '').replace(/[\/,.юЮбБ<>]/gi, '.'); // iget:ignore

                // заменяем .2 -> 0.2
                number = number.replace(/^\.([^.]+)$/, '0.$1');

                //удаляем первый 0 в числах вида 034 -> 34, если number == '0' пропускаем
                number = number.replace(/^0+(?!\.|$)/g, '');

                //если number - строка с записью числа в формате с экспонентой, преобразовываем в обычный формат
                if (!isNaN(number) && !!(number + '').match(/e/)) {
                    number = (+number).toFixed(options.precision);
                }

                if (!isNaN(number)) return number;

                number = number.replace(/[^0-9.]/g, '') //удаляем все символы кроме чисел и .

                return (isNaN(number) || number == '') ? options.fail : number;
            }

        };

    u.register({
        numberFormatter: formatter
    });
})();
