(function() {
    /**
     * Объект с полями размеров
     * @typedef Object SizeObject
     * @property {Number} width
     * @property {Number} height
     */

    //Название ключа-шаблона для getPreviewUrl
    var templateKey = '__template__',
        chunkType = {
            COMMON: 'COMMON',
            TEMPLATE: 'TEMPLATE'
        },
        chunksToString = function(chunks) {
            return chunks.map(function(chunk) {
                return chunk.value;
            }).join('');
        },
        getPreviewUrl,
        replaceParams,
        addParams,
        RUS_LETTERS = '\u0430-\u043F\u0440-\u044F\u0410-\u042F\u0451\u0401',
        RUS_LAT_LETTERS = 'a-zA-Z' + RUS_LETTERS,
        RUS_LAT_LETTERS_AND_DIGITS = RUS_LAT_LETTERS + '\\d',
        TITLE_LENGTH_LIMIT = 56,
        TITLE_PX_LIMIT = 517,
        RE_SIGNS = /[.!?…]/g,
        RE_TO_SPLIT_SENTENCES = /([…!?]+\s*|\.{3}\s*|\.+\s+|\.+(?=[A-ZА-ЯЁҐЄІЇÇĞİÖŞÜ]))/,
        END_SENTENCE_EXCLUSION = ['им', 'с', 'яндекс', 'и', 'тыс', 'млн', 'абон', 'м', 'юр', 'г', 'руб', 'р', 'ак', '₽'],
        // END_SENTENCE_EXCLUSION_REGEXP используется для определения исключений в следующих выражениях
        // текст600руб. || 600руб. || 600 руб. согласно правилу 7 и правилу 22 в требованиях
        // https://wiki.yandex-team.ru/users/nadiano/movingtext/
        END_SENTENCE_EXCLUSION_REGEXP = new RegExp('([0-9]|\\s)+(' + END_SENTENCE_EXCLUSION.join('|') + ')$', 'i'),
        TITLE_SENTENCE_SEPARATOR = ' – ',
        PAIR_DIFFERENT_SIGN = [
            ['(', ')'],
            ['{', '}'],
            ['<', '>'],
            ['«', '»']
        ],
        PAIR_SAME_SIGN = ['"', '¨', "'"],
        _bracketRegExp,
        /**
         * Проверяет строку на наличие в ней скобок/кавычек, указанных в PAIR_DIFFERENT_SIGN и PAIR_SAME_SIGN
         * @param {String} str
         * @returns {Boolean}
         */
        isBracket = function(str) {
            _bracketRegExp || (_bracketRegExp = new RegExp(PAIR_DIFFERENT_SIGN
                .reduce(function(result, item) { return result.concat(item) }, PAIR_SAME_SIGN)
                .map(u.escape.regExp)
                .join('|')));

            return _bracketRegExp.test(str);
        },
        _apostropheRegexp,
        /**
         * Проверяет строку на наличие в ней апострофа (одинарной кавычки)
         * по бокам от кавычки должны быть буквы
         * @param {String} str
         * @returns {Boolean}
         */
        isApostrophe = function(str) {
            _apostropheRegexp ||
                (_apostropheRegexp = new RegExp('[' + RUS_LAT_LETTERS_AND_DIGITS + ']\'[' + RUS_LAT_LETTERS_AND_DIGITS + ']', 'i'));

            return _apostropheRegexp.test(str);
        },
        _bracketsHelperObject,
        /**
         * Возвращает вспомогательный объект, для подсчета открывающихся и закрывающихся скобок/кавычек
         * @returns {Object}
         */
        getBracketsHelperObject = function() {
            if (!_bracketsHelperObject) {
                _bracketsHelperObject = {};

                //{ '>': '<', '}': '{', ')': '(' },
                _bracketsHelperObject = PAIR_DIFFERENT_SIGN.reduce(function(res, item) {
                    res[item[1]] = item[0];

                    return res;
                }, _bracketsHelperObject);

                // { ": null, ': null }
                _bracketsHelperObject = PAIR_SAME_SIGN.reduce(function(res, item) {
                    res[item] = null;

                    return res;
                }, _bracketsHelperObject);
            }

            return _bracketsHelperObject;
        },
        /**
         * Возвращает список индексов, между которыми нужно игнорирвать разделитель предложений
         * возвращает массив объектов, `s`, в которых - индекс с которого нужно начинать игнорировать разделитель,
         * `e` - индекс с которого нужно перестать игнорировать. Если возвращает `false` - значить нарушен баланс
         * скобок/кавычек, пустой массив - значит скобок/кавычек нет
         * @param {String} str
         * @returns {{ s: number, e: number }[]|[]|false}
         */
        getIgnoredIndexesByBrackets = function(str) {
            if (!isBracket(str)) return [];

            var bracketsHelperObject = getBracketsHelperObject(),
                openBrackets = [],
                isClean = true,
                strLength = str.length,
                symbol,
                i,
                ignoredMap = [],
                ignoredMapIndex = 0,
                countOpenBrackets = 0,
                /**
                 * Увеличивает счетчик открытых скобок/кавычек, заполняет список игнорируемых индексов
                 * @param {Number} i индекс
                 */
                incrementOpenBracket = function(i) {
                    countOpenBrackets || (ignoredMap[ignoredMapIndex] = { s: i });
                    countOpenBrackets++;
                },
                /**
                 * Уменьшает счетчик открытых скобок/кавычек, заполняет список игнорируемых индексов
                 * @param {Number} i индекс
                 */
                decrementOpenBracket = function(i) {
                    countOpenBrackets--;
                    if (!countOpenBrackets) {
                        ignoredMap[ignoredMapIndex].e = i;
                        ignoredMapIndex++;
                    }
                },
                qtyQuote = Array.prototype.reduce.call(str, function(sum, char) {
                    return char === '\'' ? ++sum : sum;
                }, 0);

            // проверяем на сбалансированность
            for (i = 0; isClean && i < strLength; i++) {
                symbol = str[i];

                // если символ скобка/кавычка, но не апостроф
                if (isBracket(symbol) &&
                    (symbol === "'" ?
                        (qtyQuote !== 1 && !isApostrophe((str[i - 1] || '') + symbol + (str[i + 1] || ''))) : true)) {

                    // открывающий и закрывающий символы одинаковые
                    if (bracketsHelperObject[symbol] === null) {
                        if (openBrackets[openBrackets.length - 1] === symbol) {
                            openBrackets.pop();
                            decrementOpenBracket(i);
                        } else {
                            openBrackets.push(symbol);
                            incrementOpenBracket(i);
                        }
                    // открывающий и закрывающий символы разные
                    } else {
                        if (bracketsHelperObject[symbol]){
                            isClean = (openBrackets.pop() === bracketsHelperObject[symbol]);
                            decrementOpenBracket(i);
                        } else {
                            openBrackets.push(symbol);
                            incrementOpenBracket(i);
                        }
                    }
                }
            }

            return isClean && !openBrackets.length && ignoredMap;
        },
        /**
         * Вытаскивает из, разбитого по знаками препинания, массива описания фразу
         * Возращает фразу, которую вытащили (с учетом исключений) для подстановки в заголовок
         * @param {String[]} splitedBody разбитое по знаком препинаня описание баннера
         * @param {Object[]} ignoredIndexes массив с индексами, разделители между которыми нужно игнорировать
         * @param {Number} [prevLength=0] Длина предыдущей части предложения
         * @returns {String}
         */
        getFirstSentenceFromSplittedBody = function(splitedBody, ignoredIndexes, prevLength) {
            // пример, splitedBody ==
            // Array [ "Большой выбор экскурсий", "... ", "От 400 руб/чел", ". ", "Бесплатно", "…", " Каждый день" ]
            if (splitedBody.length < 1) return '';

            prevLength || (prevLength = 0);

            // Вынимаем текст и знак препинания (по которому этот кусок отделился)
            var text = splitedBody.shift(),
                punctuation = splitedBody.shift() || '',
                phrase = text + punctuation,
                indexPunctuationInWholePhrase = prevLength + text.length,
                ignoreThisPunctuation = ignoredIndexes.some(function(obj) {
                    return obj.s < indexPunctuationInWholePhrase && obj.e > indexPunctuationInWholePhrase;
                });

            // Если в первом предложении есть сокращения-исключения (возможно, с числом в начале без пробела),
            // то ищется следующий знак окончания предложения.
            if (ignoreThisPunctuation ||
                punctuation.trim() === '.' && END_SENTENCE_EXCLUSION_REGEXP.test(text)) {

                phrase += getFirstSentenceFromSplittedBody(splitedBody, ignoredIndexes, prevLength + phrase.length);
            }

            return phrase.trim();
        },
        /**
         * Подставляет домен в заголовок
         * Если полученная строка не привышает TITLE_LENGTH_LIMIT - возвращаем результат
         * Иначе возвращается исходный заголовок
         * @param {String} title заголовок
         * @param {String} body описание
         * @param {String} domain домен
         * @returns {{title: string, body: string}}
         */
        getTitleAndBodyWithoutSubstitution = function(title, body, domain) {
            var titleWithDomain = title + (domain ? ' / ' + u.stripWww(domain) : '');

            return {
                title: titleWithDomain.length > TITLE_LENGTH_LIMIT ? title : titleWithDomain,
                body: body
            };
        };

    /**
     * Возвращает url с замененными параметрами
     * @param {String} url исходная ссылка
     * @param {Object} substParams хэш с параметрами замены
     * @returns {String}
     */
    replaceParams = function(url, substParams) {
        if (typeof substParams[templateKey] === 'undefined') {
            substParams[templateKey] = null; //добавляем ключ для шаблона, чтобы подставить позже значение по умолчанию
        }

        //Замена параметров
        Object.keys(substParams).forEach(function(key) {
            var substValue = substParams[key];

            if (key === templateKey) {
                url = url.replace(/#(.*)#/, substValue || '$1');
            } else {
                url = url.replace(new RegExp('{' + key + '}', 'g'), encodeURIComponent(substValue));
            }
        }, this);

        return url;
    };

    /**
     * Возвращает url с добавленными параметрами
     * @todo DIRECT-42123 использовать i-utils__url#formatUrl после исправления в ней ошибки с лишним слешем
     *
     * @param {String} url исходная ссылка
     * @param {Object} addParams хеш с параметрами, которые нужно добавить
     * @returns {String}
     */
    addParams = function(url, addParams) {
        var addKeys = Object.keys(addParams),
            hashPos,
            hash = '';

        if (addKeys.length) {
            // есть ли в ссылке якорь
            if (url.indexOf('#') != -1) {
                hashPos = url.lastIndexOf('#');
                hash = url.substring(hashPos);
                url = url.substring(0, hashPos);
            }

            Object.keys(addParams).forEach(function(key) {
                url += (url.indexOf('?') !== -1 ? '&' : '?') + key + '=' + addParams[key];
            }, this);

            url += hash;
        }

        return url;
    };

    /**
     * Возвращает url с замененными плейсхолдерами параметров
     * @param {String} href ссылка с параметрами
     * @param {PreviewUrlParams} params параметры замены
     * @returns {String}
     */
    getPreviewUrl = function(href, params) {
        if (!href) return '';

        if (!/^(http|https):/i.test(href)) {
            href = 'http://' + href;
        }

        params = params || {};

        href = replaceParams(href, params.subst || {});
        href = addParams(href, params.add || {});

        return href;
    };

    u.register({
        /**
         * Заменяет в строке вхождение шаблона на элемент template блока b-banner-preview
         *
         * @example u.replaceTemplate('купите #запчасти#', 33, 'карданный вал') // 'купите <div class='b-banner-preview__template'>карданный вал</div>'
         * @example u.replaceTemplate('купите #запчасти#', 33, 'стабилизатор поперечной устойчивости') // 'купите <div class='b-banner-preview__template'>запчасти</div>'
         * @example u.replaceTemplate('купите #запчасти#', 10, 'карданный вал') // 'купите <div class='b-banner-preview__template'>зап</div>'
         *
         * @param {String} string Исходная строка
         * @param {Number} limit Максимальная длина строки-результата. Если строка с подставленным templateValue укладывает в limit, то вернется она
         *                       Иначе вернется строка с подставленным значением по умолчанию (обрезанной до ограничения)
         * @param {String} [templateValue] Значение, которое нужно подставить вместо шаблона.
         *                                 Если не задано, то шаблон будет заменен на значение по умолчанию (содержимое между #)
         * @param {String} [templateClassName='b-banner-preview__template'] имя класса обертки шаблона
         * @param {Boolean} [withoutEscaping=false] нужно ли эскейпить шаблон
         * @returns {String} Строка с подставленным шаблоном
         */
        replaceTemplate: function(string, limit, templateValue, templateClassName, withoutEscaping) {
            var rawChunks,
                templatedChunks,
                resultChunks = [],
                templateFound;

            rawChunks = string.split(/(#[^#]*#)/).map(function(chunk) {
                var matches = chunk.match(/#([^#]*)#/),
                    result = {};

                if (matches && !templateFound) {
                    result.type = chunkType.TEMPLATE;
                    result.value = matches[1];

                    templateFound = true;
                } else {
                    result.type = chunkType.COMMON;
                    result.value = chunk;
                }
                return result;
            });

            if (templateValue) {
                templatedChunks = rawChunks.map(function(chunk) {
                    var result = { type: chunk.type };

                    result.value = chunk.type === chunkType.TEMPLATE ? templateValue : chunk.value;

                    return result;
                });
            }

            //ограничение по длине
            if (templatedChunks && chunksToString(templatedChunks).length <= limit) {
                resultChunks = templatedChunks;
            } else if (chunksToString(rawChunks).length <= limit) {
                resultChunks = rawChunks;
            } else {
                var resultLength = 0,
                    i = 0;

                do {
                    var chunk = rawChunks[i],
                        chunkString = chunk.value;

                    if (resultLength + chunkString.length > limit) {
                        chunkString = chunkString.substr(0, limit - resultLength);
                        chunk.value = chunkString;
                    }
                    resultChunks.push(chunk);
                    resultLength += chunkString.length;
                    i++;
                } while (resultLength < limit && typeof rawChunks[i] !== 'undefined');
            }

            //подстановка шаблона и экранирование
            return resultChunks.map(function(chunk) {
                var value = withoutEscaping ? chunk.value : u.escapeHTML(chunk.value);

                return chunk.type === chunkType.TEMPLATE ?
                    (templateClassName === null ?
                        value :
                        '<span class="' + (templateClassName || 'b-banner-preview__template') + '">' + value + '</span>'
                    ) :
                    value;
            }).join('');
        },

        /**
         * Проверяет текст на наличие в нем шаблона
         * @param {String} text
         * @returns {boolean}
         */
        matchTemplate: function(text) {
            return text && !!text.match(/#(.*)#/);
        },

        /**
         * Возвращает случайно сгенерированный yclid
         * @returns {Number}
         */
        generateYclid: function() {
            return Math.floor(Math.random() * Math.pow(10, 15));
        },

        /**
         * @typedef {Object} PreviewUrlParams
         * @property {Object} subst хеш, ключами которого являются имена, а значениями - строки, на которые параметры нужно заменить.
         *                    Например, { param1: 'blabla' } означает, что {param1} в ссылке будет заменен на 'blabla'
         * @property {Object} add хеш, пары ключ-значение которого будут добавлены к ссылке как query-параметры
         *                    Например, { _openstat: 'y1241ydec3' } означает, что к ссылке будет добавлена строка '&_openstat=y1241ydec3'
         */

        /**
         * @typedef {Object} Phrase
         * @property {String} phrase текст фразы
         * @property {String} param1 значение первого параметра фразы
         * @property {String} param2 значение второго параметра фразы
         */

        /**
         * Возвращает параметры для getPreviewUrl
         * @param {Phrase} [activePhrase]
         * @param {Boolean} openstat
         * @param {Boolean} clicktrack
         * @returns {PreviewUrlParams}
         */
        getPreviewUrlParams: function(activePhrase, openstat, clicktrack) {
            var hrefParams = {
                subst: {},
                add: {}
            };

            if (arguments.length < 3 && typeof activePhrase !== 'object') {
                clicktrack = openstat;
                openstat = activePhrase;
                activePhrase = undefined;
            }

            if (activePhrase) {
                ['param1', 'param2'].forEach(function(key) {
                    if (activePhrase[key]) {
                        hrefParams.subst[key] = activePhrase[key];
                    }
                });

                hrefParams.subst[templateKey] = activePhrase.key_words;
            } else {
                hrefParams.subst[templateKey] = '';
            }

            if (openstat) {
                hrefParams.add._openstat = 'dGVzdDsxOzE7';
            }

            if (clicktrack) {
                hrefParams.add.yclid = u.generateYclid();
            }

            return hrefParams;
        },

        getPreviewUrl: getPreviewUrl,

        preview: {

            /**
             * Возвращает base64 иконку 64x64 для превью иконки по умолчанию
             * @returns {String}
             */
            getEmptyIcon: function() {
                // новая картинка: DIRECT-52753
                /*jshint ignore:start*/
                return 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAABACAYAAACqaXHeAAAAAXNSR0IArs4c6QAABIRJREFUeAHtm91OE0EUx7fblq8CBeQC7tRXMMZLY0iI8drEhzHGGOPDmHhtDIkxXhrjK6h3cFGBAuWrlHp+q0O22247s0y3MwsnGZbunp2d/2++T7elIMVOTk7unp2dvby4uHgmabXb7VYvLy9LKe5OnQ7DsFsqldqVSqUh6eP09PS72dnZ34MK2SdIhNb39vbei/hNXwQPEhY/BxCBsLW8vPxCwDTj13oAtFqtB5K+ttvtWtypKP9Xq9VWrVZ7LOmH0nQFAPGHh4ffOp1OWV0s4rFcLncWFhYeKQghImn21HzRxaMVjWhFM58jAPT5ojZ7RCYNrWjmfInRfn9//2dRBryk2LTPDIxLS0v3QxntX9008UBBM9pDmeOfplEq+nm0A2C16ELT9KG9wgovzUHnvGQSkDBZdUVJ5z4XfNBeydr/z8/Pg4ODg4Bj3KampoLFxcWAo+uG9mgaNC2ozBxBo9HoE08+AOEaPj6YMQCau0ybI7Xho7rGSOcJOhgDkFUUK8eRRcYHX9fNGMDp6am2JhNf7UwtOxoDkLW0dhFMfLUztexoDECWkNpFMPHVztSyo76a/w82md5MfC3r0s7OGMDc3Jx25ia+2pladjQGMDMzE+gIwwdf180YAILq9XowPz+fqo1r+PhglSyFlMBitNyllpnqJMAQZSMxt6jW2RP4YtcqKUKHtQQfIGTqAi4JY7m9u7tLrC9TsbwGoMSzAZMYH1EeYwjeAqDG46IVDFMIXgJQ4pPNHghxKDrNwTsA1DAiETvImJFMIHgFYJR4BcQEgjcAiC8QZFFrDiU27agLwQsAiKdZJ+OPaeLVeSAAbdjA6DwAVfOm4hUE7hsGwWkAiG82m4F8g6P0ZDoqCOSXNKcBEHa3FVYDAt0oCcFZAIi3HVofBMFJAPKiRnB8fJxsrVY+JyE4B+Do6Gjs4fQ4BC0A9JthU4mVqpFM+B4BAHkYEJgdRsYD1EjMYESQU143i4IeBEVsGk2epp+nMbsMBUCtQwlaGEcSA5S8dhbBAMp1YTDYkeckLBUAOy3ED1p60ipoESR56ypqEQRACYmZGuKZ6ydlAwGobWVyuzmokPjQd0kqJggMwIwyAE5SPOXrA6AGhyyDHq2FRF8eNV7Q/2hhk7YeAKpGkqulLIUcNl4oyFnytX3PFYBx9cX4eMF3hXQPnmUDsg0YEQDm3jzmX7rVuFZ4WWFUmH5cK1RWMVnuC2+yeIBpLYWzkPXlnlsAvtTUuMp52wJkydofKBsXbsfyRXsof/59ue9Y4fIoDtpD2db+yeNhLj4D7aEsTbdcLFweZRLtn0LZtb25ieMAmkX723Btbe2XvN7+OQ/iLj0DzWiPpkH58dBzifW5/2azJYJoRTPZRQBWVlaa8rPSJ/LSU7YXbSwVLI9s0IhWNPO8ntDu9vb2Q/n66Ivs1wv501lqHvHr6+vfFeweAJyUN67qEqr6INvkDYn39V1XN/p0ZMCjz9PsVc2r8qcK3NnZuSehq9cSJtuU+N0dgVH1BQiCJbWZ55nmmekY8JTo+PEvdKHFg0nqdD4AAAAASUVORK5CYII=';
                /*jshint ignore:end*/
            },

            /**
             * Возвращает максимальный размер обрезки изображения
             * @returns {SizeObject}
             */
            getCropMaxSize: function() {
                return { width: 5000, height: 2812 };
            },

            /**
             * Возвращает хэш размеров иконок для сетей
             * @returns {Object} хэш с ключами идентификаторами размеров и значениями типа SizeObject
             */
            getPCodeIconSizes: function() {
                return {
                    'xld-retina': { width: 300, height: 300 },
                    'ldd-retina': { width: 180, height: 180 },
                    'ld-retina': { width: 160, height: 160 },
                    's-retina': { width: 32, height: 32 },
                    xld: { width: 150, height: 150 },
                    ldd: { width: 90, height: 90 },
                    ld: { width: 80, height: 80 },
                    s: { width: 16, height: 16 }
                };
            },

            /**
             * Возвращает фразу, пригодную к использованию в превью: удаляет операторы, раскрывает скобки
             * Не обрабатывает минус-слова, потому что минус-слова в интерфейсе - отдельная сущность (они изымаются в i-phrases-separate-into-groups)
             *
             * Про операторы можно почитать тут http://yandex.ru/support/direct/efficiency/refine-keywords.xml#refine-keywords
             * и тут http://yandex.ru/support/direct-news/n-2012-01-30.xml
             * @param {String} phrase ненормализованная фраза
             * @returns {String}
             */
            formatPhrase: function(phrase) {
                return phrase
                    .replace(/\!(\S+)/g, '$1') //!
                    .replace(/\+(\S+)/g, '$1') //+
                    .replace(/\[([^\]]+)\]/g, '$1') //[]
                    .replace(/"([^"]+)"/g, '$1'); //""
            },

            /**
             * Подставляет первое предложение из описания баннера в заголовок, есть ряд исключений, смотреть DIRECT-42716
             * @param {String} title заголовок
             * @param {String} body описание
             * @param {String} [domain] домен
             * @returns {{title: string, body: string}}
             */
            serpTitleSubstitution: function(title, body, domain) {
                title = title.trim();
                body = body.trim();

                var ignoredIndexes = getIgnoredIndexesByBrackets(body);

                // если нарушена сбалансированность скобок/кавычек то просто добавляем домен
                if (!ignoredIndexes) {
                    return getTitleAndBodyWithoutSubstitution.apply(this, arguments);
                }

                var // разбиваем описание на фразы по знакам препинания
                    // пример, 'Большой выбор экскурсий... От 400 руб/чел. Бесплатно… Каждый день' ->
                    // Array [ "Большой выбор экскурсий", "... ", "От 400 руб/чел", ". ", "Бесплатно", "…", " Каждый день" ]
                    splitedBody = body.split(RE_TO_SPLIT_SENTENCES).filter(function(a) { return a }),
                    // получаем первое предложение с учетом исключений
                    sentence = getFirstSentenceFromSplittedBody(splitedBody, ignoredIndexes),
                    // остаток описания, без выбранной фразы
                    remnantBody = (body.split(new RegExp('^' + u.escape.regExp(sentence), 'ig'))[1] || '').trim(),
                    titleLowerCaseWithoutSigns = title.toLowerCase().replace(RE_SIGNS, ''),
                    phraseLowerCase = sentence.toLowerCase().replace(RE_SIGNS, '');

                //Если в конце фразы для подстановки точка (и это не часть троеточия), то ее нужно обрезать DIRECT-58951
                if (!sentence.match(/[\.]{3}$/)) {
                    sentence = sentence.replace(/\.$/, '');
                }

                // Ширина заголовка после подстановки не должна привышать TITLE_PX_LIMIT
                // Если текст содержит только одну фразу, то перестановки не происходит.
                // При вхождении первого предложения в заголовок или заголовка в первое предложение текста без учета регистра, перестановки не происходит
                if (!remnantBody.length ||
                    u.charLength.getLength(title + TITLE_SENTENCE_SEPARATOR + sentence, 'arial', 18) > TITLE_PX_LIMIT ||
                    u._.contains(titleLowerCaseWithoutSigns, phraseLowerCase) ||
                    u._.contains(phraseLowerCase, titleLowerCaseWithoutSigns)) {

                    return getTitleAndBodyWithoutSubstitution.apply(this, arguments);
                }

                return {
                    title: title + TITLE_SENTENCE_SEPARATOR + sentence,
                    body: remnantBody
                };

            },

            /**
             * Типограф для бедных
             * @param {String} text Исходный текст
             * @returns {String} "Облагороженный" текст
             */
            prettifyText: function(text) {
                // не обрабатываем укр. объявления
                if (/[\u0404\u0406\u0407\u0454\u0456\u0457\u0490\u0491]/.test(text)) return text;

                var dashRegexp = new RegExp( '([' + RUS_LAT_LETTERS + ']) - ([' + RUS_LAT_LETTERS + '])', 'g'),
                    // [Аа]|[Бб]ез|[Вв](?:|ы|ас|ам|се|сё)|[Гг]де|[Дд](?:о|ля)|[Зз]а|[Ии](?:|з)|[Кк](?:|о|ак)|[Мм]ы|[Нн](?:а|ам|ас|е|и|о)|[Оо](?:|б|т)|[Пп](?:о|ро)|[Сс](?:|о)|[Тт](?:о|ут|ы)|[Уу]|[Чч]то|[Ээ]то
                    rusParts = '[\u0410\u0430]|[\u0411\u0431]\u0435\u0437|[\u0412\u0432](?:|\u044B|\u0430\u0441|\u0430\u043C|\u0441\u0435|\u0441\u0451)|[\u0413\u0433]\u0434\u0435|[\u0414\u0434](?:\u043E|\u043B\u044F)|[\u0417\u0437]\u0430|[\u0418\u0438](?:|\u0437)|[\u041A\u043A](?:|\u043E|\u0430\u043A)|[\u041C\u043C]\u044B|[\u041D\u043D](?:\u0430|\u0430\u043C|\u0430\u0441|\u0435|\u0438|\u043E)|[\u041E\u043E](?:|\u0431|\u0442)|[\u041F\u043F](?:\u043E|\u0440\u043E)|[\u0421\u0441](?:|\u043E)|[\u0422\u0442](?:\u043E|\u0443\u0442|\u044B)|[\u0423\u0443]|[\u0427\u0447]\u0442\u043E|[\u042D\u044D]\u0442\u043E',
                    nbsp = '\u00A0',
                    dash = '\u2013'; // &ndash;, если надо &mdash;, то будет \u2014

                return text
                    // больше одного пробела подряд
                    .replace(/([\s\u00A0]){2,}/g, '$1')
                    // убираем висячие короткие союзы/частицы (склеиваем через nbsp)
                    .replace(new RegExp(
                        '(^|\\(|\\s)(' + rusParts + ')\\s([\u00AB"$\\d' + RUS_LAT_LETTERS + '])', 'g'),
                             '$1$2' + nbsp + '$3')
                    // убираем пробелы перед некоторыми знаками препинания
                    .replace(/[\s\u00A0]+([.,!?])/g, '$1')
                    // заменяем на ndash, потому что mdash обычно слишком широкий
                    .replace(dashRegexp, '$1' + nbsp + dash + ' $2')
                    // заменяем в информации о скидках вида:
                    //    Скидка -50%
                    // дефис на нормальный минус (критично в шрифтах типа Arial)
                    .replace(/ \-(\d\d?)%/g, ' \u2212$1%')
                    // висячие слова в конце объявления длиной 1-2 символа склеиваем с основной строкой
                    .replace(/ ([^\s]{1,2})$/, nbsp + '$1')
                    // неразрываемый пробел в годах, вида 2012 г.
                    .replace(/(19\d\d|20[012]\d)[\s\u00A0]*(\u0433\.?)/g, '$1' + nbsp + '$2')
                    // неразрываемый пробел в ценах в рублях, вида 5 000 руб
                    .replace(/(\d+ (\d{3} ){0,2})\u0440\u0443\u0431([.,?!:;\s]|$)/g, function(str, p1, p2, p3) {
                        return p1.replace(/ /g, nbsp) + '\u0440\u0443\u0431' + p3;
                    });

            },

            /**
             * Форматирует значения для отображения в "видимой ссылке" баннера
             *
             * @param {String} value
             * @returns {string}
             */
            prettifyDisplayHref: function(value) {
                // şŞıİçÇöÖüÜĞğ - турецкие символы
                // ІіЇїЄєҐґ - украинские символы
                // ӘҒҚҢӨҮҰҺәғқңөүұһұү - казахские символы
                // ҐґІіЎў - белорусские символы

                return (value || '')
                    .replace(/[ _]/g, '-')
                    .replace(/[^a-zA-ZА-Яа-яЁё0-9şŞıİçÇöÖüÜĞğІіЇїЄєҐґӘҒҚҢӨҮҰҺәғқңөүұһұүЎў\/№#%-]/g, '')
                    .replace(/[\/]+/g, '/')
                    .replace(/[-]+/g, '-');
            },

            /**
             * Проверяет ссылку на наличие в ней протокола
             * Если его нет, подставляет протокол из protocol
             * @param {String} href
             * @param {String} protocol
             * @returns {String}
             */
            prepareHref: function(href, protocol) {
                return /^(http:|https:)/i.test(href) ?
                    href :
                    //protocol может приходить undefined
                    (protocol || 'http://') + href;
            },

            /**
             * Возвращает счетчик оставшихся символов по лимиту
             * без учета "узких" (бесплатных) символов
             * @param {String} str исходная строка
             * @param {Number} limit лимит символов
             * @param {String} narrowSymbols "узкие" символы
             * @returns {number}
             */
            strCounterWithoutNarrow: function(str, limit, narrowSymbols) {
                var narrowSymbolsRegExp = new RegExp('([' + narrowSymbols + '])', 'g'),
                    strWithoutNarrow = (str || '').replace(narrowSymbolsRegExp, '');

                return limit - strWithoutNarrow.length;
            },

            /**
             * Возвращает строку убирая знак решетки.
             * @param {String} str исходная строка
             * @returns {string} строка без знаков решетки
             */
            skipSharpSign: function(str) {
                return (str || '').replace(/\#([^\#]*?)\#/g, '$1');
            },

            TITLE_SENTENCE_SEPARATOR: TITLE_SENTENCE_SEPARATOR
        }
    });
}());
