(function() {
    var idx = {}, // плейсхолдер для индекса
        /**
         * Разбивает путь на массив
         * toPath('a.b[0].c') => ['a', 'b', '0','c']
         * Аналог lodash@4 _.toPath
         * @param {String} path
         * @returns {String[]}
         */
        toPath = function(path) {
            return path === '' ? [] : path.replace(/(\[(\d+)\])+?/g, '.$2').split('.');
        },
        /**
         * Хелпер, который позволяет преобразовывать функции, возвращающие ошибки в функции,
         * возвращающие { code: string, message: function }, где code - ключ в исходном хеше, а message - исходная функция
         * Такой формат ожидается в makeErrorMapper
         * @param {Object} obj объект, ключи которого - коды ошибок, а значения - функции-фабрики функций
         *                 получения текста ошибок (то, что ожидается в message в getRules в makeErrorMapper)
         * @returns {Object}
         */
        makeCommonCodes = function(obj) {
            return Object.keys(obj).reduce(function(acc, key) {
                acc[key] = function(fargs) {
                    return {
                        code: key,
                        message: obj[key](fargs)
                    };
                };

                return acc;
            }, {});
        };

    /**
     * @typedef {Object} getRulesOptions объект опций, поппадающий в функцию getRules
     * @property {Object} idx маркер индекса в пути
     */

    u.register({
        'web-api-errors': {
            /**
             * Функция, генерирующая функцию-маппер ошибок в тексты сообщений
             * @param {Function} getRules функция, принимающая getRulesOptions и возвращающая массив правил
             * @param {Function} getDefaultMessage функция, принимающая объект запроса
             *                                     и объект ошибки и возвращающая сообщение по умолчанию
             * @returns {Function}
             * @throws {String} исключение про неверные входные параметры
             */
            makeErrorMapper: function(getRules, getDefaultMessage) {
                var list = getRules({ idx: idx }).map(function(item) {
                    var itemPath = [].concat(item.path || []);

                    return {
                        pathLength: itemPath.length,
                        testPath: function(errObj) {
                            var path = errObj.path,
                                pathArr = toPath(path),
                                templatePathArr = pathArr.map(function(x) { return isFinite(x) ? idx : x; }),
                                startPath = templatePathArr.slice(0, itemPath.length);

                            return u._.isEqual(startPath, itemPath);
                        },
                        codes: item.codes.reduce(function(acc, next) {
                            acc[next.code] = next.message;

                            return acc;
                        }, {})
                    };
                });

                list = list.sort(function(a, b) {
                    return b.pathLength - a.pathLength;
                });

                return function(obj, err) {
                    for (var i = 0; i < list.length; i++) {
                        var item = list[i];

                        if (item.testPath(err) && err.code && item.codes[err.code]) {
                            return item.codes[err.code](obj, err);
                        }
                    }
                    return getDefaultMessage(obj, err);
                };
            },
            /**
             * Набор функций для получения стандартных пар {code, message} для общеупотребительных кодов ошибок
             */
            commonCodes: makeCommonCodes({
                'StringDefectIds.LENGTH_CANNOT_BE_MORE_THAN_MAX': function(opts) {
                    var fieldName = opts.fieldName;

                    return function(obj, err) {
                        return iget2('i-utils', 'znachenie-v-pole-fieldname', 'Значение в поле "{fieldName}" не должно превышать {maxLength} символов', {
                            fieldName: fieldName,
                            maxLength: err.params.maxLength
                        });
                    };
                },
                'StringDefectIds.MUST_CONTAIN_ONLY_SYMBOLS': function(opts) {
                    var fieldName = opts.fieldName;

                    return function(obj, err) {
                        return iget2(
                            'i-utils',
                            'v-pole-fieldname-dopuskaetsya',
                            'В поле "{fieldName}" допускается использование букв латинского, турецкого, русского, украинского, казахского алфавитов, цифр и знаков пунктуации',
                            {
                                fieldName: fieldName
                            }
                        );
                    };
                }
            })
        }
    });
})();
