BEM.decl('image-crop', {

    /**
     * Объект с координатами и размерами выбранной области
     *
     * @typedef {Object} jQueryCropSelection
     * @property {Number} x координата x1
     * @property {Number} y координата y1
     * @property {Number} x2 координата x2
     * @property {Number} y2 координата y2
     * @property {Number} w ширина
     * @property {Number} h высота
     */

    onSetMod: {

        js: function() {
            var onSelectionChange = this._onSelectionChange.bind(this);

            this._cropApi = this.params.cropApi;
            this._cropApi.setOptions({
                onChange: onSelectionChange,
                onSelect: onSelectionChange
            });
        }

    },

    /**
     * Активания редактора
     */
    enable: function() {
        this._cropApi.enable();
    },

    /**
     * Деактивация редактора
     */
    disable: function() {
        this._cropApi.release();
        this._cropApi.disable();
    },

    /**
     * Установка выбранной области
     * @param {Array} selection Массив координат
     */
    setSelect: function(selection) {
        this._cropApi.setSelect(selection);
    },

    /**
     * Установка фокуса
     */
    focus: function() {
        this._cropApi.focus();
    },

    /**
     * Получение координат выбранной области на миниатюре
     * @returns {jQueryCropSelection}
     */
    getScaledSelection: function() {
        return this._cropApi.tellScaled();
    },

    /**
     * Получения соотношения сторон выбранной области на миниатюре
     * @returns {Number}
     */
    getSelectionRatioValue: function() {
        var selection = this.getScaledSelection();

        return selection.w / selection.h;
    },

    /**
     * Получение размеров всей области обрезки
     * @returns {*}
     */
    getBounds: function() {
        return this._cropApi.getBounds();
    },

    /**
     * Установка желаемого соотношения сторон для области выбора
     * @param {String} ratio Соотношение сторон через двоеточие
     */
    setPreservedRatio: function(ratio) {
        this._ratio = ratio;
        this._cropApi.setOptions({ aspectRatio: this._ratioToValue(ratio) });
    },

    /**
     * Получения текущего желаемого соотношения сторон
     * @returns {String}
     */
    getPreservedRatio: function() {
        return this._ratio;
    },

    /**
     * Числовое значение заданного желаемого соотношения сторон
     * @returns {*}
     */
    getPreservedRatioValue: function() {
        return this._ratioToValue(this._ratio);
    },

    destruct: function() {
        this._cropApi.destroy();
    },

    /**
     * Конвертация строкового представления соотношения сторон в числовую
     * @param {String} ratio Соотношение сторон через двоеточие
     * @returns {Number}
     * @private
     */
    _ratioToValue: function(ratio) {
        return +ratio.replace(/^(\d+)(:(\d+))*$/, function(a, width, b, height) {
            return isNaN(height) ? 0 : width / height;
        });
    },

    /**
     * Обработчик события изменения выбранной области
     * @param {jQueryCropSelection} selection Координаты выбранной области
     * @private
     */
    _onSelectionChange: function(selection) {
        var minSize = this.params.minSize;

        // jсrop некорректно обрабатывает клик на картинку при aspectRatio != 1,
        // сбрасывая полностью выделенную область,
        // вместо того чтобы выделить прямоугольник размера minSize
        // Реализуем эту логику тут
        if (!this.params.allowEmptySelection && minSize && (selection.w === 0 || selection.h === 0)) {
            this.setSelect(this._getAdjustedSelection(
                selection.x,
                selection.y,
                minSize[0],
                minSize[1]
            ));
            return;
        }

        this.trigger('select', selection);
    },

    /**
     * Вписывает переданный прямоугольник внутрь области обрезки
     * и возвращает его координаты
     * @param {Number} x Координата по оси х верхней левой точки
     * @param {Number} y Координата по оси y верхней левой точки
     * @param {Number} width Ширина прямоугольника
     * @param {Number} height Высота прямоугольника
     * @returns {*[]}
     * @private
     */
    _getAdjustedSelection: function(x, y, width, height) {
        var size = this.getBounds(),
            selectionX = Math.min(x, size[0] - width),
            selectionY = Math.min(y, size[1] - height);

        return [
            selectionX,
            selectionY,
            selectionX + width,
            selectionY + height
        ];
    }

}, {

    /**
     * Создает редактор на заданной картинке
     * @param {jQuery} imageElem Элемент картинки
     * @param {Object} options Опции плагина jcrop
     * @returns {*}
     */
    create: function(imageElem, options) {
        var dfd = $.Deferred(),
            cropOptions = u._.extend({
                bgColor: 'transparent'
            }, options);

        imageElem.Jcrop(cropOptions, function() {
            dfd.resolve(BEM.create('image-crop', {
                cropApi: this,
                minSize: options.minSize,
                allowEmptySelection: options.allowEmptySelection
            }));
        });

        return dfd;
    }

});
