(function(){

    /**
     * @typedef {Object} RetargetingConditionGroupRuleDTO "группа правил" используется в "условии подбора аудитории"
     *
     * @property {String} type тип - как влияют друг на друга правила
     * @property {RetargetingConditionRuleDTO[]} ruleCollection массив "правил"
     */

    /**
     * @typedef {Object} RetargetingConditionRuleDTO "правило" используется в "группу правил"
     * @property {String} goalId  id записи на сервере
     * @property {String} name название правила
     * @property {String} domain домен
     * @property {String} type тип
     * @property {String} [day] кол-во дней, за которое считать посетителей
     * @property {Boolean} allowToUse доступно ли правило. Если цель/счетчик становятся не доступными = false
     */

    var retargetingConditionErrorMapper = u['web-api-errors'].makeErrorMapper(
        function(opts) {
            var idx = opts.idx,
                fieldNames = {
                    name: iget2('retargeting-condition-dto', 'nazvanie-usloviya-podbora-auditorii', 'Название условия подбора аудитории'),
                    description: iget2('retargeting-condition-dto', 'opisanie-usloviya-podbora-auditorii', 'Описание условия подбора аудитории')
                };

            return [
                {
                    path: ['name'],
                    codes: [
                        u['web-api-errors'].commonCodes['StringDefectIds.LENGTH_CANNOT_BE_MORE_THAN_MAX']({
                            fieldName: fieldNames.name
                        }),
                        u['web-api-errors'].commonCodes['StringDefectIds.MUST_CONTAIN_ONLY_SYMBOLS']({
                            fieldName: fieldNames.name
                        }),
                        {
                            code: 'DefectIds.INCONSISTENT_STATE_ALREADY_EXISTS',
                            message: function(obj, err) {
                                return iget2('retargeting-condition-dto', 'uslovie-podbora-auditorii-s', 'Условие подбора аудитории с таким именем уже существует');
                            }
                        }
                    ]
                },
                {
                    path: ['description'],
                    codes: [
                        u['web-api-errors'].commonCodes['StringDefectIds.LENGTH_CANNOT_BE_MORE_THAN_MAX']({
                            fieldName: fieldNames.description
                        }),
                        u['web-api-errors'].commonCodes['StringDefectIds.MUST_CONTAIN_ONLY_SYMBOLS']({
                            fieldName: fieldNames.description
                        })
                    ]
                },
                {
                    path: ['conditions'],
                    codes: [
                        {
                            code: 'DefectIds.INCONSISTENT_STATE_ALREADY_EXISTS',
                            message: function(obj, err) {
                                return iget2(
                                    'retargeting-condition-dto',
                                    'uslovie-podbora-auditorii-s-100',
                                    'Условие подбора аудитории с таким набором правил уже существует'
                                );
                            }
                        }
                    ]
                },
                {
                    path: ['conditions', idx, 'goals'],
                    codes: [
                        {
                            code: 'CollectionDefectIds.INVALID_COLLECTION_SIZE',
                            message: function(obj, err) {
                                var params = err.params;

                                return iget2(
                                    'retargeting-condition-dto',
                                    'kolichestvo-celey-v-pravile',
                                    'Количество целей в правиле условия подбора аудитории должно быть от {min} до {max}',
                                    {
                                        min: params.minSize,
                                        max: params.maxSize
                                    }
                                );
                            }
                        }
                    ]
                },
                {
                    path: ['conditions', idx, 'goals', idx, 'id'],
                    codes: [
                        {
                            code: 'DefectIds.OBJECT_NOT_FOUND',
                            message: function(obj, err) {
                                var id = u._.get(obj, err.path);

                                return iget2('retargeting-condition-dto', 'ne-naydena-cel-s', 'Не найдена цель с идентификатором {foo}', {
                                    foo: id
                                });
                            }
                        }
                    ]
                },
                {
                    path: [],
                    codes: [
                        {
                            code: 'RetargetingDefectIds.Gen.INVALID_GOALS_FOR_TYPE',
                            message: function(obj, err) {
                                return iget2(
                                    'retargeting-condition-dto',
                                    'uslovie-sostoyashchee-tolko-iz',
                                    'Условие, состоящее только из наборов с типом "Не выполнено ни одной", может использовать только цели Яндекс.Метрики'
                                );
                            }
                        }
                    ]
                }
            ];
        },
        function(obj, err) {
            return err.message || iget2('retargeting-condition-dto', 'ne-udalos-sohranit-uslovie', 'Не удалось сохранить условие подбора аудитории');
        });

    u.register({

        /**
         * Объект данных: "условия ретаргетинга"
         * @namespace retargeting-condition-dto
         */
        'retargeting-condition-dto': {

            /**
             * Тип операции
             * @const
             */
            _GROUP_RULE_TYPE: {
                OR: {
                    server: {
                        id: 'or'
                    }
                },
                ALL: {
                    server: {
                        id: 'all'
                    }
                },
                NONE: {
                    server: {
                        id: 'not'
                    }
                }
            },

            /**
             * Конвертирует внешние данные в формат приложения
             * @param {String} idSrv
             * @returns {String|null}
             *
             */
            convGroupRuleTypeToApp: function(idSrv) {
                var idApp,
                    type = this._GROUP_RULE_TYPE;

                for (idApp in type) {
                    if (type.hasOwnProperty(idApp) && type[idApp].server.id === idSrv) {

                        return idApp;
                    }
                }

                return null;
            },

            /**
             * Конвертирует данные приложения для передачи из приложения
             * @param {String} idApp
             * @returns {String}
             */
            convGroupRuleTypeToServer: function(idApp) {
                return this._GROUP_RULE_TYPE[idApp].server.id;
            },

            /**
             * Создает Data Transport Object "условия подбора аудиторий"
             * Конвертирует данные от сервера в формат приложения
             * @param {{ ret_cond_id, ClientID, condition_name, condition_desc, is_accessible, properties, condition,
             *         is_used }} raw
             * @constructor
             * @throw { message: String, items: String[], sourceData: * } объект ошибки
             */
            RetargetingConditionDTO: function(raw) {
                if (!(this instanceof u['retargeting-condition-dto'].RetargetingConditionDTO)) {
                    return new u['retargeting-condition-dto'].RetargetingConditionDTO(raw);
                }


                /**
                 * Объект ошибки при валидации
                 * @type {{ message: String, items: String[], sourceData: * }}
                 */
                var error = {
                    items: []
                };

                /**
                 * id записи
                 * @type {Number}
                 */
                this.id = raw.id;

                /**
                 * Название
                 * @type {String}
                 */
                this.name = raw.name;

                /**
                 * Комментарий
                 * @type {String}
                 */
                this.comment = raw.description;

                /**
                 * Доступность всех "целей/сегментов", указанных условии
                 * @type {Boolean}
                 */
                this.isAccessible = Boolean(raw.is_accessible);

                /**
                 * Используется ли условие в кампаниях
                 * Если данных нет, то по умолчанию считаем, что условие используется
                 * @type {boolean}
                 */
                this.isUsed = 'is_used' in raw ? Boolean(raw.is_used) : true;

                /**
                 * "Группы правил"
                 * @type {Array}
                 */
                this.groupRuleCollection = raw.conditions.map(groupRule);

                /**
                 * Условие имеет все groupRule только type=NONE
                 * Называем такое условие "негативное"
                 * Создаем на клиенте, на сервер не отправляем
                 * @type {boolean}
                 */
                this.isNegative = this.groupRuleCollection.every(negative);

                // проверка валидности данных
                isFinite(this.id) && this.id > 0 || error.items.push('id должен быть больше 0');

                if (error.items.length) {
                    error.message = 'Ошибка конвертации данных в объект dto';
                    error.sourceData = raw;
                    throw error;
                }

                /**
                 * Конвертирует "группу правил"
                 * @param {{ type: String, goals: Array }} raw
                 * @returns {Object}
                 */
                function groupRule(raw) {
                    return {
                        type: u['retargeting-condition-dto'].convGroupRuleTypeToApp(raw.type),
                        ruleCollection: raw.goals.map(rule)
                    };
                }

                /**
                 * Конвертирует "правило"
                 * @param {{ goal_type: String, goal_id: String, time: String, goal_domain: String, goal_name: String,
                 *         allow_to_use: Boolean, goal_subtype}} raw
                 * @returns {Object}
                 */
                function rule(raw) {
                    var rule = {
                        type:  u['retargeting-goal-dto'].convTypeToApp(raw.type),
                        goalId: raw.id,
                        day: raw.time || 0,
                        domain: raw.domain,
                        audienceType: u['retargeting-goal-dto'].convAudienceTypeToApp(raw.subtype)/* || 'UPLOADING_EMAIL'*/,
                        name: raw.name,
                        allowToUse: !!raw.allow_to_use,
                        counter: raw.counter_id
                    };
                    rule.audienceTypeName = u['retargeting-goal-dto']
                        .getAudienceTypeName(rule.audienceType, raw.uploading_source_id);
                    if (!rule.name && !rule.allowToUse) {
                        switch (rule.type) {
                        case 'METRIKA_GOAL':
                        case 'ECOMMERCE_GOAL':
                            rule.name = 'Цель не найдена';
                            break;
                        case 'METRIKA_SEGMENT':
                            rule.name = 'Сегмент не найден';
                            break;
                        case 'AUDIENCE_SEGMENT':
                            rule.name = 'Сегмент не найден';
                            break;
                        }
                    }

                    return rule;
                }

                /**
                 * Проверка "группы правил" - является ли она "негативной"
                 * @param {{ type: String }} groupRule
                 * @returns {boolean}
                 */
                function negative(groupRule) {
                    return groupRule.type === 'NONE';
                }
            },

            /**
             * Создает Data Transport Object "условия подбора аудиторий" для отправки на сервер
             * Конвертирует данных приложения в формат сервера
             * @param {RetargetingConditionDTO} dto
             * @constructor
             */
            RetargetingConditionDTOServer: function(dto) {
                if (!(this instanceof u['retargeting-condition-dto'].RetargetingConditionDTOServer)) {
                    return new u['retargeting-condition-dto'].RetargetingConditionDTOServer(dto);
                }

                this.id = String(dto.id || '');
                this.name = dto.name;
                this.description = dto.comment;
                this.conditions = dto.groupRuleCollection.map(groupRule);

                /**
                 * Конвертация "группы правил"
                 * @param {Object} dto
                 * @returns {Object}
                 */
                function groupRule(dto) {
                    return {
                        type: u['retargeting-condition-dto'].convGroupRuleTypeToServer(dto.type),
                        goals: dto.ruleCollection.map(rule)
                    };
                }

                /**
                 * Конвертация "правил"
                 * @param {Object} dto
                 * @returns {Object}
                 */
                function rule(dto) {
                    return {
                        type: u['retargeting-goal-dto'].convTypeToServer(dto.type),
                        id: dto.goalId,
                        time: u['b-retargeting-condition-edit-rule'].TYPE_HAS_DAY[dto.type] && dto.day
                    };
                }
            },

            /**
             * Создает Data Transport Object "ошибка валидации условия подбора аудиторий"
             * Конвертирует данные от сервера в формат приложения
             * @param { description: String, path: String } серверный объект ошибки
             * @constructor
             * @throw
             */
            RetargetingValidationError: function(raw) {
                if (!(this instanceof u['retargeting-condition-dto'].RetargetingValidationError)) {
                    return new u['retargeting-condition-dto'].RetargetingValidationError(raw);
                }

                this.description = retargetingConditionErrorMapper({}, raw); // объект для формирования текстов не нужен

                this.path = u['b-errors-presenter2'].formatPath(raw.path, {
                    conditions: 'groupRuleCollection',
                    goals: 'ruleCollection'
                });
            },

            /**
             * Возвращает пустой DTO
             * @param condition
             * @returns {RetargetingConditionDTO}
             * @constructor
             */
            RetargetingEmptyConditionDTO: function(condition) {
                return {
                    name: '',
                    comment: '',
                    isAccessible: true,
                    isNegative: false,
                    groupRuleCollection: [
                        {
                            type: 'OR',
                            ruleCollection: [
                                {
                                    type: 'METRIKA_GOAL',
                                    day: u['b-retargeting-condition-edit-rule'].MAX_DAYS,
                                    valid: false,
                                    goalId: 0,
                                    name: '',
                                    domain: '',
                                    allowToUse: true
                                }
                            ]
                        }
                    ]
                }
            }
        }
    });
}())
